/**
 *  '$RCSfile: EMLParserTest.java,v $'
 *  Copyright: 2000 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *
 *   '$Author: walbridge $'
 *     '$Date: 2008-11-05 23:00:46 $'
 * '$Revision: 1.6 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA    02111-1307    USA
 */

package org.ecoinformatics.emltest;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Vector;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.ecoinformatics.eml.EMLParser;

/**
 * A JUnit test for testing the EMLParser
 *
 * @author     Chad Berkley
 */
public class EMLParserTest extends TestCase
{
    private final static String TEST_DIR = "./src/test/resources";
    private final static String INVALID_DIR = TEST_DIR + "/invalidEML";
    private final static String ERROR1 = INVALID_DIR + "/eml-error1.xml";
    private final static String ERROR3 = INVALID_DIR + "/eml-error3.xml";
    private final static String ERROR4 = INVALID_DIR + "/eml-error4.xml";

    private EMLParser emlp;

    /**
     * Constructor to build the test
     *
     * @param name  the name of the test method
     */
    public EMLParserTest(String name)
    {
        super(name);
    }

    /** Establish a testing framework by initializing appropriate objects  */
    public void setUp()
    {

    }

    /** Release any objects after tests are complete  */
    public void tearDown()
    {
    }

    /**
     * Create a suite of tests to be run together
     *
     * @return   The test suite
     */
    public static Test suite()
    {
        TestSuite suite = new TestSuite();
        suite.addTest(new EMLParserTest("initialize"));
        suite.addTest(new EMLParserTest("testParse"));
        suite.addTest(new EMLParserTest("testParseString"));
        return suite;
    }

    /**
     * Check that the testing framework is functioning properly with a trivial
     * assertion.
     */
    public void initialize()
    {
        assertTrue(true);
    }

    public void testParse()
    {

        // all valid documents should validate
        File testDir = new File(TEST_DIR);
        Vector fileList = getXmlFiles(testDir);
        for (int i=0; i < fileList.size(); i++) {
            File testFile = (File)fileList.elementAt(i);
            try {
                System.err.println("Validating file: " + testFile.getName());
                emlp = new EMLParser(testFile);
            } catch (Exception e) {
                e.printStackTrace(System.err);
                fail("Document NOT valid!\n\n" + e.getClass().getName() +
                         "(" + e.getMessage() + ")" );
            }
        }

        // All of the invalid files should not validate
        // NOTE: EMLParser does not validate against the schema (see SAXParserTest)
        /* Comment out this sblock because the old EMLParser does not know how
         * to detect many validity isssues, and so thesse checkss fail for
         * known-valid files.  The new EMLValidator class handles these, so
         * don't worry about testing it for EMLParser.
        int failures = 0;
        File invalidDir = new File(INVALID_DIR);
        Vector invalidList = getXmlFiles(invalidDir);
        int invalidFileCount = invalidList.size();
        System.err.println("Checking invalid files: " + invalidFileCount);
        for (int i=0; i < invalidFileCount; i++) {
            File invalidFile = (File)invalidList.elementAt(i);
            System.err.println("Invalidating file: " + invalidFile.getName());
            try {
                emlp = new EMLParser(invalidFile);
                System.err.println("    Valid.");
            } catch (Exception e) {
                System.err.println("    Invalid.");
                failures++;
                assertTrue(e.getMessage() != null);
            }
        }
        if (failures != invalidFileCount) {
        	System.err.println(failures + "/" + invalidFileCount + " failures in directory.");
        	fail("Error: An EMLParserException should have been thrown for all invalid files.");
        }
        */

        try {
            File f = new File(ERROR1);
            System.err.println("Validating file: " + f.getName());
            emlp = new EMLParser(f);
            fail("Error 1. An EMLParserException should have been thrown.");
        } catch(Exception e) {
            assertTrue(e.getMessage().indexOf("valid") != -1);
        }

        try {
            File f = new File(ERROR3);
            System.err.println("Validating file: " + f.getName());
            emlp = new EMLParser(f);
            fail("Error 3. An EMLParserException should have been thrown.");
        } catch(Exception e) {
            assertTrue(e.getMessage().indexOf("valid") != -1);
        }

        try {
            File f = new File(ERROR4);
            System.err.println("Validating file: " + f.getName());
            emlp = new EMLParser(f);
            fail("Error 4. An EMLParserException should have been thrown.");
        } catch(Exception e) {
            assertTrue(e.getMessage().indexOf("valid") != -1);
        }
    }

    public void testParseString() {
        // document should validate when passed in as a String
        File testDir = new File(TEST_DIR);
        File testFile = new File(TEST_DIR, "eml-sample.xml");
        try {
            System.err.println("Validating file: " + testFile.getName());
            EMLParser emlp = new EMLParser(testFile);
        } catch (Exception e) {
            e.printStackTrace(System.err);
            fail("Document NOT valid!\n\n" + e.getClass().getName() +
                    "(" + e.getMessage() + ")" );
        }
    }

    /**
     * Get the list of files in a directory.
     *
     * @param directory the directory to list
     * @return a vector of File objects in the directory
     */
    private Vector getXmlFiles(File directory)
    {
        String[] files = directory.list();
        Vector fileList = new Vector();

        for (int i=0; i < files.length; i++) {
            String filename = files[i];
            File currentFile = new File(directory, filename);
            if (currentFile.isFile() && filename.endsWith(".xml")) {
                fileList.addElement(currentFile);
            }
        }
        return fileList;
    }

}
