package edu.ucsb.nceas.metacat;

import java.net.ConnectException;
import java.util.*;

/** <p>Authenticate based on a GSSContext; can pass through to a
 *  username/password-based authentication mechanism if the GSSContext
 *  is not provided.   Expects a backing AuthInterface to provide Group info,
 *  etc, based on {@link GsiToUsernameMap}.</p>
 *
 *  <p>All methods except for authentication itself are automatically passed
 *  through to a delegated {@link AuthInterface} -- group memberships, user
 *  listings, etc.</p>
 *
 *  <p>Configuration options:</p>
 *
 *  <ul>
 *     <li>If no GSSContext is present, authentication info on to delegated
 *     authentication mechanism?  See {@link #setAuthnDelegationAllowed}.</li>
 *     <li>If both a GSSContext and username+password are present, which takes
 *     precedence?  See {@link #setPrecedence}. </li>
 *  </ul> */
public class AuthGsi implements AuthInterface {
	private AuthInterface delegate;
	private GsiToUsernameMap dnMap;
	private boolean authnDelegationAllowed = false;
	private String precedence = PRECEDENCE_GSS;

	/** Parameter for {@link #setPrecedence}.  Signifies that if both a
	 *  {@link org.ietf.jgss.GSSContext} and username+password are present,
	 *  the GSSContext takes precedence for authentication. */
	public static final String PRECEDENCE_GSS = "gss";

	/** Parameter for {@link #setPrecedence}.  Signifies that if both a
	 *  {@link org.ietf.jgss.GSSContext} and username+password are present,
	 *  the username and password take precedence for authentication. */
	public static final String PRECEDENCE_UN_PW = "username+password";

	/** Used internally for validation. */
	private static final Set PRECEDENCE_LEGAL_VALUES
		= Collections.unmodifiableSet(new HashSet(Arrays.asList
			(new String[] { PRECEDENCE_GSS, PRECEDENCE_UN_PW })));

	/** The config key to tell whether or not to trust localhost connections. */
	private static final String CONFIG_TRUST_LOCAL = "trustLocalHost";

	/** If not enough information is provided to authenticate a user, should we
	 *  delegate identification to the contained {@link AuthInterface}?
	 *  In practice, that will usually mean falling back to username+password
	 *  if a GSSContext is not available.  Default false. */
	public boolean isAuthnDelegationAllowed() { return authnDelegationAllowed; }

	/** If not enough information is provided to authenticate a user, should we
	 *  delegate identification to the contained {@link AuthInterface}?
	 *  In practice, that will usually mean falling back to username+password
	 *  if a GSSContext is not available.  Default false. */
	public void setAuthnDelegationAllowed(boolean authnDelegationAllowed) {
		this.authnDelegationAllowed = authnDelegationAllowed;
	}

	/** If more authentication info is present than is necessary to identify a
	 *  user, which of it takes precedence?  For example, if both a GSSContext
	 *  and a username+password are present, which should we use to identify the
	 *  user?  Default is {@link #PRECEDENCE_GSS}.
	 *
	 *  @return either {@link #PRECEDENCE_GSS} or {@link #PRECEDENCE_UN_PW} */
	public String getPrecedence() { return precedence; }

	/** The authentication mechanism that we delegate to for group info, etc,
	 *  and sometimes authentication, if allowed.  Null if none. */
	public AuthInterface getDelegate() { return delegate; }

	/** The authentication mechanism that we delegate to for group info, etc,
	 *  and sometimes authentication, if allowed.  Null if none. */
	public void setDelegate(AuthInterface delegate) { this.delegate = delegate; }

	/** If more authentication info is present than is necessary to identify a
	 *  user, which of it takes precedence?  For example, if both a GSSContext
	 *  and a username+password are present, which should we use to identify the
	 *  user?  Default is {@link #PRECEDENCE_GSS}.
	 *
	 *  @param precedence can take values {@link #PRECEDENCE_GSS} or
	 *  {@link #PRECEDENCE_UN_PW} */
	public void setPrecedence(String precedence) {
		if (!PRECEDENCE_LEGAL_VALUES.contains(precedence))
			throw new IllegalArgumentException
				("Unknown precedence value: \"" + precedence + "\".");
		this.precedence = precedence;
	}

	/** Should we trust logins without a password from localhost? Necessary
	 *  for {@link edu.ucsb.nceas.metacat.client.Metacat#login(String)}.
	 *  Configured via "trustLocalHost" in metacat config file. */
	public static boolean isLocalHostTrusted() {
		String result = MetaCatUtil.getOption(CONFIG_TRUST_LOCAL);
		return "yes".equalsIgnoreCase(result) || "true".equalsIgnoreCase(result);
	}

	/** Just check GSS authentication. */
	private boolean gssAuthn(AuthInfo info) {
		if (isLocalHostTrusted() && info.isLocal()
			&& info.getGssContext() == null && info.isAuthenticated())
		{
			MetaCatUtil.debugMessage
				("Authenticating via localhost trust: passed", 20);
			return true;
		}
		else {
			boolean result = info.getGssContext() != null && info.getUserDN() != null;
			MetaCatUtil.debugMessage
				("Authenticating with GSI: " + (result ? "passed" : "failed"), 20);
			MetaCatUtil.debugMessage("User: " + info, 20);
			return result;
		}
	}

	/** The mapping from GSS Distinguished Name (DN) to a username that will be
	 *  understood by the delegated authentication mechanism (such as LDAP). */
	public GsiToUsernameMap getDnMap() { return dnMap; }

	/** The mapping from GSS Distinguished Name (DN) to a username that will be
	 *  understood by the delegated authentication mechanism (such as LDAP). */
	public void setDnMap(GsiToUsernameMap dnMap) { this.dnMap = dnMap; }

	/** Map GSS DN to delegate auth mechanism's username, if it hasn't already
	 *  been done.
	 *  <p>Note: if the password is blank, and the login request was local, and
	 *  we are configured to trust local host, then we treat the username
	 *  as a GSI DN, and map it to a local username using our map file. */
	private void doMap(AuthInfo info) {
//		new Exception("=-= Mapping: before = " + info.toString(true)).printStackTrace();
		debug("=-= Mapping; before: " + info.toString(true));
		String userDN = info.getUserDN();
		// if local, the username may actually be set to be the user DN
		// (but only if the password is null)
		boolean tryingUsernameAsDN = false;
		debug("    = userDN = " + userDN);
		debug("    = local = " + info.isLocal());
		debug("    = pw blank = " + info.isPasswordBlank());
		debug("    = local trusted = " + isLocalHostTrusted());
		if (userDN == null && info.isLocal()
			&& info.isPasswordBlank() && isLocalHostTrusted())
		{
			userDN = info.getUsername();
			tryingUsernameAsDN = true;
		}
		if (dnMap != null && userDN != null) {
			String username = dnMap.getUsername(userDN);
			if (username != null) {
				// preserve the DN, if we've recognized the username as a DN
				if (tryingUsernameAsDN) {
					info.setUserDN(userDN);
					info.setAuthenticated(true);
				}
				info.setUsername(username);
			}
		}
		debug("=-= Mapping; after: " + info.toString(true));
//		new Exception("=-= Mapping: after = " + info.toString(true)).printStackTrace();
	}

	/** Is <tt>info</tt> already marked as authenticated?  If not,
	 *  try to authenticate it and mark whether it passes.
	 *
	 *  @see AuthInfo#isAuthenticated() */
	private boolean checkAuth(AuthInfo info) throws ConnectException {
		if (info.getAuthenticated().equals(AuthInfo.AUTHENTICATED_UNKNOWN))
			// note side effect: doMap() is called also
			info.setAuthenticated(authenticate(info));
		return info.isAuthenticated();
	}

	// -----------------------------
	// AuthInterface Implementation:
	// -----------------------------

	private void debug(String s) {
//		MetaCatUtil.debugMessage(s, 35);
	}

	public boolean authenticate(AuthInfo info) throws ConnectException {
		debug("-=- Authenticating GSI");
		debug("    - delegation allowed? " + isAuthnDelegationAllowed());
		doMap(info);
		if (!isAuthnDelegationAllowed()) return gssAuthn(info);
		else {
			boolean enoughInfoUnPw
				= (info.getUsername() != null && info.getPassword() != null);
			debug("    - un=" + info.getUsername() + "; pw=" + info.getPassword());
			debug("    - enough info to delegate? " + enoughInfoUnPw);
			debug("    - precedence=" + getPrecedence() + " (" + PRECEDENCE_UN_PW.equals(getPrecedence()) + ")");
			debug("    - delegate = " + delegate);
			if (enoughInfoUnPw
					&& PRECEDENCE_UN_PW.equals(getPrecedence())
					&& delegate != null) {
				debug("    - delegating (" + info.toString(true) + ")");
				return getDelegate().authenticate(info);
			}
			else {
				debug("    - NOT delegating");
				return gssAuthn(info);
			}
		}
	}

	public String[][] getUsers(AuthInfo info) throws ConnectException {
		checkAuth(info);
		return delegate.getUsers(info);
	}

	public String[] getUsers(AuthInfo info, String group) throws ConnectException {
		checkAuth(info);
		return delegate.getUsers(info, group);
	}

	public String[][] getGroups(AuthInfo info) throws ConnectException {
		checkAuth(info);
		return delegate.getGroups(info);
	}

	public String[][] getGroups(AuthInfo info, String foruser) throws ConnectException {
		checkAuth(info);
		return delegate.getGroups(info, foruser);
	}

	public HashMap getAttributes(String foruser) throws ConnectException {
		return delegate.getAttributes(foruser);
	}

	public HashMap getAttributes(AuthInfo info, String foruser) throws ConnectException {
		checkAuth(info);
		return delegate.getAttributes(info, foruser);
	}

	public String getPrincipals(AuthInfo info) throws ConnectException {
		checkAuth(info);
		return delegate.getPrincipals(info);
	}
}
