/**
 *  '$RCSfile$'
 *  Copyright: 2018 Regents of the University of California and the
 *             National Center for Ecological Analysis and Synthesis
 *
 *   '$Author$'
 *     '$Date$'
 * '$Revision$'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package edu.ucsb.nceas.metacat;

import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import edu.ucsb.nceas.metacat.properties.PropertyService;
import edu.ucsb.nceas.metacat.util.AuthUtil;
import edu.ucsb.nceas.utilities.PropertyNotFoundException;

/**
 * This class represents a filter to prevent the events generated by some ip addresses and subjects
 * to be logged into db
 * @author tao
 *
 */
public class EventLogFilter {
    private Vector<String> blackIPList = null;
    private Vector<String> blackSubjectList = null;
    private Log logMetacat = LogFactory.getLog(EventLogFilter.class);
    
    /**
     * Default Construct
     */
    public EventLogFilter() {
        String blackIPString = null;
        String blackSubjectString = null;
        try {
            blackIPString = PropertyService.getProperty("event.log.blacklist.ipaddress");
        } catch (PropertyNotFoundException pnfe) {
           logMetacat.warn("EventLogFilter.EventLogFilter - Could not get metacat property: event.log.blacklist.ipaddress. "
                    + "Any ip address will be allowed to log: "
                    + pnfe.getMessage());
        }try {
            blackSubjectString = PropertyService.getProperty("event.log.blacklist.subject");
        } catch (PropertyNotFoundException pnfe) {
           logMetacat.warn("EventLogFilter.EventLogFilter - Could not get metacat property: event.log.blacklist.subject. "
                    + "Any subject will be allowed to log: "
                    + pnfe.getMessage());
        }        
        blackIPList = AuthUtil.split(blackIPString, AuthUtil.DELIMITER, AuthUtil.ESCAPECHAR);
        logMetacat.debug("EventLogFilter.EventLogFilter - the black ip addresses are: "+blackIPList);
        blackSubjectList = AuthUtil.split(blackSubjectString, AuthUtil.DELIMITER, AuthUtil.ESCAPECHAR);
        logMetacat.debug("EventLogFilter.EventLogFilter - the black subjects are: "+blackSubjectList);
    }
    
    /**
     * Determine if the given EventLogData object will be filtered out (not logging) or not.
     * @param logData  the EventLogData will be evaluated
     * @return true if the event log data should be filtered out(not logging); otherwise false.
     */
    public boolean filter(EventLogData logData) {
        boolean filteredOut = false;//default value is to log
        if(logData != null) {
            String ipAddress = logData.getIpAddress();
            if(ipAddress != null && !ipAddress.trim().equals("") && blackIPList != null && !blackIPList.isEmpty() && blackIPList.contains(ipAddress)) {
                filteredOut = true;
            }
            if(!filteredOut) {
                //If the ip address is the black list, we will ignore the subject comparison; otherwise we will try to compare the subject
                String subject = logData.getPrincipal();
                if(subject != null && !subject.trim().equals("") && blackSubjectList != null && !blackSubjectList.isEmpty() && blackSubjectList.contains(subject)) {
                    filteredOut = true;
                }
            }
        } else {
            logMetacat.warn("EventLogFilter.filter - the logData object is null. So we filter it out");
            filteredOut=true;
        }
        return filteredOut;
    }

}
