package edu.ucsb.nceas.metacat.client.gsi;

import edu.ucsb.nceas.metacat.client.MetacatAuthException;
import edu.ucsb.nceas.metacat.client.MetacatClient;
import edu.ucsb.nceas.metacat.client.MetacatInaccessibleException;
import edu.ucsb.nceas.utilities.HttpMessage;
import edu.ucsb.nceas.utilities.Options;
import org.ietf.jgss.GSSCredential;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Properties;

/** An extension of the Metacat client that uses Grid Security Infrastructure
 *  (GSI) enabled HTTPS instead of HTTP to communicate.
 *
 *  <p>Note that not all client deployments will include the JARs necessary to
 *  run this version of the Metacat client; therefore, we should make sure that
 *  the superclass (MetacatClient) can run even if this class can't be loaded.
 *  That is, catch (and log) NoClassDefFoundError, etc. */
public class MetacatGsiClient extends MetacatClient {

	/** The current user's GSS credential, as an alternative to
	 *  username/password. Needed for every connection.
	 *  Set via {@link #login(GSSCredential)}. */
	private GSSCredential credential;

	private static final boolean[] registered = { false };

	private static final String HANDLER_PACKAGE_A = "edu.ucsb.nceas.protocols";
	private static final String HANDLER_PACKAGE_B = "org.globus.net.protocol";
//	public static final String GSI_HTTPS_PROTOCOL = "https";
	public static final String GSI_HTTPS_PROTOCOL = "httpg";

	private static final String HANDLER_PROPERTY = "java.protocol.handler.pkgs";

	/** Register the "httpg" protocol, which this client is designed to
	 *  handle. */
	public static void register() {
		// do we need to initialize Options?
		if (Options.getInstance() == null) {
//			Properties sysProps = System.getProperties();
//			Enumeration i = sysProps.propertyNames();
//			while (i.hasMoreElements()) {
//				String k = (String) i.nextElement();
//				System.err.println("    - " + k + ": " + System.getProperty(k));
//			}
		}

//		System.err.println("--- Before: handler packages registered: " + System.getProperty(HANDLER_PROPERTY));

		if (registered[0]) return;
		synchronized(registered) {
			if (registered[0]) return; // avoid race condition

			// set up our handler for HTTPG via a system property
			String handlerPackages = System.getProperty(HANDLER_PROPERTY);
			if (handlerPackages == null) handlerPackages = "";
			// list A first
			if (handlerPackages.indexOf(HANDLER_PACKAGE_A) == -1) {
				handlerPackages += (handlerPackages.length() == 0 ? "" : "|")
					+ HANDLER_PACKAGE_A;
				System.setProperty(HANDLER_PROPERTY, handlerPackages);
			}
			// list B after A -- its HTTPG handler doesn't work for us
			if (handlerPackages.indexOf(HANDLER_PACKAGE_B) == -1) {
				handlerPackages += (handlerPackages.length() == 0 ? "" : "|")
					+ HANDLER_PACKAGE_B;
				System.setProperty(HANDLER_PROPERTY, handlerPackages);
			}
			registered[0] = true;
//			String handlerClassName
//				= HANDLER_PACKAGE + "." + GSI_HTTPS_PROTOCOL + ".Handler";
//			try {
//				Class handlerClass = Class.forName(handlerClassName);
//				Object handler = handlerClass.newInstance();
//				MetaCatUtil.debugMessage
//					(GSI_HTTPS_PROTOCOL + " handler = " + handler, 40);
//			} catch (Throwable e) {
//				String msg = "Warning: unable to instantiate HTTPG handler ("
//					+ handlerClassName + ").";
//				if (MetaCatUtil.debugMessage(msg, 5))
//					e.printStackTrace();
//			}

//			System.err.println("--- After: handler packages registered: " + System.getProperty(HANDLER_PROPERTY));
			// alternate method: set up a stream handler factory
//			URL.setURLStreamHandlerFactory(new MetacatURLStreamHandlerFactory());
//			registered[0] = true;
		}
	}
	/** Be sure to put this *after* the definition of {@link #registered} to
	 *  avoid a null pointer exception. */
	static {
		register();
	}

	private void initCredential(GSSCredential credential)
		throws MetacatAuthException
	{
		if (credential == null)
			throw new NullPointerException("Credential is null.");
		if (this.credential != null)
			throw new MetacatAuthException
				("Credential already initialized; please create a new "
					+ getClass().getName() + " to start a new session.");
		this.credential = credential;
	}

	public String login(GSSCredential credential)
			throws MetacatAuthException, MetacatInaccessibleException
	{
		initCredential(credential);

		// code below mostly copied from super.login(username, password)
		Properties prop = new Properties();
		prop.put("action", "login");
		prop.put("qformat", "xml");

		String response;
		try {
			response = sendDataForString(prop, null, null, 0);
		} catch (Exception e) {
			throw new MetacatInaccessibleException(e);
		}

		if (response.indexOf("<login>") == -1) {
			setSessionId("");
			throw new MetacatAuthException(response);
		} else {
			int start = response.indexOf("<sessionId>") + 11;
			int end = response.indexOf("</sessionId>");
			if ((start != -1) && (end != -1)) {
				setSessionId(response.substring(start,end));
			}
		}
		return response;
	}

	/** Parse the Metacat URL and, if we are using a GSI credential,
	 *  ensure that the protocol is an SSL-based one (HTTPS or HTTPG). */
	private URL parseAndCheckURL()
		throws MetacatAuthException, MetacatInaccessibleException
	{
		URL url;

		try {
			url = new URL(getMetacatUrl().trim());
		} catch (MalformedURLException e) {
			throw new MetacatInaccessibleException
				("Unable to parse URL to contact Metacat server: \""
					+ getMetacatUrl() + "\".", e);
		}

		if (credential != null) {
			String protocol = url.getProtocol();
			if (!protocol.toLowerCase().equals(GSI_HTTPS_PROTOCOL))
				throw new MetacatAuthException
					("SSL + GSI connection (" + GSI_HTTPS_PROTOCOL
						+ ") is required to pass GSI credentials; "
						+ "cannot send via protocol \"" + protocol + "\".");
		}

		return url;
	}

	/** Create an HttpMessage that can send messages to the server.
	 *  If using a GSI credential, use the credential to set up an SSL
	 *  connection (HTTPS / HTTPG).  If using HTTP and username/password,
	 *  just use a regular HTTP conenction. */
	protected HttpMessage createHttpMessage()
		throws MetacatInaccessibleException, MetacatAuthException, IOException
	{
		if (credential != null)
			return new HttpGsiMessage(credential, parseAndCheckURL());
		else
			return super.createHttpMessage();
	}
}
