/**
 *    '$RCSfile: RExpression.java,v $'
 *
 *     '$Author: higgins $'
 *       '$Date: 2006/03/15 22:16:08 $'
 *   '$Revision: 1.30 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.R;

import java.io.BufferedWriter;
import java.io.File; 
import java.io.FileWriter;
import java.io.StringReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.text.NumberFormat;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.ArrayList;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.actor.gui.style.TextStyle;
import ptolemy.data.ArrayToken;
import ptolemy.data.BooleanToken;
import ptolemy.data.DoubleToken;
import ptolemy.data.RecordToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Nameable;
import ptolemy.kernel.util.StringAttribute;
import ptolemy.kernel.util.Workspace;

//////////////////////////////////////////////////////////////////////////
//// RExpression
/**
  * The RExpression actor is an actor designed to run an R script or function with inputs
  * and outputs determined by the ports created by the user. Port names will correspond to
  * R object names. The RExpression actor is modeled after the Ptolemy expression actor,
  * except that that instead of using a single mathematical expression in Ptolemy's
  * expression language, it uses a set of the more powerful, higher order expressions 
  * available under R. Both input and output port will usually be added to the actor;
  * The names of these ports correspond to R variables used in the R script.
  *
  * @author Dan Higgins, NCEAS, UC Santa Barbara
  * @version 3/3/2006
  * @UserLevelDocumentation This actor let the user insert R scripts in a Kepler workflow.
  * It requires the R system to be installed on the computer executing the workflow
*/

public class RExpression extends TypedAtomicActor {


      ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** 
     * The output port. 
    */
    public TypedIOPort output;

    /**
     * The expression that is evaluated to produce the output.
    */
    public StringAttribute expression;

    /** 
     * This setting determines whether or not to save the R
     * workspace when R is closed; set to '--save' if you
     * need to retreive the workspace later in a workflow in
     * another RExpression actor.
    */
    public StringParameter save_nosave;

    /** 
     * The 'R' working directory (home dir by default)
    */
    public StringParameter Rcwd;

    private static String NO_SAVE = "--no-save";
    private static String SAVE = "--save";
    private static String NO_RESTORE = "--no-restore";
    private static String RESTORE = "--restore";  
    
    private static String RdotExe = "R";

    private int cntr = 0;

   /** 
    * If <i>true</i>, then create a graphics output port. (the default);
    * If <i>false</i>, then don't.
    */
    public Parameter graphicsOutput;

   /** 
    *The width of the output graphics bitmap in pixels 
    */
    public StringParameter numXPixels;
    
   /** 
    * The height of the output graphics bitmap in pixels 
    */
    public StringParameter numYPixels;

    private String saveString;
    private String restoreString = NO_RESTORE;
    
    // if arrays sent to R are longer than this value, then use a file
    // rather than a string on the command line to pass the data
    // This is necessary because apparently R has a fixed buffer
    // for passing long commands
    private int maxCommandLineLength = 50000;

    /** The name of the default graphics output file created
     *  by the actor
     */
    public TypedIOPort graphicsFileName;

    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public RExpression(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);

        expression = new StringAttribute(this, "expression");
        expression.setDisplayName("R function or script");
        TextStyle expressionTS = new TextStyle(expression, "R Expression");
        expression.setExpression("2 + 2");

        Rcwd = new StringParameter(this, "Rcwd");
        Rcwd.setDisplayName("R working directory");
        Rcwd.setExpression(System.getProperty("user.home"));

        save_nosave = new StringParameter(this, "save_nosave");
        save_nosave.setDisplayName("Save or not");
        save_nosave.setExpression(NO_SAVE);
        save_nosave.addChoice(NO_SAVE);
        save_nosave.addChoice(SAVE);

//  restore parameter is removed for now because it doesn't work
//  .RData is saved in the working directory by 'save' but R doesn't look there
//  To restore a saved workspace, add the command 'load(".RData') to the script
//        restore_norestore = new StringParameter(this, "restore or not");
//        restore_norestore.setExpression(NO_RESTORE);
//        restore_norestore.addChoice(NO_RESTORE);
//        restore_norestore.addChoice(RESTORE);


        graphicsOutput = new Parameter(this, "graphicsOutput");
        graphicsOutput.setDisplayName("Graphics Output");
        graphicsOutput.setTypeEquals(BaseType.BOOLEAN);
        graphicsOutput.setToken(BooleanToken.TRUE);

        numXPixels = new StringParameter(this, "numXPixels");
        numXPixels.setDisplayName("Number of X pixels in image");
        numXPixels.setExpression("480");
        numYPixels = new StringParameter(this, "numYPixels");
        numYPixels.setDisplayName("Number of Y pixels in image");
        numYPixels.setExpression("480");


        graphicsFileName = new TypedIOPort(this, "graphicsFileName", false, true);
        graphicsFileName.setTypeEquals(BaseType.STRING);

        output = new TypedIOPort(this, "output", false, true);
        output.setTypeEquals(BaseType.STRING);

            _attachText("_iconDescription",
                "<svg>\n"
                + "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"30\" "
                + "style=\"fill:white\"/>\n"
                + "<text x=\"20\" y=\"25\" "
                + "style=\"font-size:30; fill:blue; font-family:SansSerif\">"
                + "R</text>\n"
                + "</svg>\n");

    }

        /** Override the base class to set type constraints.
     *  @param workspace The workspace for the new object.
     *  @return A new instance of RExpression.
     *  @exception CloneNotSupportedException If a derived class contains
     *   an attribute that cannot be cloned.
     */
    public Object clone(Workspace workspace) throws CloneNotSupportedException {
        RExpression newObject = (RExpression) super.clone(workspace);
        newObject.output.setTypeEquals(BaseType.STRING);
        newObject.graphicsFileName.setTypeEquals(BaseType.STRING);
        return newObject;
    }

    /*
     * The fire method should first call the superclass.
     * Then all the input ports should be scanned to see which ones have tokens.
     * The names of those ports should be used to create a named R object (array?).
     * R script for creating objects corresponding to these ports should be inserted before the
     * script in the expressions parameter.
     * Then the R engine should be started and run, with the output sent to the output port.
    */
    public synchronized void fire() throws IllegalActionException {
        super.fire();

          boolean graphicsOutputValue = ((BooleanToken)graphicsOutput.getToken()).booleanValue();

        saveString = save_nosave.stringValue();
//        restoreString = restore_norestore.stringValue();

        String nxs = numXPixels.stringValue();
        try {
          int nxp = (new Integer(nxs)).intValue();
        } catch (Exception w) {
          nxs = "480";
        }

        String nys = numYPixels.stringValue();
        try {
          int nyp = (new Integer(nys)).intValue();
        } catch (Exception w1) {
          nys = "480";
        }

        String home = Rcwd.stringValue();
        File homeFile = new File(home);
        // if not a directory, use 'home'
        if (!homeFile.isDirectory()) home = System.getProperty("user.home");

        home = home.replace('\\','/');
        String setCWD = "setwd('"+home+"')\n";
        graphicsOutputFile = getUniqueFileName("jpg");
        String graphicsDevice = "";
          // Various image parameters (e.g. 'width' and 'height' should be made into actor params
        if (graphicsOutputValue) {
          graphicsDevice = "jpeg(filename = '"+ graphicsOutputFile + "'"+",width = "+nxs+", height = "+nys+", pointsize = 12,quality = 75, bg = 'white')";
        }
        List ipList = inputPortList();
        Iterator iter_i = ipList.iterator();
        opList = outputPortList();
        iter_o = opList.iterator();
        String RPortInfo = "";
        RPortInfo = setCWD + graphicsDevice + "\n";
        Token at;
        String temp1; 
        cntr=0;
        while (iter_i.hasNext()) {
          TypedIOPort tiop = (TypedIOPort)iter_i.next();
          try{
            if (tiop.hasToken(0)) {
              String temp = tiop.getName();
              Token token = tiop.get(0);
              String token_type_string = token.getType().toString();
              String token_class_name = token.getType().getTokenClass().getName();
              // check token type and convert to R appropriately
              if (token_class_name.equals("ptolemy.data.RecordToken")) {
                String Rcommands = recordToDataFrame((RecordToken)token, temp);
                Rcommands = breakIntoLines(Rcommands);
                RPortInfo = RPortInfo + Rcommands + "\n";              }
              else if ((token_type_string.equals("double"))||
                  (token_type_string.equals("int"))||
                  (token_type_string.equals("string"))
                 )
              {
                at = (Token)token;
                temp1 = at.toString();
                temp1 = temp1.replace('\\','/');
                // assume that the token's string value might be 'nil' for a missing value
                temp1 = temp1.replaceAll("nil", "NA");
                temp = temp + " <- " + temp1;
                RPortInfo = RPortInfo + temp + "\n";
              }
              else if ((token_type_string.equals("{double}"))||
                  (token_type_string.equals("{int}"))||
                  (token_type_string.equals("{string}"))
                 )
             {
              // token is an arrayToken !!!
                at = (Token)token;
                temp1 = at.toString();
                temp1 = temp1.replace('{' , '(');
                temp1 = temp1.replace('}' , ')');
                temp1 = temp1.replace('"', '\'');
                 // assume that the token's string value might be 'nil' for a missing value
                temp1 = temp1.replaceAll("nil", "NA");
               // if string is long, create a temp file for passing array data 
                if (temp1.length()>maxCommandLineLength) {
                  temp1 = temp1.replace('(',' '); 
                  temp1 = temp1.replace(')',' '); 
                  String filename = writeDataFile(temp1, cntr);
                  cntr++; 
                  temp = temp + " <- scan('"+filename+"', sep=',')";
                  temp = temp + "\n" + "file.remove('"+filename+"')";
                  RPortInfo = RPortInfo + temp + "\n";  
                } else {   // otherwise use the modified string
                   temp = temp + " <- c" + temp1;
                   temp = breakIntoLines(temp);
                   RPortInfo = RPortInfo + temp + "\n"; 
                }
             }
           }
          } catch (IllegalActionException iae) {
              // just continue (port is probably not connected)
          }
        }
//System.out.println("RPortInfo: "+RPortInfo);
        // The following command casues R to output a series of 4 dashes which are used as a marker
        // Any R output after this marker is used to construct information for the actor output
        // ports. This information is removed from the R output text displayed to the user.
        String r_out = "cat('----')\n";
        while (iter_o.hasNext()) {
          TypedIOPort tiop_o = (TypedIOPort)iter_o.next();
          String temp_o = tiop_o.getName();
          // now need to create an R script that returns info about an R object with the
          // port name for use in creating Kepler output object
          if ((!temp_o.equals("output")) && (!temp_o.equals("graphicsFileName")))
          {
             r_out = r_out + "if (exists('"+temp_o+"', mode='numeric')) dput("+temp_o+")\n";
             r_out = r_out + "if (exists('"+temp_o+"', mode='character')) dput("+temp_o+")\n";
             r_out = r_out + "if (exists('"+temp_o+"', mode='logical')) dput("+temp_o+")\n";
             // the 'logical mode will handle objects that are 'TRUE', 'FALSE', or 'NA'
           }
        }

        String script = expression.getExpression();
        script = RPortInfo + script + "\n"+ r_out + "\nquit()\n";
        try{
          _exec();
        } catch (Exception w) {
          System.out.println("Error in _exec()");
        }

        String outputString = "";
        String errorString = "";
        try {
          _inputBufferedWriter.write(script);
          _inputBufferedWriter.flush();
          _inputBufferedWriter.close();
        } catch (IOException ex) {
           throw new IllegalActionException(this, ex, "Problem writing input '");
        }
        try{
          int result = _process.waitFor();
          System.out.println("Process complete: "+ result);
        } catch (Exception www) {
          System.out.println("Exception waiting for _process to end!");
        }

      while (outputString.equals("")) {
        try{
          Thread.currentThread().sleep(100);
        } catch (Exception e) {
          System.out.println("Error in TestApp while sleeping!");
        }
           outputString = _outputGobbler.getAndReset();
           errorString = _errorGobbler.getAndReset();
           int loc = outputString.indexOf("cat('----')");
           int loc1 = outputString.lastIndexOf("----");
  //System.out.println("output: "+ outputString);
          String outputStringDisp = outputString;
           if ((loc>-1)&&(loc1>-1)) {
             outputStringDisp = outputString.substring(0,loc);
             String rem = outputString.substring(loc1+5,outputString.length());
             getOutput(rem);
           }
           output.send(0, new StringToken(outputStringDisp + "\n" + errorString));
           if (!graphicsDevice.equals("")) {
             graphicsFileName.send(0, new StringToken(home+"/"+graphicsOutputFile));
           }
      }
    }


    public boolean postfire() throws IllegalActionException {
        _errorGobbler.quit();
        _outputGobbler.quit();

        return super.postfire();
    }

    ///////////////////////////////////////////////////////////////////
    ////                     private methods                       ////

    private void getOutput(String str) {
      int pos = -1;
      int temp = -1;
      temp = str.indexOf("if (exists('");
      while (temp > -1) {
        temp = temp + 12;
        pos = str.indexOf("'",temp);
        // name is now between temp and pos
        String portName = str.substring(temp,pos);
        String temp2 = "dput(" + portName + ")";
        pos = str.indexOf(temp2, pos) + temp2.length();
        temp = str.indexOf(">", pos);
        if (temp<0) temp = str.length()-1;
        String vectorVal = str.substring(pos, temp);
        vectorVal = vectorVal.trim();
//System.out.println("portName: "+portName+ " value: " +vectorVal);
        if (vectorVal.length()>0) {
          setOutputToken(portName, vectorVal);
        }

        temp = str.indexOf("if (exists('", temp);
      }
    }

    private void setOutputToken(String portName, String tokenValue) {
      opList = outputPortList();
      iter_o = opList.iterator();
      while (iter_o.hasNext()) {
        TypedIOPort tiop_o = (TypedIOPort)iter_o.next();
        String temp_o = tiop_o.getName();
        if ((!temp_o.equals("output")) && (!temp_o.equals("graphicsFileName"))) {
          if (temp_o.equals(portName)) {
            try{
              if (tokenValue.equals("TRUE")) {
                BooleanToken btt = BooleanToken.getInstance(true);
                tiop_o.setTypeEquals(BaseType.BOOLEAN);
                tiop_o.send(0,btt);
              }
              if (tokenValue.equals("FALSE")) {
                BooleanToken btf = BooleanToken.getInstance(false);
                tiop_o.setTypeEquals(BaseType.BOOLEAN);
                tiop_o.send(0,btf);
              }
              if (tokenValue.equals("NA")) {
                StringToken st = new StringToken("nil");
                tiop_o.setTypeEquals(BaseType.STRING);
                tiop_o.send(0, st);
                // this solution just sends a string token with value 'nil'
                // in R, 'NA' is considered a boolean state
                // i.e. 3 state logic, so this isn't really correct
                // but nil support for Ptolemy BooleanTokens not yet available
              }
              
              if (tokenValue.startsWith("\"")) {  // these are strings
                // now remove the start and end quotes
                tokenValue = tokenValue.substring(1,tokenValue.length()-1);
                StringToken st = new StringToken(tokenValue);
                tiop_o.setTypeEquals(BaseType.STRING);
                tiop_o.send(0, st);
              }
              NumberFormat nf = NumberFormat.getInstance();
              try{
                Number num = nf.parse(tokenValue);
                DoubleToken dt = new DoubleToken(tokenValue);
                tiop_o.setTypeEquals(BaseType.DOUBLE);
                tiop_o.send(0, dt);
              } catch (Exception eee) {
                // just continue if not a number
              }

              if (tokenValue.startsWith("c(")) {  // handles R vectors
                String temp = "{" + tokenValue.substring(2, tokenValue.length());
                temp = temp.replace(')', '}');
                 // convert NA values to 'nil'
                temp = temp.replaceAll("NA", "nil");
                ArrayToken at = new ArrayToken(temp);
                tiop_o.send(0, at);
              }
            } catch (Exception w) {
              System.out.println("Problem setting output port!");
            }
            return;
          }
        }
      }
    }

    // given a recordToken and a portName, create the R script to make a dataframe with the
    // portName as its R name. Should check that all the items in the record are the same length
    private String recordToDataFrame(RecordToken recordToken, String portName) {
      String ret = "";
      String temp = "";
      String tempA = "";
      String labellist = "";
      cntr = 0;
      int arrayLength = -1;
      Set labels = recordToken.labelSet();
      Iterator iter_l = labels.iterator();
      while (iter_l.hasNext()) {
        String label = (String)iter_l.next();
//  System.out.println("Label: "+label);
        Token labelvaltoken = (recordToken).get(label);
//  System.out.println("Token: "+labelvaltoken.getType().toString());
        String token_type_string = labelvaltoken.getType().toString();
        if ((token_type_string.equals("{double}"))||
            (token_type_string.equals("{int}"))||
            (token_type_string.equals("{string}"))
            )
          {
          // for now, assume that token is an arrayToken !!!
          // other token types are just ignored
          labellist = labellist + label + ",";
          if (arrayLength==-1) {
            arrayLength = ((ArrayToken)labelvaltoken).length();
          } else {
            int a_len = ((ArrayToken)labelvaltoken).length();
            if (a_len!=arrayLength) {
              System.out.println("record elements are not all the same length!");
              return "";
            }
          }
          temp = labelvaltoken.toString();
          temp = temp.replace('{' , '(');
          temp = temp.replace('}' , ')');
          temp = temp.replace('"', '\'');
          // assume that the token's string value might be 'nil' for a missing value
          temp = temp.replaceAll("nil", "NA");
         // if string is long, create a temp file for passing array data 
          String temp1 = temp;
          if (temp1.length()>maxCommandLineLength) {
            temp1 = temp1.replace('(',' '); 
            temp1 = temp1.replace(')',' '); 
            String filename = writeDataFile(temp1, cntr);
            cntr++;
            tempA = label + " <- scan('"+filename+"', sep=',')";
            tempA = tempA + "\n" + "file.remove('"+filename+"')";
            ret = ret + tempA + "\n";  
          } else {   // otherwise use the modified string
              tempA = label + " <- c" + temp;
              ret = ret + tempA + "\n";
          }
 
        }
      }
      labellist = labellist.substring(0,labellist.length()-1); // remove last ','
      ret = ret + portName + " <- data.frame(" + labellist +")";
//  System.out.println("ret: "+ret);
      return ret;
    }

    // there is ta problem when length of lines sent to R are too long
    // thus, break the line into pieces with newlines;
    // assume pieces of are approx. 512 chars but must be separate at ','
    // (R will accept multiple lines but the seperation cannot be arbitrart; i.e. not in
    // middle of floating point number)

    private String breakIntoLines(String temp) {
      int size = 512;
      int pieces = (int)temp.length()/size;
      int start = size;
      int indx = 0;
      for (int k=0;k<pieces-1;k++) {
        indx = temp.indexOf(",", start);
        temp = temp.substring(0,indx) + "\n" + temp.substring(indx,temp.length());
        start = start + size;
      }
      return temp;
    }

    // Execute a command, set _process to point to the subprocess
    // and set up _errorGobbler and _outputGobbler to read data.
    private void _exec() throws IllegalActionException {
        Runtime runtime = Runtime.getRuntime();
        String[] commandArray;

        String osName = System.getProperty("os.name");
        if (osName.equals("Windows NT") || osName.equals("Windows XP") || osName.equals("Windows 2000")) {
          // checkRLocation is commented out for now since it slows down the first execution of a
          // workflow with an RExpression actor too much (>= 30 sec for a 'cold' machine)
          //checkRLocation();  
          commandArray = new String[6];
          commandArray[0] = "cmd.exe";
          commandArray[1] = "/C";
          commandArray[2] = RdotExe;
          commandArray[3] = "--silent";
          commandArray[4] = restoreString;
          commandArray[5] = saveString;
       }
       else if (osName.equals("Windows 95")) {
          //checkRLocation();  
          commandArray = new String[6];
          commandArray[0] = "command.com";
          commandArray[1] = "/C";
          commandArray[2] = RdotExe;
          commandArray[3] = "--silent";
          commandArray[4] = restoreString;
          commandArray[5] = saveString;
       }
       else {
          commandArray = new String[4];
          commandArray[0] = RdotExe;
          commandArray[1] = "--silent";
          commandArray[2] = restoreString;
          commandArray[3] = saveString;
       }

//     System.out.println("commandArray :"+commandArray);
       try{
//     System.out.println("ready to create _process!");
          _process = runtime.exec(commandArray);
        System.out.println("Process :"+_process);
        } catch (Exception e) {
          System.out.println("Problem with creating process in RExpression!");
        }
//  System.out.println("Ready to create threads");
        // Create two threads to read from the subprocess.
        _outputGobbler = new _StreamReaderThread(_process.getInputStream(),
                        "Exec Stdout Gobbler-" + _streamReaderThreadCount++,
                        this);
        _errorGobbler = new _StreamReaderThread(_process.getErrorStream(),
                        "Exec Stderr Gobbler-" +  _streamReaderThreadCount++,
                        this);
        _errorGobbler.start();
        _outputGobbler.start();

        if (_streamReaderThreadCount > 1000) {
          // Avoid overflow in the thread count.
          _streamReaderThreadCount = 0;
        }

        OutputStreamWriter inputStreamWriter = new OutputStreamWriter(_process.getOutputStream());
        _inputBufferedWriter = new BufferedWriter(inputStreamWriter);


    }

    private void checkRLocation() {
        if (RdotExe.equals("R")) {
          List l = new ArrayList();
          // first check 'Program Files' (default)
          findFile(new File("C:/Program Files"),"R.exe", l);
          if (!l.isEmpty()) {
             RdotExe = (File)l.get(0)+"";
             return;
          }
          // then check if in R directly under root
          findFile(new File("C:/R"),"R.exe", l);
          if (!l.isEmpty()) {
             RdotExe = (File)l.get(0)+"";
          }
        }
    }
    
    private String getUniqueFileName(String extender) {
      String time = (new Long(System.currentTimeMillis())).toString();
      String res = time + "." + extender;
      return res;
    }  

    private String writeDataFile(String dat, int cnt) {
      String fn = "";
      try { 
        String home = System.getProperty("user.home");
        home = home.replace('\\','/');
        fn = home+"/"+getUniqueFileName("dat")+cntr;
        File dataFile = new File(fn);
        StringReader is = new StringReader(dat);
        FileWriter os = new FileWriter(dataFile);
        int c;
        while ((c = is.read()) != -1) {
          os.write(c);
        }
        is.close();
        os.close(); 
      } catch (Exception exc) {
        System.out.println("error writing data file! - RExpression");
      }
      return fn;
    }

    private void findFile(File f, String name, List r) {
 	  if (f.isDirectory()) {
	    File[] files = f.listFiles();
        if (files==null) return;
	    for (int i = 0; i < files.length; i++) {
//            System.out.println(files[i]+"");
              findFile(files[i], name, r);
          }
      } else {
            String fn = f +"";
//          System.out.println("fn: "+fn);
            if (fn.indexOf(name)>-1){
			    r.add(f);
          }
		}
	  }


    ///////////////////////////////////////////////////////////////////
    ////                         inner classes                     ////

    // Private class that reads a stream in a thread and updates the
    // stringBuffer.
    private class _StreamReaderThread extends Thread {

        /** Create a _StreamReaderThread.
         *  @param inputStream The stream to read from.
         *  @param name The name of this StreamReaderThread,
         *  which is useful for debugging.
         *  @param actor The parent actor of this thread, which
         *  is used in error messages.
         */
        _StreamReaderThread(InputStream inputStream, String name,
                Nameable actor) {
            super(name);
            _inputStream = inputStream;
            _inputStreamReader =
                new InputStreamReader(_inputStream);
            _actor = actor;
            _stringBuffer = new StringBuffer();
            _keepRunning = true;
            chars = new char[100001];
        }

        /** Read any remaining data in the input stream and return the
         *  data read thus far.  Calling this method resets the
         *  cache of data read thus far.
         */
        public String getAndReset() {
            if (_debugging) {
                try {
                    _debug("getAndReset: Gobbler '" + getName()) ;
 //                           + "' Ready: " + _inputStreamReader.ready()
 //                           + " Available: " + _inputStream.available());
 // the previous lines (now commented out) cause a thread problem because (?)
 // the inputStreamReader is used by the threads monitoring process io.

                } catch (Exception ex) {
                    throw new InternalErrorException(ex);
                }
            }

            // do a final _read before clearing buffer in case some characters
            // are available; this was added to collect information that was
            // sometimes missing on a newer, faster computer ! -- DFH 11/2005
            _read(); //DFH - last chance to read 
            
            String results = _stringBuffer.toString();
            _stringBuffer = new StringBuffer();

            return results;
        }

        /** Read lines from the inputStream and append them to the
         *  stringBuffer.
         */
        public void run() {
            Thread _myThread = Thread.currentThread();
            while (_keepRunning) {
                //System.out.println("Starting read");
                _read();
                try{
                  Thread.currentThread().sleep(100);
                } catch (Exception e) {
                  System.out.println("Error in TestApp while sleeping!");
                }

                //System.out.println("Finishing read");
            }
        }

        public void quit() {
          _keepRunning = false;
        }

        // Read from the stream until we get to the end of the stream
        private void _read() {
            // We read the data as a char[] instead of using readline()
            // so that we can get strings that do not end in end of
            // line chars.

//            char [] chars = new char[20001];
            int length; // Number of characters read.

            try {
                // Oddly, InputStreamReader.read() will return -1
                // if there is no data present, but the string can still
                // read.
                length = _inputStreamReader.read(chars, 0, 20000);
                    if (_debugging) {
                        // Note that ready might be false here since
                        // we already read the data.
                        System.out.println("_read(): Gobbler '" + getName()
                                + "' Ready: " + _inputStreamReader.ready()
                                + " Value: '"
                                + String.valueOf(chars, 0, length) + "'");
                    }
                    if (length>0) {
                    String temp = String.valueOf(chars, 0, length);
  //                  _stringBuffer.append(chars, 0, length);
                    _stringBuffer.append(temp);
                    }
            } catch (Throwable throwable) {
                 System.out.println("In catch block of _read!");
                 _keepRunning = false;
            }
        }
    
        

        
        // character array
        private char [] chars;

        // The actor associated with this stream reader.
        private Nameable _actor;

        // StringBuffer to update.
        private StringBuffer _stringBuffer;

        // Stream from which to read.
        private InputStream _inputStream;

        // Stream from which to read.
        private InputStreamReader _inputStreamReader;

        // this thread
        private boolean _keepRunning;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private variables                 ////

    // The subprocess gets its input from this BufferedWriter.
    private BufferedWriter _inputBufferedWriter;

    // StreamReader with which we read stderr.
    private _StreamReaderThread _errorGobbler;

    // StreamReader with which we read stdout.
    private _StreamReaderThread _outputGobbler;

    // The Process that we are running.
    private Process _process;

    // Indicator that stopFire() has been called.
    private boolean _stopFireRequested = false;

    private String graphicsOutputFile = "";

    // Instance count of output and error threads, used for debugging.
    // When the value is greater than 1000, we reset it to 0.
    private static int _streamReaderThreadCount = 0;

    private List opList;
    private Iterator iter_o;

}
