/**
 *    '$RCSfile: NewActorWizardTableau.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2005/11/01 20:39:07 $'
 *   '$Revision: 1.8 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.workflow.gui;

import java.awt.Color;
import java.awt.Container;
import java.awt.event.KeyEvent;
import java.util.List;

import javax.swing.JMenu;
import javax.swing.JMenuItem;

import ptolemy.actor.gui.Configuration;
import ptolemy.actor.gui.Effigy;
import ptolemy.actor.gui.PtolemyEffigy;
import ptolemy.actor.gui.Tableau;
import ptolemy.actor.gui.TableauFactory;
import ptolemy.actor.gui.TableauFrame;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;

/**
 * A tableau that creates a wizard to create a new actor.  This actor may be
 * a stub class or it may be implemented by some other class.
 */
public class NewActorWizardTableau extends Tableau
{
  private static Color BACKGROUND_COLOR = new Color(0xe5e5e5);

  private CompositeEntity actorLibrary;

  /**
   *  Create a new panel for the wizard with the given
   *  effigy.  The tableau is itself an entity contained by the effigy
   *  and having the specified name.  The frame is not made visible
   *  automatically.  You must call show() to make it visible.
   *
   * @param  container                     The containing effigy.
   * @param  name                          The name of this tableau within the
   *   specified effigy.
   * @exception  IllegalActionException    If the tableau is not acceptable
   *   to the specified container.
   * @exception  NameDuplicationException  If the container already contains
   *   an entity with the specified name.
   */
  public NewActorWizardTableau(PtolemyEffigy container, String name)
    throws IllegalActionException, NameDuplicationException
  {
    super(container, name);

    //create the wizard frame
    Effigy e = container.topEffigy();
    Container parent = e.showTableaux().getFrame();
    if(parent == null) System.out.println("parent is null");
    System.out.println("e:");
    printList(e.entityList()); 
    System.out.println("");

    /*
      OK, so here's the lowdown on 'actor library'.  the structure of this beast
      is as follows:
      actor library
        actor library
          sources
          sinks
          io
          math
          ...
          dynamicActor

      Note the nested actor library.  I'm not sure why that is but it may
      have something to do with the container object.
    */

    //System.out.println("getting configuration: " + parent.toString());
    Configuration c = (Configuration)e.toplevel();
    
    if(c == null) System.out.println("c is null");
    actorLibrary = (CompositeEntity)c.getEntity("actor library");
    if(actorLibrary == null) System.out.println("actorLibrary is null");

    System.out.println("actorLibrary:");
    printList(actorLibrary.entityList()); 
    System.out.println("");
    
    CompositeEntity actorLibNest =
      (CompositeEntity)actorLibrary.getEntity("Actors");
    if(actorLibNest == null) System.out.println("actorLibNest is null");

    CompositeEntity dynActor;
    try
    {
      dynActor =
        (CompositeEntity)actorLibNest.getEntity("dynamicActor");
      printList(dynActor.entityList());
    }
    catch(NullPointerException npe)
    {
      throw new IllegalActionException("The library dynamicActor could not be " +
        "found.  Please create this library in the Actor Library MOML and " +
        "try again.");
    }

    NewActorFrame NAFrame = new NewActorFrame(parent);
    //tell the actor frame to add new actors to this library
    NAFrame.setActorLibrary(dynActor);
    NAFrame.setContainer(container);
    NAFrame.setBackground(BACKGROUND_COLOR);

    setFrame(NAFrame);
  }

  /**
   * prints a list
   */
  private static void printList(List l)
  {
    for(int i=0; i<l.size(); i++)
    {
      //DebugPrinter.getInstance().print(((NamedObj)l.get(i)).getName(), 2);
      System.out.print(((NamedObj)l.get(i)).getName() + ",");
    }
  }

  /**
   * A factory that creates run control panel tableaux for Ptolemy models.
   */
  public static class Factory extends TableauFactory
  {

    /**
     * Create a factory with the given name and container.
     *
     * @param  container                     The container.
     * @param  name                          The name.
     * @exception  IllegalActionException    If the container is incompatible
     *   with this attribute.
     * @exception  NameDuplicationException  If the name coincides with
     *   an attribute already in the container.
     */
    public Factory(NamedObj container, String name)
      throws IllegalActionException, NameDuplicationException
    {
      super(container, name);
    }

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /**
     * If the specified effigy already contains a tableau named
     *  "runTableau", then return that tableau; otherwise, create
     *  a new instance of RunTableau for the effigy, and
     *  name it "runTableau".  If the specified effigy is not an
     *  instance of PtolemyEffigy, then do not create a tableau
     *  and return null. It is the responsibility of callers of
     *  this method to check the return value and call show().
     *
     * @param  effigy         The model effigy.
     * @return                A new run tableau if the effigy is a PtolemyEffigy,
     *    or null otherwise.
     * @exception  Exception  If the factory should be able to create a
     *   tableau for the effigy, but something goes wrong.
     */
    public Tableau createTableau(Effigy effigy)
      throws Exception
    {
      if(effigy instanceof PtolemyEffigy)
      {
        // First see whether the effigy already contains a RunTableau.
        NewActorWizardTableau tableau =
            (NewActorWizardTableau)effigy.getEntity("NewActorWizardTableau");

        if(tableau == null)
        {
          tableau = new NewActorWizardTableau(
              (PtolemyEffigy)effigy, "NewActorWizardTableau");
        }
        // Don't call show() here, it is called for us in
        // TableauFrame.ViewMenuListener.actionPerformed()
        return tableau;
      }
      else
      {
        return null;
      }
    }
  }

  /**
   * A factory that creates run control panel tableaux for the model
   *  associated with a top-level effigy (one that has a file
   *  representation).
   */
  public static class TopFactory extends Factory
  {

    /**
     * Create a factory with the given name and container.
     *
     * @param  container                     The container.
     * @param  name                          The name.
     * @exception  IllegalActionException    If the container is incompatible
     *   with this attribute.
     * @exception  NameDuplicationException  If the name coincides with
     *   an attribute already in the container.
     */
    public TopFactory(NamedObj container, String name)
      throws IllegalActionException, NameDuplicationException
    {
      super(container, name);
    }

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /**
     * Create a tableau to run the model associated with the specified
     *  effigy.  The top-level effigy, as returned by
     *  {@link Effigy#topEffigy()}, is the one that is run.
     *  If that effigy already contains a tableau named
     *  "runTableau", then return that tableau; otherwise, create
     *  a new instance of RunTableau for the top effigy, and
     *  name it "runTableau".  If the specified effigy is not an
     *  instance of PtolemyEffigy, then do not create a tableau
     *  and return null. It is the responsibility of callers of
     *  this method to check the return value and call show().
     *
     * @param  effigy         The model effigy.
     * @return                A new run tableau if the effigy is a PtolemyEffigy,
     *    or null otherwise.
     * @exception  Exception  If the factory should be able to create a
     *   tableau for the effigy, but something goes wrong.
     */
    public Tableau createTableau(Effigy effigy)
      throws Exception
    {
      return super.createTableau(effigy.topEffigy());
    }
  }

  ///////////////////////////////////////////////////////////////////
  ////                         inner classes                     ////

  /**
   * The frame that is created by an instance of RunTableau.
   */
  public class WizardFrame extends TableauFrame
  {

    ///////////////////////////////////////////////////////////////////
    ////                         protected variables               ////

    /**
     * Debug menu for this frame.
     */
    protected JMenu _debugMenu;

    /**
     * Construct a frame to control the specified Ptolemy II model.
     *  After constructing this, it is necessary
     *  to call setVisible(true) to make the frame appear.
     *  This is typically accomplished by calling show() on
     *  enclosing tableau.
     *
     * @param  model    The model to put in this frame, or null if none.
     * @param  tableau  The tableau responsible for this frame.
     */
    public WizardFrame(Tableau tab)
    {
      super(tab);
    }

    ///////////////////////////////////////////////////////////////////
    ////                         protected methods                 ////

    /**
     * Add a Debug menu.
     */
    protected void _addMenus()
    {
      super._addMenus();

      JMenuItem[] debugMenuItems = {
          new JMenuItem("Listen to Manager", KeyEvent.VK_M),
          new JMenuItem("Listen to Director", KeyEvent.VK_D),
          };

      // NOTE: This has to be initialized here rather than
      // statically because this method is called by the constructor
      // of the base class, and static initializers have not yet
      // been run.
      _debugMenu = new JMenu("Debug");
      _debugMenu.setMnemonic(KeyEvent.VK_D);


      // Set the action command and listener for each menu item.
      for(int i = 0; i < debugMenuItems.length; i++)
      {
        debugMenuItems[i].setActionCommand(debugMenuItems[i].getText());
        _debugMenu.add(debugMenuItems[i]);
      }
      _menubar.add(_debugMenu);
    }
  }
}

