/**
 * For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.authentication;

/**
 * AuthenticationManager manages all the authentication issue for the user.
 * It helps the user to get the proxy from the GAMA server.
 * For details: http://kepler-project.org/Wiki.jsp?page=KeplerAuthenticationFramework
 * Note: all of the get Proxy methods are synchronized to avoid multiple entries (which
 * may cause multiple user login dialogs appear) from the actors.
 * @author Zhijie Guan 
 * guan@sdsc.edu
 *
 */

public class AuthenticationManager {

	// ProxyRepository is used to store all of the proxies 
	private ProxyRepository proxyRepository = new ProxyRepository();
	
	// Singleton template is used here to make sure there is only one AuthenticationManager in the system
	private static AuthenticationManager authenticationManager = new AuthenticationManager();
	
	private AuthenticationManager() {}		// Constructor. Do nothing. Use "private" to avoid creating a new AuthenticationManager
	
	/**
	 * Part of singleton template, the only way to get an instance of AuthenticationManager
	 * @return The unique AuthenticationManager instance
	 */
	public static AuthenticationManager getManager() {
		return authenticationManager;
	}
	
	/**
	 * This function retrieves back the default proxy for the user
	 * @return The default proxy
	 */
	public synchronized ProxyEntity getProxy() {
		while (proxyRepository.getDefaultProxy() == null) {	// No proxy exists
			LoginGUI.fire(proxyRepository);					// Launch the login dialog
			proxyRepository.waitForUserLogin();				// Wait in ProxyRepository for user login
		}
		return proxyRepository.getDefaultProxy();
	}
	
	/**
	 * This function retrieves back the user proxy within specified domain 
	 * @param domain Specified domain
	 * @return The requested proxy
	 */
	public synchronized ProxyEntity getProxy(Domain domain) {
		int proxyIndex;
		while ( ( proxyIndex = proxyRepository.searchProxyInRepository(domain) ) == -1 ) { 	// No such proxy exists
			LoginGUI.fire(proxyRepository); 												// Launch the login dialog
			proxyRepository.waitForUserLogin();												// Wait in ProxyRepository for user login
		}
		return proxyRepository.getProxyAt(proxyIndex);
	}
	
	/**
	 * This function retrieves the user proxy with specified lifetime
	 * Note: since currently GAMA server does not support setting lifetime for proxy,
	 *       we decide to postpone the implementation of this function.
	 * @param lifetime
	 * @return The default proxy
	 */
	public synchronized ProxyEntity getProxy(LifeTime lifetime) {
		return this.getProxy();
	}
	
	/**
	 * This function retrieves the user proxy with specified lifetime and domain
	 * Note: since currently GAMA server does not support setting lifetime for proxy,
	 *       we decide to postpone the implementation of this function. 
	 * @param domain
	 * @param lifetime
	 * @return The default proxy
	 */
	public synchronized ProxyEntity getProxy(Domain domain, LifeTime lifetime) {
		return this.getProxy();
	}
}
