/*
 *  The node controller for actor instances.
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.gui;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Constructor;
import java.util.Iterator;

import javax.swing.JOptionPane;
import javax.swing.tree.TreePath;

import org.ecoinformatics.seek.sms.AnnotationEngine;
import org.ecoinformatics.seek.sms.KeplerLocalLSIDService;
import org.ecoinformatics.seek.sms.OntologyCatalog;

import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.NamedObj;
import ptolemy.vergil.basic.BasicGraphFrame;


/**
 * This action adds actors to the tree
 *
 *@author     Chad Berkley
 *@created    February 17, 2005
 */
public class AddActorAction extends TreeAction
{
  private final static String LABEL = "Add Actor to Folder...";
  private NewActorFrame naFrame;

  /**
   * Constructor
   *
   *@param  path    the TreePath where the actor is being added.
   *@param  parent  Description of the Parameter
   */
  public AddActorAction(TreePath path, Component parent)
  {
    super(path, parent, LABEL);
  }
  
  /**
   * Invoked when an action occurs.
   *
   *@param  e  ActionEvent
   */
  public void actionPerformed(ActionEvent e)
  {
    super.actionPerformed(e);
    Component current = parent;
    while (parent != null && !(parent instanceof BasicGraphFrame))
    {
      parent = current.getParent();
      current = parent;
    }

    naFrame = new NewActorFrame(parent);
    ActionHandler handler = new ActionHandler();
    naFrame.addActionListener(handler);
    naFrame.setVisible(true);
  }


  /**
   * Listener used to changes from the NewFolderFrame
   *
   *@author     berkley
   *@created    March 1, 2005
   */
  private class ActionHandler implements ActionListener
  {
    /**
     *  Description of the Method
     *
     *@param  event  Description of Parameter
     */
    public void actionPerformed(ActionEvent event)
    {
      String command = event.getActionCommand();
      boolean success = false;

      if (command.equals("okbutton_clicked"))
      {
        String actorName = naFrame.getActorName();
        String className = naFrame.getClassName();
        String conceptLabel = ((CompositeEntity) path.getLastPathComponent()).getName();
        success = addActor(className, actorName, conceptLabel);
        if (success)
        {
          naFrame.setVisible(false);
          naFrame.dispose();
          notifyListeners("new_actor_created");
        }
      }
      else if (command.equals("cancelbutton_clicked"))
      {
        naFrame.setVisible(false);
        naFrame.dispose();
      }
    }


    /**
     *  Adds a feature to the Actor attribute of the ActionHandler object
     *
     *@param  className     The feature to be added to the Actor attribute
     *@param  actorName     The feature to be added to the Actor attribute
     *@param  conceptLabel  The feature to be added to the Actor attribute
     *@return               Description of the Return Value
     */
    private boolean addActor(String className, String actorName, String conceptLabel)
    {
      KeplerLocalLSIDService service = KeplerLocalLSIDService.instance();
      AnnotationEngine engine = AnnotationEngine.instance();
      OntologyCatalog catalog = OntologyCatalog.instance();
      String namespace = "local";

      // get the MoML Object for the class
      NamedObj actor;
      try
      {
        actor = createNamedObj(className, actorName);
      }
      catch (Exception e)
      {
        String msg = "Unable to insert Actor: Be sure the class path exists (e.g., " +
            "ptolemy.actor.lib.Const) and that \n the ComponentEntity(CompositeEntity, String) " +
            "constructor is implemented by the actor.";
        JOptionPane.showMessageDialog(null, msg, "alert", JOptionPane.ERROR_MESSAGE);
        return false;
      }

      // get the concept name for the parent folder
      String conceptName = catalog.getConceptNameWithLabel(conceptLabel);
      if (conceptName == null)
      {
        JOptionPane.showMessageDialog(null, "Something bad happened", "alert", JOptionPane.ERROR_MESSAGE);
        return false;
      }

      // check if it already exists in the library
      Iterator lsidIter = service.getLSIDFor(actor);

      try
      {
        // if it doesn't already exist
        if (!lsidIter.hasNext())
        {
          String newLSID = service.createLocallyUniqueLSID(namespace);
          service.assignLSID(newLSID, actor);
          engine.addActorAnnotation(newLSID, conceptName);
        }
        else
        {
          // it exists, so update with new annotation and actor
          while (lsidIter.hasNext())
          {
            String nextLSID = (String) lsidIter.next();
            service.updateLSID(nextLSID, actor);
            engine.addActorAnnotation(nextLSID, conceptName);
          }
        }
        service.commitChanges();
      }
      catch (Exception e)
      {
        JOptionPane.showMessageDialog(null, e.getMessage(), "alert", JOptionPane.ERROR_MESSAGE);
        return false;
      }

      return true;
    }


    /**
     *  We assume for now that added items have the construct:
     *  ComponentEntity(CompositeEntity container, java.lang.String
     *  name)
     *
     *@param  className      Description of the Parameter
     *@param  actorName      Description of the Parameter
     *@return                Description of the Return Value
     *@exception  Exception  Description of the Exception
     */
    private NamedObj createNamedObj(String className, String actorName)
      throws Exception
    {
      // create the class
      Class classDef = Class.forName(className);

      //add the arg types
      Class[] args = new Class[]{CompositeEntity.class, String.class};

      //create a constructor
      Constructor constructor = classDef.getConstructor(args);

      // set the args
      Object[] argImp = new Object[]{new CompositeEntity(), actorName};

      //create the object
      NamedObj obj = (NamedObj) constructor.newInstance(argImp);

      return obj;
    }

  }
}

