/**
 *    '$RCSfile: ImportArchiveAction.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2006/03/07 18:53:02 $'
 *   '$Revision: 1.4 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.gui;

import java.io.*;
import java.util.*;

import java.awt.event.ActionEvent;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.KeyStroke;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import diva.gui.GUIUtilities;
import ptolemy.actor.gui.TableauFrame;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.StaticResources;
import ptolemy.vergil.toolbox.FigureAction;
import ptolemy.moml.EntityLibrary;
import ptolemy.kernel.util.Attribute;

import org.kepler.objectmanager.*;
import org.kepler.objectmanager.lsid.*;
import org.kepler.objectmanager.cache.*;
import org.kepler.objectmanager.library.*;
import org.kepler.sms.*;

import com.ibm.lsid.LSIDException;


/**
 * This action opens the Actor Tabbed Dialog.
 *
 *@author     Matthew Brooke
 *@created    26 February 2006
 */
public class ImportArchiveAction extends FigureAction {

  //////////////////////////////////////////////////////////////////////////////
  // LOCALIZABLE RESOURCES - NOTE that these default values are later
  // overridden by values from the uiDisplayText resourcebundle file
  //////////////////////////////////////////////////////////////////////////////

  private static String DISPLAY_NAME
    = StaticResources.getDisplayString("actions.actor.displayName",
                                       "Import Archive (KAR)");
  private static String TOOLTIP
    = StaticResources.getDisplayString("actions.actor.tooltip",
                                       "Import a KAR file archive.");
  private static ImageIcon LARGE_ICON = null;
  private static KeyStroke ACCELERATOR_KEY = null;
//    = KeyStroke.getKeyStroke(KeyEvent.VK_C, Toolkit.getDefaultToolkit().
//                             getMenuShortcutKeyMask());
////////////////////////////////////////////////////////////////////////////////


  /**
   * Constructor
   *
   * @param parent the "frame" (derived from ptolemy.gui.Top) where the menu is
   *   being added.
   */
  public ImportArchiveAction(TableauFrame parent) {
    super("");
    if (parent == null) {
      IllegalArgumentException iae = new IllegalArgumentException(
        "ImportArchiveAction constructor received NULL argument for TableauFrame");
      iae.fillInStackTrace();
      throw iae;
    }
    this.parent = parent;

    this.putValue(Action.NAME, DISPLAY_NAME);
    this.putValue(GUIUtilities.LARGE_ICON, LARGE_ICON);
    this.putValue("tooltip", TOOLTIP);
    this.putValue(GUIUtilities.ACCELERATOR_KEY, ACCELERATOR_KEY);
  }


  /**
   * Invoked when an action occurs.
   *
   *@param  e  ActionEvent
   */
  public void actionPerformed(ActionEvent e) {

    //must call this first...
    super.actionPerformed(e);
    //...before calling this:
    NamedObj target = super.getTarget();
    JFileChooser chooser = new JFileChooser();
    KARFileFilter filter = new KARFileFilter();
    chooser.setFileFilter(filter);
    int returnVal = chooser.showOpenDialog(parent);
    if(returnVal == JFileChooser.APPROVE_OPTION) 
    {
      //process the given kar file
      File karFile = chooser.getSelectedFile();
      processKSW(karFile);
    }
  }
  
  /**
   * process the new kar file into the actor library
   * @param kswFile the file to process
   */
  private void processKSW(File kswFile)
  {
    try
    {
      KeplerLSID[] actorLSIDs;
      CacheManager cache = CacheManager.getInstance();
      KARCacheObject kco = new KARCacheObject(kswFile);
      cache.insertObject(kco);
      try
      {
        actorLSIDs = kco.getActors();
      }
      catch(LSIDException lside)
      {
        JOptionPane.showMessageDialog(null,
               "Error getting actor lsid: " + lside.getMessage(), "alert", 
               JOptionPane.ERROR_MESSAGE);
        return;
      }
      
      if(actorLSIDs != null)
      {
        for(int j=0; j<actorLSIDs.length; j++)
        {
          ActorCacheObject aco = (ActorCacheObject)cache.getObject(actorLSIDs[j]);
          ActorMetadata am = aco.getMetadata();
          try
          { //build an entity for each actor in the ksw and add it to the actorLibrary
            NamedObj actorEntity = am.getActorAsNamedObj(null);
            LSIDGenerator lsidGenerator = LSIDGenerator.getInstance();
            //KeplerLSID newlsid = lsidGenerator.getNewLSID(am.getLSID());
            
            Iterator attItt = actorEntity.attributeList().iterator();
            Vector subtrees = new Vector();
            while(attItt.hasNext())
            {
              Attribute a = (Attribute)attItt.next();
              if(a.getClassName().equals("org.kepler.sms.SemanticType"))
              {
                SemanticType st = (SemanticType)a;
                LibraryIndexItem subtree = 
                  LibraryIndex.getInstance().findItem(st.getConceptName());
                if(subtree != null)
                {
                  subtrees.addElement(subtree);
                }
              }
            }
            
            if(subtrees.size() == 0)
            { //if there are no semantic types, give an error msg
              JOptionPane.showMessageDialog(null,
                "There are no semantic types defined for the actor in  " +
                "this KAR file.  Without a semantic type, it cannot be " + 
                "placed in the tree.  Please add a semantic type.", "ALERT", 
                 JOptionPane.ERROR_MESSAGE);
              return;
            }
            
            Iterator subTreeItt = subtrees.iterator();
            while(subTreeItt.hasNext())
            { //if there are semantic types, put the actor in the tree
              LibraryIndexItem subtree = (LibraryIndexItem)subTreeItt.next();
              LibraryIndex.getInstance().add(
                new LibraryIndexComponentItem(am.getName(), am.getLSID()), subtree);
              System.out.println("actor " + actorEntity.getName() +
                " loaded into the library in category " +
                subtree.getName() + " with id " + 
                am.getId());
            }
            LibraryIndex.getInstance().refresh();
            util.EmptyChangeRequest request = 
              new util.EmptyChangeRequest(actorEntity, "request change");
            actorEntity.requestChange(request);
          }
          catch(Exception e)
          {
            e.printStackTrace();
            JOptionPane.showMessageDialog(null,
               "Error creating ComponentEntity from actor " +
              "metadata: " + e.getMessage(), "alert", 
               JOptionPane.ERROR_MESSAGE);
            return;
          }
        }
      }
    }
    catch(CacheException ce)
    {
      JOptionPane.showMessageDialog(null,
             "Error with the cache: " + ce.getMessage(), "alert", 
             JOptionPane.ERROR_MESSAGE);
      return;
    }
  }

  private TableauFrame parent;
  private TabbedDialog actorDialog;

  private static final Log log
    = LogFactory.getLog("UI." + ActorDialogAction.class.getName());

  private static final boolean isDebugging = log.isDebugEnabled();
  
  /**
   * private class to select kar files
   */
  private class KARFileFilter extends javax.swing.filechooser.FileFilter
  {
    public KARFileFilter()
    {
      
    }
    
    public String getDescription()
    {
      return "KAR Files";
    }
    
    public boolean accept(File f)
    {
      if(f.isDirectory() || f.getName().endsWith(".kar"))
      {
        return true;
      }
      
      return false;
    }
  }
}
