/*
 *  An object that can create a tableau for a model.
 *  Copyright (c) 1997-2003 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 *  @ProposedRating Green (eal@eecs.berkeley.edu)
 *  @AcceptedRating Yellow (celaine@eecs.berkeley.edu)
 */
package org.kepler.gui;

import java.util.Iterator;

import ptolemy.actor.gui.Configuration;
import ptolemy.actor.gui.Effigy;
import ptolemy.actor.gui.Tableau;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.vergil.tree.EntityTreeModel;

//////////////////////////////////////////////////////////////////////////
//// TableauFactory
/**
 *  This class is an attribute that creates a tableau to view a specified
 *  effigy. When a model is opened, if the top-level of that model contains this
 *  attribute or a subclass, then that attribute handles constructing a tableau.
 *  Otherwise, the configuration specifies which tableau is used. A
 *  configuration contains an instance of this class, and uses it to create a
 *  tableau for a model represented by an effigy. This base class assumes that
 *  it contains other tableau factories. Its createTableau() method defers to
 *  each contained factory, in the order in which they were added, until one is
 *  capable of creating a tableau for the specified effigy. Subclasses of this
 *  class will usually be inner classes of a Tableau, and will create the
 *  Tableau, or might themselves be aggregates of instances of TableauFactory.
 *  <p>
 *
 *  When there are multiple distinct TableauFactory classes that are capable of
 *  providing views on the same effigy, then instances of these factories should
 *  be aggregated into a single factory contained herein. Those instances can be
 *  presented as alternative views of the data when any single view is opened.
 *
 *@author     Steve Neuendorffer and Edward A. Lee
 *@created    February 17, 2005
 *@version    $Id: LibraryPaneFactory.java,v 1.2 2004/08/17 18:22:42 berkley Exp
 *      $
 *@since      Ptolemy II 1.0
 *@see        Configuration
 *@see        Effigy
 *@see        Tableau
 */
public class LibraryPaneFactory extends Attribute
{

  /**
   *  Create a factory with the given name and container.
   *
   *@param  container                     The container.
   *@param  name                          The name.
   *@exception  IllegalActionException    If the container is incompatible with
   *      this attribute.
   *@exception  NameDuplicationException  If the name coincides with an
   *      attribute already in the container.
   */
  public LibraryPaneFactory(NamedObj container, String name)
    throws IllegalActionException, NameDuplicationException
  {
    super(container, name);
  }


  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////

  /**
   *  Create a tableau for the specified effigy. The tableau will be created
   *  with a new unique name with the specified effigy as its container. If this
   *  factory cannot create a tableau for the given effigy (perhaps because the
   *  effigy is not of the appropriate subclass), then return null. This base
   *  class assumes that it contains other tableau factories. This method defers
   *  to each contained factory in order until one is capable of creating a
   *  tableau for the specified effigy. As with all attributes, the order is
   *  simply the order of creation. Subclasses of this class will usually be
   *  inner classes of a Tableau, and will create the Tableau. A subclass that
   *  actually creates a tableau is responsible for setting the container of the
   *  tableau to the specified effigy, and for naming the tableau. Subclasses
   *  should not call show() in createTableau(), it is the responsibility of the
   *  caller to check the return value and call show() after doing things like
   *  adjusting the size.
   *
   *@param  _libraryModel  Description of the Parameter
   *@param  config         Description of the Parameter
   *@return                A tableau for the effigy, or null if one cannot be
   *      created.
   */
  public LibraryPane createLibraryPane(EntityTreeModel _libraryModel,
      Configuration config)
  {
    LibraryPane pane = null;
    Iterator factories = attributeList(LibraryPaneFactory.class).iterator();
    while (factories.hasNext() && pane == null)
    {
      LibraryPaneFactory factory = (LibraryPaneFactory) factories.next();
      pane = factory.createLibraryPane(_libraryModel, config);
    }
    return pane;
  }
}

