/**
 *    '$RCSfile: CacheManager.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2006/03/14 21:34:30 $'
 *   '$Revision: 1.22 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.objectmanager.cache;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import org.ecoinformatics.util.Config;
import org.ecoinformatics.util.DBConnection;
import org.ecoinformatics.util.DBConnectionFactory;
import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.objectmanager.lsid.LSIDResolver;
import org.kepler.objectmanager.lsid.LSIDTree;

/**
 * This class represents a disk cache of CacheObjects.  The cache manages
 * cache objects by calling their lifecycle event handlers and serializing
 * every time a change is made.  Objects in the cache each have a unique
 * lsid.  Once an object is in the cache, it will not be updated unless
 * the lsid changes.
 * 
 * this class uses hsql to keep track of cache entries.  The tables look like 
 * this:
 * cacheContentTable
 * -----------------
 * varchar name //name of the entry
 * varchar lsid //lsid of the entry
 * varchar date //timestamp of the entry
 * varchar file //filename of the serialized entry
 */
public class CacheManager
{
  private static CacheManager cache = null;
  
  private Hashtable objectHash = new Hashtable();
  
  private Vector listeners = new Vector();
  
  protected static String cachePath = Config.getCacheDirPath();
  private static String objectPath = cachePath + "objects";
  public final static String tmpPath = cachePath + "tmp";
  private static String CACHETABLENAME = "cacheContentTable";
  
  private LSIDTree lsidTree = LSIDTree.getInstance();
  
  private DBConnection conn = null;
  
  private PreparedStatement insertStatement = null;
  private PreparedStatement selectByLSIDStatement = null;
  private PreparedStatement updateStatement = null;
  
  /**
   * Construct a new CacheManager
   */
  protected CacheManager()
    throws CacheException
  {
    try {
    	conn = DBConnectionFactory.getDBConnection();
    }
    catch( Exception e ) {
        e.printStackTrace();
        throw new CacheException("Error obtaining database connection: " + 
          e.getMessage());
    }
    /*
     * Create the prepared statements.
     */
    try {
    	insertStatement = conn.prepareStatement("insert into " + CACHETABLENAME + " (name, lsid, date, file, expiration) values ( ?, ?, ?, ?, ? )");
    	selectByLSIDStatement = conn.prepareStatement("select name, lsid, file, expiration from " + CACHETABLENAME + " where lsid= ?");
    	updateStatement = conn.prepareStatement("update " + CACHETABLENAME + " set name = ?, date = ?, file = ?, expiration = ? where lsid = ?");
    }
    catch( SQLException e ) {
    	e.printStackTrace();
    	throw new CacheException("Unable to create prepared statements.");
    }

    try {
        Vector expiredItems = new Vector();
        String sql = "select lsid, expiration from " + CACHETABLENAME;
        ResultSet rs = conn.executeSQLQuery(sql);
        if ( rs.first() ) {
            while (rs.next() ) {
                try {
                    String lsidString = rs.getString("lsid");
                    CacheExpiration expiry = CacheExpiration.newInstance( rs.getString("expiration"));
                    if ( expiry == CacheExpiration.SESSION || expiry.isExpired() ) {
                        expiredItems.add( lsidString );
                    }
                }
                catch( Exception e ) {
                    System.err.println("Unable to process cache entry");
                    e.printStackTrace();
                }

            }
        }
        for( Iterator i = expiredItems.iterator(); i.hasNext(); ) {
            try {
                String lsid = (String) i.next();
                removeObject( new KeplerLSID(lsid) );
            }
            catch (Exception e ) {
                System.err.println("Unable to remove cache entry");
                e.printStackTrace();
            }
        }
    } catch ( SQLException e ) {
        e.printStackTrace();
        throw new CacheException("Problems clearing stale cache entries.");
    }

  }
  
  /**
   * create a new singleton instance of CacheManager
   */
  public static synchronized CacheManager getInstance()
    throws CacheException
  {
    if(cache == null)
    {
      cache = new CacheManager();
    }
    return cache;
  }
  
  /**
   * for testing serialization only.  do not use in production classes.  use
   * getInstance instead.
   */
  public static CacheManager getNewInstance()
    throws CacheException
  {
    return new CacheManager();
  }
  
  /**
   * insert a new CacheObjectInterface into the cache.
   */
  public synchronized void insertObject(CacheObjectInterface co)
    throws CacheException
  {
    //get the critical info
    String name = co.getName();
    String lsid = co.getLSID().toString();
    String date = new Long(System.currentTimeMillis()).toString();
    String filename = co.getLSID().createFilename();
    String expiration = co.getExpiration().toString();
    //save the entry to the DB
    try
    {
      insertStatement.setString(1,name);
      insertStatement.setString(2,lsid.toString());
      insertStatement.setString(3,date);
      insertStatement.setString(4,filename);
      insertStatement.setString(5,expiration);
      insertStatement.executeUpdate();
      insertStatement.clearParameters();
      objectHash.put(lsid, co);
      serializeObjectInFile(co, filename);
      lsidTree.addLSID(new KeplerLSID(lsid));
      conn.commit();
    }
    catch(org.kepler.objectmanager.lsid.LSIDTreeException lsidte)
    {
      throw new CacheException("The LSID " + lsid + 
        " is already in use: " + lsidte.getMessage());
    }
    catch(Exception sqle)
    {
      try
      {
    	  conn.rollback();
      }
      catch(Exception e)
      {
        throw new CacheException("Could not roll back the database after error " + sqle.getMessage(), e);
      }
      sqle.printStackTrace();
      throw new CacheException("Could not create hsql entry for new CacheObjectInterface: ", sqle );
    }
    notifyListeners(co, "add");
  }
  
  /**
   * update a CacheObjectInterface in the cache.
   */
  public synchronized void updateObject(CacheObjectInterface co)
    throws CacheException
  {
    //get the critical info
    String name = co.getName();
    String lsid = co.getLSID().toString();
    String date = new Long(System.currentTimeMillis()).toString();
    String filename = co.getLSID().createFilename();
    String expiration = co.getExpiration().toString();
    //save the entry to the DB
    try
    {
      updateStatement.setString(1,name);
      updateStatement.setString(2,date);
      updateStatement.setString(3,filename);
      updateStatement.setString(4,expiration);
      updateStatement.setString(5,lsid.toString());
      updateStatement.executeUpdate();
      updateStatement.clearParameters();
      serializeObjectInFile(co, filename);
      conn.commit();
    }
    catch(Exception sqle)
    {
      try
      {
    	  conn.rollback();
      }
      catch(Exception e)
      {
        throw new CacheException("Could not roll back the database after error " + sqle.getMessage(),e);
      }
      throw new CacheException("Could not create hsql entry for new CacheObjectInterface: ",sqle);
    }
    notifyListeners(co, "update");
  }

  /**
   * remove the CacheObjectInterface with the specified lsid and return it.
   * @param lsid
   */
  public void removeObject(KeplerLSID lsid)
    throws CacheException
  {
    // grab a copy from the hash to use for calling listeners.  There will be
    // no listeners on an object returned from the db.
    CacheObjectInterface co = (CacheObjectInterface)objectHash.get(lsid.toString());
    try
    {
      String sql = "select file from " + CACHETABLENAME + " where lsid='" + lsid.toString() + "'";
      ResultSet rs = conn.executeSQLQuery(sql);
      rs.first();
      File f = new File(objectPath, rs.getString("file"));
      f.delete();
      sql = "delete from " + CACHETABLENAME + " where lsid='" + lsid.toString() + "'";
      conn.executeSQLCommand(sql);
      conn.commit();
      objectHash.remove(lsid.toString());
      lsidTree.removeLSID(lsid);
    }
    catch(Exception e)
    {
    	try {
    		conn.rollback();
    	}
    	catch( Exception e1 ) {
    	        throw new CacheException("Could not roll back the database after error " + e.getMessage(), e1);    	
    	}
      throw new CacheException("Error removing object " + lsid + " from the cache",e);
    }
    if ( co != null ) {
        notifyListeners(co, "remove");
    }
  }
  
  /**
   * return the CacheObjectInterface with the specified lsid.  Returns null if the object is not in the
   * cache and cannot be resolved.
   * @param lsid
   * @throws CacheException when there is an issue with the cache.
   */
  public CacheObjectInterface getObject(KeplerLSID lsid)
    throws CacheException
  {
    //first look in the hash table
    CacheObjectInterface co = (CacheObjectInterface)objectHash.get(lsid.toString());
    
    // Found it in the hash - return it.
    if ( co != null ) {
        // Test if it has expired by date.  Note, the getObject() will not expire SESSION objects.
        // That is done at startup.
        if ( co.getExpiration().isExpired() ) {
            removeObject(co.getLSID());
            return null;
        }
        return co;
    }

    // Now look in the database:
    try
      {
    	selectByLSIDStatement.setString(1,lsid.toString());
        ResultSet rs = selectByLSIDStatement.executeQuery();
        
        if ( rs.next() ) {
        	// found it in the database.
        	String name = rs.getString("name");
        	String file = rs.getString("file");
            CacheExpiration expiry = CacheExpiration.newInstance( rs.getString("expiration"));
        	System.out.println("ser file: " + new File(objectPath, file).getAbsolutePath());
        	ObjectInputStream ois = new ObjectInputStream(new FileInputStream(new File(objectPath, file)));
        	//deserialize the CacheObjectInterface
        	co = (CacheObjectInterface)ois.readObject();
            co.setName( name );
            co.setLSID( lsid );
            co.setExpiration( expiry );
        	//add the CacheObjectInterface to the hashtable for easier access next time
            if ( co.getExpiration().isExpired() ) {
                removeObject(co.getLSID());
                return null;
            }
        	objectHash.put(lsid.toString(), co);
        	return co;
        }
      }
      catch(SQLException sqle)
      { 
    	  sqle.printStackTrace();
    	  throw new CacheException( "SQL exception when getting object", sqle );
      }
      catch(Exception e)
      {
    	  throw new CacheException( "Exception occurred while deserializing object", e);
      }
      
      //if there is no result, we need to use the lsid resolver to get the object
      // TODO - this code is broken.  in particular the resulting cache object from the
      // resolver is not added into the cache hash.
      try
      {
    	  LSIDResolver resolver = LSIDResolver.instanceOf();
          InputStream is = resolver.resolveID(lsid);
          //write it to a temp file
          String tmpFileName = System.currentTimeMillis() + ".tmp";
          File tmpFile = new File(tmpPath, tmpFileName);
          FileOutputStream tempFileStream = new FileOutputStream(tmpFile);
          CacheUtil.writeInputStreamToOutputStream(is, tempFileStream);
          //try to open it as a karCO
          try
          {
            KARCacheObject karCO = new KARCacheObject(tmpFile);
          }
          catch(CacheException ce)
          {  //if it isn't, create a RawDataCacheObject
            RawDataCacheObject rdco = new RawDataCacheObject(null, lsid);
            rdco.setData(new FileInputStream(tmpFile));
          }
          //delete the temp file
          //lsidTree.addLSID(lsid);
          tmpFile.delete();
       }
       catch(org.kepler.objectmanager.IDNotFoundException idnfe)
       {
    	   System.out.println("Could not resolve lsid " + lsid.toString() );
    	   return null;
       }
       catch(FileNotFoundException fnfe)
       {
          throw new CacheException("Error creating temp file in the cache", fnfe);
       }
       catch(IOException ioe)
       {
          throw new CacheException("Error reading or writing data to the cache", ioe);
       }
    return co;
  }
  
  /**
   * return an iterator of all of the CacheObjectInterfaces in the cache
   */
  public Iterator getCacheObjectIterator()
  throws CacheException
  {
	  try
	  {
		  Vector items = new Vector();
		  String sql = "select name, lsid, file, expiration from " + CACHETABLENAME;
		  ResultSet rs = conn.executeSQLQuery(sql);
		  if ( rs.first() ) {
			  
			  do
			  {
				  String name = rs.getString("name");
				  String file = rs.getString("file");
				  String lsidString = rs.getString("lsid");
                  CacheExpiration expiry = CacheExpiration.newInstance( rs.getString("expiration"));
				  CacheObjectInterface co = (CacheObjectInterface)objectHash.get(lsidString);
				  if(co == null)
				  { //deserialize the object and put it in the hash
					  File f = new File(objectPath, file);
					  FileInputStream fis = new FileInputStream(f);
					  ObjectInputStream ois = new ObjectInputStream(fis);
					  co = (CacheObjectInterface)ois.readObject();
                      co.setName( name );
                      co.setExpiration( expiry );
                      co.setLSID( new KeplerLSID( lsidString ) );
					  objectHash.put(lsidString, co);
				  }
				  items.add(co);
			  }
			  while(rs.next());
		  }
		  return items.iterator();
	  }
	  catch(Exception e)
	  {
		  e.printStackTrace();
		  throw new CacheException("Error creating CacheObjectInterface iterator. "
				  + "Try removing the ~/.kepler directory?: " + 
				  e.getMessage());
	  }
  }
  
  /**
   * return true of the given lsid has an associated object in the cache
   * @param lsid
   */
  public boolean isContained(KeplerLSID lsid)
    throws CacheException
  {
    try
    {
      String sql = "select name from " + CACHETABLENAME + 
          " where lsid='" + lsid.toString() + "'";
      ResultSet rs = conn.executeSQLQuery(sql);
      if(rs.next())
      {
        return true;
      }
      else
      {
        return false;
      }
    }
    catch(Exception e)
    {
      throw new CacheException("Error determining contents of cache: " + 
        e.getMessage());
    }
  }
  
  /**
   * clear the cache of all contents
   */
  public void clearCache()
    throws SQLException, CacheException
  {
	  // Remove the data files.
    CacheUtil.cleanUpDir(new File(objectPath));
      // Clear our objectHash.
    objectHash.clear();
      // Need to clear the lsidTree too.
    lsidTree.clearTree();
    
    String sql = "delete from " + CACHETABLENAME;
    conn.executeSQLCommand(sql);
    conn.commit();
    System.out.println("Committed the delete from cachetable");
        
  }
  
  /**
   * add a CacheListener to listen for cache events
   */
  public void addCacheListener(CacheListener listener)
  {
    listeners.add(listener);
  }
  
  /**
   * notifies any listeners as to an action taking place.
   */
  private void notifyListeners(CacheObjectInterface co, String op)
  {
    CacheEvent ce = new CacheEvent(co);
    
    if(op.equals("add"))
    {
      for(int i=0; i<listeners.size(); i++)
      {
        CacheListener cl = (CacheListener)listeners.elementAt(i);
        cl.objectAdded(ce);
      }
      co.objectAdded();
    }
    else if(op.equals("remove"))
    {
      for(int i=0; i<listeners.size(); i++)
      {
        CacheListener cl = (CacheListener)listeners.elementAt(i);
        cl.objectRemoved(ce);
      }
      co.objectRemoved();
    }
    else if(op.equals("purge"))
    {
      for(int i=0; i<listeners.size(); i++)
      {
        CacheListener cl = (CacheListener)listeners.elementAt(i);
        cl.objectPurged(ce);
      }
      co.objectPurged();
    }
  }
  
  private void serializeObjectInFile( CacheObjectInterface co, String filename ) throws CacheException {
		try {
			File outputFile = new File(CacheManager.objectPath, filename);
			FileOutputStream fos = new FileOutputStream(outputFile);
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			oos.writeObject(co); //serialize the CacheObjectInterface itself
			oos.flush();
			oos.close();
		}
		catch(IOException e) {
			throw new CacheException("Unable to serialize " + co.getName(), e);
		}
  
  }
  
}
