/**
 *    '$RCSfile: DataTypeResolver.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/23 15:33:55 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.objectmanager.data;

import org.apache.xpath.CachedXPathAPI;
import org.apache.xpath.XPathAPI;
import org.ecoinformatics.util.Config;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class takes in information from the metadata parsers and tries to
 * intelligently guess as to the data type of an attribute based on
 * provided metadata.  since different metadata standards provide differing
 * levels of complexity in their attribute level descriptions, some guesses
 * may be better than others, but this class, should at least be able
 * to distinguish between, whole (integer), real (float) and text (string)
 * types.
 */
public class DataTypeResolver
{
  private static DataTypeResolver resolver = null; //singleton variable
  private static String DEFAULT_TYPE = "STRING";

  private DataType[] dataTypes;

  /**
   * constructor
   */
  public DataTypeResolver()
  {
    try
    {
      NodeList nl = Config.getInstance().getNodeListFromPath("//dataType");
      dataTypes = new DataType[nl.getLength()];
      for(int i=0; i<nl.getLength(); i++)
      {
          CachedXPathAPI xpathapi = new CachedXPathAPI();
        Node n = nl.item(i);
        Node nameNode = xpathapi.selectSingleNode(n, "./name");
        Node numberTypeNode = xpathapi.selectSingleNode(n, "./numberType");
        if(nameNode == null || numberTypeNode == null)
        {
          throw new RuntimeException("Expecting a name and numberType node " +
            "child of dataType in " +
            "config.xml.");
        }

        String name = nameNode.getFirstChild().getNodeValue();
        String numberType = numberTypeNode.getFirstChild().getNodeValue();

        NodeList aliasNL = xpathapi.selectNodeList(n, "./alias");
        String[] aliases = new String[aliasNL.getLength()];
        for(int j=0; j<aliasNL.getLength(); j++)
        {
          Node aliasNode = aliasNL.item(j);
          aliases[j] = aliasNode.getFirstChild().getNodeValue();
        }

        Node numericTypeNode = xpathapi.selectSingleNode(n, "./numericType");
        String maxval = null;
        String minval = null;
        String textencoding = null;
        if(numericTypeNode != null)
        {
          Node minValNode = xpathapi.selectSingleNode(numericTypeNode, "./minValue");
          Node maxValNode = xpathapi.selectSingleNode(numericTypeNode, "./maxValue");
          if(minValNode == null || maxValNode == null)
          {
            throw new RuntimeException("Expecting a minValue and " +
              "maxValue children of numericType in config.xml.");
          }

          minval = minValNode.getFirstChild().getNodeValue();
          maxval = maxValNode.getFirstChild().getNodeValue();
        }
        else
        {
          Node textTypeNode = xpathapi.selectSingleNode(n, "./textType");
          if(textTypeNode == null)
          {
            throw new RuntimeException("Each dataType defined in the " +
              "config.xml file must have either a textType or a numericType " +
              "defined to be valid.");
          }

          Node textEncodingNode = xpathapi.selectSingleNode(textTypeNode, "./encoding");
          if(textEncodingNode == null)
          {
            throw new RuntimeException("The textType node must have an " +
              "encoding node as a child in config.xml.");
          }
          textencoding = textEncodingNode.getFirstChild().getNodeValue();
        }

       

        double min = -1.0;
        double max = -1.0;
        if(minval != null && maxval != null)
        {
          min = Double.parseDouble(minval);
          max = Double.parseDouble(maxval);
        }
        DataType dt = new DataType(name, min, max, numberType, aliases);
        dataTypes[i] = dt;

      }
    }
    catch(javax.xml.transform.TransformerException te)
    {
      throw new RuntimeException("Error executing xpath expression in " +
        "DataTypeResolver.");
    }
    catch(Exception e)
    {
      throw new RuntimeException("Exception in DataTypeResolver.");
    }
  }

  /**
   * returns a singleton instance of the DataTypeResolver
   */
  public static DataTypeResolver instanceOf()
  {
    if(resolver == null)
    {
      resolver = new DataTypeResolver();
    }
    return resolver;
  }

  /**
   * returns an array of the data types defined in the config.xml file
   */
  public DataType[] getDataTypes()
  {
    return dataTypes;
  }

  /**
   * this method will look up the numberType parameter which come from metadata
   * to find the type defined in configure.xml
   * @param min the minimum value for the attribute
   * @param max the maximum value for the attribute
   * @param typeName the name of the numberType declared in the metadata
   * @throws UnresolvableTypeException
   * @return one of the DataType constants (INT, LONG, FLOAT, DOUBLE, STRING)
   * declared in DataType.java
   */
  public DataType resolveDataType(String numberType, Double min, Double max)
    throws UnresolvableTypeException
  { 
     DataType type = numberTypeLookup(numberType, min, max);
     return type;
  }

  /**
   * resolves the type based solely on the name.  This does an alias lookup
   * on the types in config.xml.
   * @param name the name of the type to resolve.
   */
  public DataType resolveTypeByAlias(String name, Double min, Double max)
    throws UnresolvableTypeException
  {
    DataType type = aliasLookup(name, min, max);
    return type;
  }

  /*
   * finad a DataType based on a numberType and range
   */
  private DataType numberTypeLookup(String type, Double min, Double max)
                                   throws UnresolvableTypeException
  {
    if (type == null)
    {
        throw new UnresolvableTypeException("The given number type is null");
    }
    if (min == null || max == null)
    {
        for (int i = 0; i < dataTypes.length; i++) {
            String  numberType = dataTypes[i].getNumberType();
            if ( numberType != null && type.trim().equals(numberType))
            {
               return dataTypes[i];
            }
            
        }
    }
    else
    {
        double minNum = min.doubleValue();
        double maxNum = max.doubleValue();
        for (int i = 0; i < dataTypes.length; i++) {
            String  numberType = dataTypes[i].getNumberType();
            if ( numberType != null && type.trim().equals(numberType) && 
                 rangeLookup(minNum, maxNum, dataTypes[i]))
            {
               return dataTypes[i];
            }
            
        }
        
    }
    throw new UnresolvableTypeException("The number type '" + type.trim() +
      "' cannot be " +
      "resolved with any type in the config.xml file.  Please add a numberType " +
      "for this type to one of the DataType elements in config.xml.");
  }

  /**
   * find a DataType based on an alias name and range
   */
  private DataType aliasLookup(String typeName, Double min, Double max)
    throws UnresolvableTypeException
  {
     if (typeName == null)
    {
        throw new UnresolvableTypeException("The given alias type is null");
    }
    if (min == null || max == null)
    {
        for (int i = 0; i < dataTypes.length; i++) {
            String aliases[] = dataTypes[i].getNames();
            String name = dataTypes[i].getName();
            for (int j = 0; j < aliases.length; j++) {
                if (typeName.trim().equals(aliases[j].trim()) ||
                    typeName.trim().equals(name)) {

                    return dataTypes[i];
                }
            }
        }
    }
    else
    {
        double minNum = min.doubleValue();
        double maxNum = max.doubleValue();
        for (int i = 0; i < dataTypes.length; i++) {
            String aliases[] = dataTypes[i].getNames();
            String name = dataTypes[i].getName();
            for (int j = 0; j < aliases.length; j++) {
                if ((typeName.trim().equals(aliases[j].trim()) || typeName.trim().equals(name)) 
                    && rangeLookup(minNum, maxNum, dataTypes[i])) {

                    return dataTypes[i];
                }
            }
        }
    }
    throw new UnresolvableTypeException("The type '" + typeName.trim() +
      "' cannot be " +
      "resolved with any type in the config.xml file.  Please add an alias " +
      "for this type to one of the DataType elements in config.xml.");
  }
  
   /**
   * If the given range fits the type.
   */
  private boolean rangeLookup(double min, double max, DataType type)
  {
    boolean fit = false;
    if(min > max)
    {
      return fit;
    }

    double dtmin = type.getMin();
    double dtmax = type.getMax();
    if(dtmin < min && dtmax > max)
    {
      fit = true;
    }
    return fit;

  }
}
