/**
 *  '$RCSfile: LibraryIndex.java,v $' 
 *  '$Author: berkley $' 
 *  '$Date: 2006/03/06 20:19:33 $' 
 *  '$Revision: 1.20 $' 
 *  For Details: http://www.kepler-project.org
 *  Copyright (c) 2004 The Regents of the University of California. All rights
 *  reserved. Permission is hereby granted, without written agreement and
 *  without license or royalty fees, to use, copy, modify, and distributedte
 *  this software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies of
 *  this software. IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO
 *  ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
 *  DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 *  IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
 *  DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN
 *  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.objectmanager.library;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.xpath.XPathAPI;
import org.ecoinformatics.util.Config;
import org.kepler.objectmanager.ActorMetadata;
import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.objectmanager.cache.CacheManager;
import org.kepler.objectmanager.cache.ActorCacheObject;
import org.kepler.objectmanager.cache.CacheObject;
import org.kepler.gui.AnnotatedPTree;
import org.kepler.objectmanager.cache.CacheException;
import org.kepler.sms.NamedOntClass;
import org.kepler.sms.NamedOntModel;
import org.kepler.sms.OntologyCatalog;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.moml.EntityLibrary;
import ptolemy.vergil.tree.EntityTreeModel;
import ptolemy.vergil.tree.VisibleTreeModel;
import ptolemy.kernel.util.Workspace;

/**
 *  builds and maintains a library index that looks like this:
 *  <library-index> <ontology name="" id=""> <concept name="" id=""> <concept
 *  name="" id=""> ... </concept> <component name="" id=""/> <component name=""
 *  id=""/> </concept> <concept> </concept> </ontology> <ontology name="" id="">
 *  </ontology> </library-index> The purpose of this class is to have a fast
 *  method of serializing and caching the contents of the Actor Library in
 *  kepler.
 *
 *@author     Chad Berkley
 *@created    December 2, 2005
 */
public class LibraryIndex
{
  private static LibraryIndex libIndex;
  private static File saveFile = new File( Config.getUserDirPath() + "libraryIndex");
  private LibraryIndexRootItem _rootItem;
  private CacheManager cacheMan;
  private CompositeEntity actorLibrary = null;
  private Workspace actorLibraryWorkspace = null;
  private Vector treeComponents = new Vector();

  /**
   *  create a new LibraryIndex. Normally, this is accessed only through the
   *  getInstance() singleton call.
   *
   *@exception  IllegalActionException  Description of the Exception
   */
  protected LibraryIndex()
    throws IllegalActionException
  {
    try
    {
      cacheMan = CacheManager.getInstance();
      if (!saveFile.exists())
      {
        _rootItem = new LibraryIndexRootItem();
        _buildNewIndex();
        // serialize()
      }
      else
      {
        read(new FileInputStream(saveFile));
      }
    }
    catch (IOException ioe)
    {
      throw new IllegalActionException("There was an error reading the file " +
          saveFile.getAbsolutePath() + ": " + ioe.getMessage());
    }
    catch(CacheException ce)
    {
      throw new IllegalActionException("Error getting an instance of the cache.");
    }
  }

  /**
   *  create a new library index from a persistent index
   *
   *@param  is                          Description of the Parameter
   *@exception  IllegalActionException  Description of the Exception
   */
  protected void read(InputStream is)
    throws IllegalActionException
  {
    _rootItem = new LibraryIndexRootItem();
    try
    {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = factory.newDocumentBuilder();
      Document doc = builder.parse(is);
      NodeList ontologies = XPathAPI.selectNodeList(doc, "LibraryIndex/ontology");
      for (int i = 0; i < ontologies.getLength(); i++)
      {
        //get the ontologies
        Node ontology = ontologies.item(i);
        String ontoName = ontology.getAttributes().getNamedItem("name").getNodeValue();
        String ontoLSID = ontology.getAttributes().getNamedItem("lsid").getNodeValue();
        LibraryIndexOntologyItem ontoComp =
            new LibraryIndexOntologyItem(ontoName, new KeplerLSID(ontoLSID));
        _rootItem.setChild(ontoComp);

        NodeList concepts = XPathAPI.selectNodeList(ontology, "concept");
        for (int j = 0; j < concepts.getLength(); j++)
        {
          //get the concepts
          Node concept = concepts.item(j);
          String conceptName = concept.getAttributes().getNamedItem("name").getNodeValue();
          String conceptLSID = concept.getAttributes().getNamedItem("lsid").getNodeValue();
          LibraryIndexConceptItem conceptComp =
              new LibraryIndexConceptItem(conceptName, new KeplerLSID(conceptLSID));
          ontoComp.setChild(conceptComp);

          NodeList components = XPathAPI.selectNodeList(concept, "component");
          for (int k = 0; k < components.getLength(); k++)
          {
            //get the components
            Node component = components.item(k);
            String compName = component.getAttributes().getNamedItem("name").getNodeValue();
            String compLSID = component.getAttributes().getNamedItem("lsid").getNodeValue();
            LibraryIndexComponentItem compComp =
                new LibraryIndexComponentItem(compName, new KeplerLSID(compLSID));
            conceptComp.setChild(compComp);
          }
        }
      }
    }
    catch (Exception e)
    {
      throw new IllegalActionException("Error deserializing the index: " +
          e.getMessage());
    }
  }

  /**
   *  persists this index
   *
   *@param  os                          Description of the Parameter
   *@exception  IllegalActionException  Description of the Exception
   */
  protected void write(OutputStream os)
    throws IllegalActionException
  {
    try
    {
      String outputString = _rootItem.toString();
      byte[] b = outputString.getBytes();
      os.write(b, 0, b.length);
      os.flush();
    }
    catch (Exception e)
    {
      throw new IllegalActionException("Could not serialize the tree: " +
          e.getMessage());
    }
  }

  /**
   *  get a singleton instance of this class
   *
   *@return                             The instance value
   *@exception  IllegalActionException  Description of the Exception
   */
  public static LibraryIndex getInstance()
    throws IllegalActionException
  {
    if (libIndex == null)
    {
      libIndex = new LibraryIndex();
    }
    return libIndex;
  }
  
  /**
   *  returns a new instance of LibraryIndex. This method is for testing the
   *  serialization/deserialization and should not be used otherwise.
   *
   *@return                             The newInstance value
   *@exception  IllegalActionException  Description of the Exception
   */
  public static LibraryIndex getNewInstance()
    throws IllegalActionException
  {
    return new LibraryIndex();
  }

  /**
   * add a tree component so that it can be updated when the tree model changes
   */
  public void addLibraryComponent(AnnotatedPTree ptree)
  {
    treeComponents.addElement(ptree);
  }

  /**
   *  serialize this index tree to xml
   *
   *@exception  IllegalActionException  Description of the Exception
   */
  public void serialize()
    throws IllegalActionException
  {
    try
    {
      FileOutputStream fos = new FileOutputStream(saveFile);
      write(fos);
      fos.close();
    }
    catch (Exception e)
    {
      throw new IllegalActionException("Error serializing the save file: " +
          e.getMessage());
    }
  }

  /**
   *  searches from the root of the tree to see if the component is indexed.
   *
   *@param  itemId       Description of the Parameter
   *@return              The indexed value
   */
  public boolean isIndexed(KeplerLSID itemId)
  {
    return isIndexed(itemId, _rootItem);
  }

  /**
   *  searches the tree starting to see if the componentId is already in the
   *  tree
   *
   *@param  itemId       Description of the Parameter
   *@param  item         Description of the Parameter
   *@return              The indexed value
   */
  public boolean isIndexed(KeplerLSID itemId, LibraryIndexItem item)
  {
    if (findItem(itemId, item) != null)
    {
      return true;
    }
    return false;
  }

  /**
   *  search for a component starting at the root
   *
   *@param  lsid  the id to search for
   *@return       Description of the Return Value
   */
  public LibraryIndexItem findItem(KeplerLSID lsid)
  {
    return findItem(lsid, _rootItem);
  }

  /**
   *  return the item with the given lsid. returns null if the item isn't found.
   *
   *@param  lsid  the lsid to search for
   *@param  item  the item to start searching in
   *@return       Description of the Return Value
   */
  public LibraryIndexItem findItem(KeplerLSID lsid, LibraryIndexItem item)
  {
    //search the tree and see if the lsid exists
    Vector children = item.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      LibraryIndexItem child = (LibraryIndexItem) children.elementAt(i);
      if (child.getLSID().equals(lsid))
      {
        return child;
      }
      else
      {
        LibraryIndexItem foundItem = findItem(lsid, child);
        if (foundItem != null)
        {
          return foundItem;
        }
      }
    }
    return null;
  }
  
  /**
   * search for name in the root of this tree. return null if not found
   */
  public LibraryIndexItem findItem(String name)
  {
    return findItem(name, _rootItem);
  }
  
  /**
   * search item for name.  return null if not found
   */
  public LibraryIndexItem findItem(String name, LibraryIndexItem item)
  {
    //search the tree and see if the lsid exists
    Vector children = item.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      LibraryIndexItem child = (LibraryIndexItem) children.elementAt(i);
      if (child.getName().equals(name))
      {
        return child;
      }
      else if(child instanceof LibraryIndexConceptItem &&
              ((LibraryIndexConceptItem)child).getConceptId().equals(name))
      { //this checks for the concept name
        return child;
      }
      else
      {
        LibraryIndexItem foundItem = findItem(name, child);
        if (foundItem != null)
        {
          return foundItem;
        }
      }
    }
    return null;
  }

  /**
   *  return the component at the root of the tree
   *
   *@return    The rootComponent value
   */
  public LibraryIndexRootItem getRootComponent()
  {
    return _rootItem;
  }
  
  /**
   *  this method adds itemToAdd as a child to subtree
   *
   *@param  itemToAdd  Description of the Parameter
   *@param  subtree    Description of the Parameter
   */
  public void add(LibraryIndexItem itemToAdd, LibraryIndexItem subtree)
  {
    subtree.setChild(itemToAdd);
  }

  /**  removes all nodes from the tree except for the root node  */
  public void clear()
  {
    Vector children = _rootItem.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      children.remove(i);
    }
  }

  /**
   *  removes an item completely from the index. If you want to remove the item
   *  from only one sub tree, you need to manipulate the tree itself starting
   *  with getRootComponent(). Note that if you remove an item with children,
   *  all of those children will be removed too.
   *
   *@param  itemId       Description of the Parameter
   */
  public void removeAll(KeplerLSID itemId)
  {
    Vector children = _rootItem.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      LibraryIndexItem child = (LibraryIndexItem) children.elementAt(i);
      if (child.getLSID().toString().equals(itemId.toString()))
      {
        children.remove(i);
      }
      else
      {
        removeComponent(itemId, child);
      }
    }
  }

  /**
   *  remove the item with the given lsid from the subtree. note if you remove
   *  an item with children, all children will also be removed
   *
   *@param  lsid     the lsid of the item to remove
   *@param  subtree  the subtree to search
   */
  public void removeComponent(KeplerLSID lsid, LibraryIndexItem subtree)
  {
    Vector children = subtree.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      LibraryIndexItem child = (LibraryIndexItem) children.elementAt(i);
      if (child.getLSID().toString().equals(lsid.toString()))
      {
        children.remove(i);
      }
      else
      {
        removeComponent(lsid, child);
      }
    }
  }

  /**
   *  returns the items in the index // Iterator<LSID>
   *
   *@return    Description of the Return Value
   */
  public Iterator items()
  {
    return getAllNodesOfType(
        new LibraryIndexComponentItem(null, null).getType(),
        _rootItem).iterator();
  }

  /**
   *  returns the ontologies used in the index // Iterator<LSID>
   *
   *@return    Description of the Return Value
   */
  public Iterator ontologies()
  {
    return getAllNodesOfType(
        new LibraryIndexOntologyItem(null, null).getType(),
        _rootItem).iterator();
  }

  /**
   *  returns the concepts from an ontology used in the index //Iterator<LSID>
   *
   *@param  ontoId  Description of the Parameter
   *@return         Description of the Return Value
   */
  public Iterator concepts(KeplerLSID ontoId)
  {
    return getAllNodesOfType(
        new LibraryIndexConceptItem(null, null).getType(),
        _rootItem).iterator();
  }
  
  /**
   * return the concepts as a vector
   */
  public Vector conceptsVector(KeplerLSID ontoId)
  {
    return getAllNodesOfType(
        new LibraryIndexConceptItem(null, null).getType(),
        _rootItem);
  }

  /**
   *  build the library for a whole ontology
   *
   *@param  item                        Description of the Parameter
   *@return                             Description of the Return Value
   *@exception  IllegalActionException  Description of the Exception
   */
  public EntityTreeModel createLibrary(LibraryIndexOntologyItem item)
    throws IllegalActionException
  {
    EntityLibrary root = new EntityLibrary();
    buildTreeModel(root, item);
    return new VisibleTreeModel(root);
  }

  /**
   *  create an EntityTreeModel for the given concept
   *
   *@param  item                        Description of the Parameter
   *@return                             Description of the Return Value
   *@exception  IllegalActionException  Description of the Exception
   */
  public EntityTreeModel createLibrary(LibraryIndexConceptItem item)
    throws IllegalActionException
  {
    EntityLibrary root = new EntityLibrary();
    buildTreeModel(root, item);
    return new VisibleTreeModel(root);
  }

  /**
   *  returns a tree model library based on the contents of the index, for all
   *  ontologies represented in the index
   *
   *@return                             Description of the Return Value
   *@exception  IllegalActionException  Description of the Exception
   */
  public EntityTreeModel createLibrary()
    throws IllegalActionException
  {
    EntityLibrary root = new EntityLibrary();
    Iterator ontos = ontologies();
    while (ontos.hasNext())
    {
      LibraryIndexOntologyItem item = (LibraryIndexOntologyItem)
          ontos.next();
      buildTreeModel(root, item);
    }
    return new VisibleTreeModel(root);
  }
  
  /**
   * get the CompositeEntity that makes up this library.
   */
  public CompositeEntity getLibrary(Workspace ws)
    throws IllegalActionException
  {
    EntityLibrary root = null;
    try
    {
      root = new EntityLibrary(ws);
      root.setName("kepler actor library");
    }
    catch(ptolemy.kernel.util.NameDuplicationException nde)
    {
      //do nothing, just leave the name blank
      System.out.println("the name 'kepler actor library' is already taken");
    }
    Iterator ontos = ontologies();
    while (ontos.hasNext())
    {
      LibraryIndexOntologyItem item = (LibraryIndexOntologyItem)
          ontos.next();
      buildTreeModel(root, item);
    }
    return root;
  }
  
  /**
   * refresh and redraw the tree
   */
  public void refresh()
    throws IllegalActionException
  {
    Iterator treeItt = treeComponents.iterator();
    while(treeItt.hasNext())
    {
      AnnotatedPTree ptree = (AnnotatedPTree)treeItt.next();
      ptree.setModel(createLibrary());
    }
  }
  
  /**
   * this method allows libraryIndex to store the actorLibrary Entity and its
   * workspace.
   */
  public void setActorLibrary(CompositeEntity actorLib, Workspace workspace)
  {
    this.actorLibrary = actorLib;
    this.actorLibraryWorkspace = workspace;
  }


  /**
   *  return the tree as a string
   *
   *@return    Description of the Return Value
   */
  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    sb.append("<?xml version=\"1.0\"?>\n");
    sb.append(_rootItem.toString());
    return sb.toString();
  }

  /**
   *  return all of the nodes in the tree that are instanceof c in comp
   *
   *@param  type  Description of the Parameter
   *@param  item  Description of the Parameter
   *@return       The allNodesOfType value
   */
  private Vector getAllNodesOfType(String type, LibraryIndexItem item)
  {
    Vector v = new Vector();
    Vector children = item.getChildren();
    for (int i = 0; i < children.size(); i++)
    {
      LibraryIndexItem lic = (LibraryIndexItem) children.elementAt(i);
      if (lic.getType().equals(type))
      {
        v.addElement(lic);
      }
      else
      {
        v.addAll(getAllNodesOfType(type, lic));
      }
    }
    return v;
  }

  /**
   *  recursive helper function to build the new tree model
   *
   *@param  parent                      Description of the Parameter
   *@param  item                        Description of the Parameter
   *@exception  IllegalActionException  Description of the Exception
   */
  private void buildTreeModel(EntityLibrary parent, LibraryIndexItem item)
    throws IllegalActionException
  {
    try
    {
      //create the folder
      EntityLibrary folder = new EntityLibrary(parent, item.getName());

      Vector children = item.getChildren();
      for (int i = 0; i < children.size(); i++)
      {
        //create a folder for each concept and get the actorMetadata
        //object for each item to create the EntityLibrary
        LibraryIndexItem childItem =
            (LibraryIndexItem) children.elementAt(i);
        if (childItem instanceof LibraryIndexConceptItem)
        {
          //recurse to create hierarchy structure
          buildTreeModel(folder, childItem);
        }
        else if (childItem instanceof LibraryIndexComponentItem)
        {
          //actor (leaf)
          ActorMetadata am = ((ActorCacheObject) cacheMan.getObject(
              childItem.getLSID())).getMetadata();
          NamedObj obj = (NamedObj) am.getActorAsNamedObj(folder);

          if (obj instanceof ComponentEntity)
          {
            ((ComponentEntity) obj).setContainer(folder);
          }
          else if (obj instanceof Attribute)
          {
            ((Attribute) obj).setContainer(folder);
          }
        }
        else
        {
          throw new IllegalActionException("Error reading tree.  " +
              "Invalid tree structure.  Was expecting a concept or an item.");
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      throw new IllegalActionException("Error building tree model: " +
          e.getMessage());
    }
  }


  /** */
  protected void _buildNewIndex()
    throws IllegalActionException
  {
    try
    {
      // read in the default ontology file for now
      OntologyCatalog cat = OntologyCatalog.instance();
      Iterator iter = cat.getLibraryNamedOntModels();
      if (!iter.hasNext())
      {
        System.out.println("ERROR: in LibraryIndex at _buildNewIndex. Cannot find library ontology");
      }
      // create a hashtable with concept id -> component id
      Hashtable table = _buildComponentTable();
      while (iter.hasNext())
      {
        // get the ontology model (assuming first for the index)
        NamedOntModel m = (NamedOntModel) iter.next();
        // create the ontology item for it
        LibraryIndexOntologyItem item =
            new LibraryIndexOntologyItem(m.getName(), new KeplerLSID(m.getNameSpace()));
        // add it to the root
        _rootItem.setChild(item);
        // build the tree
        _buildConceptHierarchy(table, item, m);
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  /**
   *  Creates a hashtable consisting of Semantic types (strings) as keys and a
   *  vector of actor metadata objects as values.
   *
   *@return    Description of the Return Value
   */
  protected Hashtable _buildComponentTable()
    throws IllegalActionException
  {
    Hashtable table = new Hashtable();
    Iterator iter;
    try
    {
      iter = cacheMan.getCacheObjectIterator();
    }
    catch(CacheException ce)
    {
      ce.printStackTrace();
      throw new IllegalActionException("Could not get the cache iterator: " + 
        ce.getMessage());
    }
    while (iter.hasNext())
    {
      CacheObject co = (CacheObject) iter.next();
      if (co instanceof ActorCacheObject)
      {
        ActorMetadata m = ((ActorCacheObject)co).getMetadata();
        Iterator types = m.getSemanticTypes().iterator();
        while (types.hasNext())
        {
          Object o = types.next();
          //System.out.println("ACTOR: " + m.getId() + " HAS TYPE: " + o);
          if (table.containsKey(o))
          {
            Vector v = (Vector) table.get(o);
            if (!v.contains(m))
            {
              v.add(m);
            }
          }
          else
          {
            Vector v = new Vector();
            v.add(m);
            table.put(o, v);
          }
        }
      }
    }
    return table;
  }

  /**
   *@param  table  Description of the Parameter
   *@param  item   Description of the Parameter
   *@param  m      Description of the Parameter
   */
  protected void _buildConceptHierarchy(Hashtable table, LibraryIndexItem item, NamedOntModel m)
  {
    Iterator iter = m.getRootClasses(true);
    while (iter.hasNext())
    {
      NamedOntClass c = (NamedOntClass) iter.next();
      _buildConceptHierarchy(table, item, c);
    }
  }

  /**
   *  Description of the Method
   *
   *@param  table       Description of the Parameter
   *@param  parentItem  Description of the Parameter
   *@param  ontClass    Description of the Parameter
   */
  protected void _buildConceptHierarchy(Hashtable table, LibraryIndexItem parentItem, NamedOntClass ontClass)
  {
    try
    {
      String fullClassName = ontClass.getNameSpace() + ontClass.getLocalName();
      LibraryIndexConceptItem subItem =
          new LibraryIndexConceptItem(ontClass.getName(), new KeplerLSID(fullClassName));
      subItem.setConceptId(ontClass.getLocalName());
      parentItem.setChild(subItem);
      // add matching components
      //System.out.println("SEARCHING FOR MATCHING TYPE: " + fullClassName);
      if (table.containsKey(fullClassName))
      {
        Iterator objs = ((Vector) table.get(fullClassName)).iterator();
        while (objs.hasNext())
        {
          ActorMetadata actor = (ActorMetadata) objs.next();
          LibraryIndexComponentItem actorItem =
              new LibraryIndexComponentItem(actor.getName(), new KeplerLSID(actor.getId()));
          subItem.setChild(actorItem);
          //System.out.println("FOUND ACTOR " + actor.getId() + " for type " + fullClassName);
        }
      }
      // recurse to child nodes
      Iterator subClasses = ontClass.getNamedSubClasses(true);
      while (subClasses.hasNext())
      {
        _buildConceptHierarchy(table, subItem, (NamedOntClass) subClasses.next());
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }

  }


  // FOR TESTING
  /**
   *  The main program for the LibraryIndex class
   *
   *@param  args  The command line arguments
   */
  public static void main(String[] args)
  {
    try
    {
      LibraryIndex idx = LibraryIndex.getNewInstance();

      System.out.println(idx);
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }

  }

}

