/**
 *  '$RCSfile: LibraryIndexItem.java,v $'
 *  '$Author: berkley $'
 *  '$Date: 2006/01/25 19:38:19 $'
 *  '$Revision: 1.4 $'
 *
 *  For Details:
 *  http://www.kepler-project.org
 *
 *  Copyright (c) 2004 The Regents of the
 *  University of California. All rights reserved. Permission is hereby granted,
 *  without written agreement and without license or royalty fees, to use, copy,
 *  modify, and distribute this software and its documentation for any purpose,
 *  provided that the above copyright notice and the following two paragraphs
 *  appear in all copies of this software. IN NO EVENT SHALL THE UNIVERSITY OF
 *  CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 *  OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 *  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY
 *  DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 *  SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.objectmanager.library;

import java.util.Vector;

import org.kepler.objectmanager.lsid.KeplerLSID;

/**
 * A LibraryIndexComponent is the base class for any type of component that
 * can fit in the following tree under the LibraryIndex root.
   <LibraryIndex>
    <ontology name="" id="">
      <concept name="" id="">
        <concept name="" id="">
          ...
        </concept>

        <component name="" id=""/>
        <component name="" id=""/>
      </concept>

      <concept>
      </concept>

    </ontology>

    <ontology name="" id="">

    </ontology>

  </LibraryIndex>
 */
public class LibraryIndexItem 
{
  protected String name;
  protected KeplerLSID lsid;
  protected String type; //extensions should be either ontology, concept or component
  protected Vector children;
  protected Vector parents;
  
  /**
   * constructor.  this allows the creating of a new component.  the type, name
   * and lsids should be set in any extension of this abstract class.
   * @param name the name of the component
   * @param lsid the lsid of the component.
   */
  public LibraryIndexItem(String name, KeplerLSID lsid)
  {
    this.name = name;
    this.lsid = lsid;
    type = "";
    children = new Vector();
    parents = new Vector();
  }
  
  /**
   * return the name of this component
   */
  public String getName()
  {
    return name;
  }
  
  /**
   * return the type (tag name in the xml) of this component
   */
  public String getType()
  {
    return type;
  }
  
  /**
   * return the components lsid
   */
  public KeplerLSID getLSID()
  {
    return lsid;
  }
  
  /**
   * add a child to this component.  a component can have multiple children
   */
  public void setChild(LibraryIndexItem component)
  {
    children.addElement(component);
    component.setParent(this);
  }
  
  /**
   * set the parent of this component
   */
  public void setParent(LibraryIndexItem parent)
  {
    parents.addElement(parent);
    //children.addElement(this);
  }
  
  /**
   * returns a Vector of LibraryIndexItems of the children of this component
   */
  public Vector getChildren()
  {
    return children;
  }
  
  /**
   * return any parent nodes this node may have
   */
  public Vector getParents()
  {
    return parents;
  }
  
  /**
   * return a string representation of this component
   */
  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    String lsidstr = "";
    String namestr = "";
    if(lsid != null)
    {
      lsidstr = "lsid=\"" + lsid + "\"";
    }
    if(name != null)
    {
      namestr = "name=\"" + name + "\"";
    }
    
    sb.append("<" + (type + " " + namestr + " " + lsidstr).trim());
    if(children.size() == 0)
    {
      sb.append("/>\n");
    }
    else
    {
      sb.append(">\n");
      for(int i=0; i<children.size(); i++)
      {
        LibraryIndexItem lic = (LibraryIndexItem)children.elementAt(i);
        sb.append(lic.toString());
      }
      sb.append("</" + type + ">\n");
    }
    return sb.toString();
  }
} 
