/**
 *   '$RCSfile: KeplerLSID.java,v $'
 *
 *   '$Author: ruland $'
 *   '$Date: 2006/01/24 19:38:28 $'
 *   '$Revision: 1.12 $'
 *
 *  For Details: http://kepler-project.org
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager.lsid;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamField;
import java.io.Serializable;
import java.util.StringTokenizer;

import com.ibm.lsid.LSID;
import com.ibm.lsid.LSIDException;
import com.ibm.lsid.MalformedLSIDException;

/**
 * A kepler class to handle lsids.  The problem with the IBM lsid classes
 * is that to do localhost lsid resolution, you either have to have a
 * resolver running on port 80 or you have to catch the LSIDException and parse
 * the horrible html formatted exception.  There is no way to tell the
 * IBM LSID class not to resolve the id, so this class does not extend LSID().
 *
 *@created    June 20, 2005
 */

public class KeplerLSID implements Serializable
{
  private String lsidStr = null;
  private LSID lsid = null;
  private boolean isResolved = true;

  //the parts of the lsid
  private String authority;
  private String namespace;
  private String object;
  private String revision;

  /**
   * Construct an lsid.  If resolve is true, attempt to resolve the lsid,
   * if false, don't attempt to resolve it.
   */
  public KeplerLSID(String lsid)
    throws LSIDException
  {
      // Note - if any changes needs to be made to this constructor,
      // the changes should be made in initializeFromString so the
      // deserialization readObject() method can benefit from it as well.
      initializeFromString(lsid);
  }

  /**
   * construct an lsid from an existing lsid
   */
  public KeplerLSID(LSID lsid)
  {
      this.isResolved = true;
      this.lsid = lsid;
      parseLSID();
  }

  /**
   * construct an lsid from components
   * @param authority the authority of the new lsid
   * @param namespace the namespace of the new lsid
   * @param the object number of the new lsid
   * @param revision the revision of the new lsid
   */
  public KeplerLSID(String authority, String namespace, int object, int revision)
    throws LSIDException
  {
    String newLsid = "urn:lsid:" + authority + ":" + namespace + ":" +
      object + ":" + revision;

    if(authority.equals("kepler-project.org") || authority.equals("localhost"))
    {
      isResolved = false;
      this.lsidStr = newLsid;
    }
    else
    {
      this.lsid = new com.ibm.lsid.LSID(newLsid);
    }
    parseLSID();
  }

  /**
   * returns true if the lsid has been resolved, false if it has not
   * been resolved.
   */
  public boolean isResolved()
  {
    return isResolved;
  }

  /**
   * returns true if the lsid authority is 'localhost', meaning that the file
   * is a local file and not provided by a true lsid authority.
   */
  public boolean isLocalhost()
  {
    if(lsidStr != null)
    {
      return true;
    }
    return false;
  }

  /**
   * return the LSID represented by this object
   */
  public String getLsid()
  {
    if(lsidStr == null)
    {
      return lsid.getLsid();
    }
    else
    {
      return lsidStr;
    }
  }

  /**
   * returns the namespace component of this lsid
   */
  public String getNamespace()
  {
    return namespace;
  }

  /**
   * returns the authority component of this lsid
   */
  public String getAuthority()
  {
    return authority;
  }

  /**
   * return the revision componenent of this lsid
   */
  public String getRevision()
  {
    return revision;
  }

  /**
   * returns the object component of this lsid
   */
  public String getObject()
  {
    return object;
  }

  /**
   * returns an IBM LSID object.  If this is a localhost lsid, this method
   * will return null.
   */
  public LSID getProperLsid()
  {
    return lsid;
  }

  /**
   * this will create a valid filename out of the lsid (without an extension).
   * It will replace any invalid characters in the lsid with '.'
   */
  public String createFilename()
  {
    if(lsid == null)
    {
      return lsidStr.replace(':', '.');
    }
    else
    {
      return lsid.getLsid().replace(':', '.');
    }
  }

  /**
   * return a string representation of the lsid
   */
  public String toString()
  {
    if(lsidStr == null)
      return lsid.toString();
    else
      return lsidStr;
  }
  
  /**
   * return true if this lsid equals the passed in lsid
   */
  public boolean equals(KeplerLSID lsid)
  {
    String auth = lsid.getAuthority();
    String ns = lsid.getNamespace();
    String obj = lsid.getObject();
    String rev = lsid.getRevision();
    if(auth.trim().equals(authority.trim()) &&
       ns.trim().equals(namespace.trim()) &&
       obj.trim().equals(object.trim()) &&
       rev.trim().equals(revision.trim()))
    {
      return true; 
    }
    return false;
  }

  /**
   * parses the lsid into its parts
   * urn:lsid:<authority>:<namespace>:<object>:<revision>
   */
  private void parseLSID()
  {
    if(lsid != null)
    {
      authority = lsid.getAuthority().getAuthority();
      namespace = lsid.getNamespace();
      object = lsid.getObject();
      revision = lsid.getRevision();
    }
    else
    {
      StringTokenizer st = new StringTokenizer(lsidStr, ":");
      int count = -1;
      while(st.hasMoreTokens())
      {
        String token = st.nextToken();
        count++;
        if(count == 0 || count == 1)
        {
          continue; //we don't care about the 'urn' or the 'lsid' parts
        }
        else if(count == 2)
        {
          authority = token;
        }
        else if(count == 3)
        {
          namespace = token;
        }
        else if(count == 4)
        {
          object = token;
        }
        else if(count == 5)
        {
          revision = token;
        }
      }
    }
  }
  
  /**
   * Initialize the lsid instance from a string representation.  This method is used from
   * both the constructor KeplerLSID(String) and the Serializable readObject() pseudo-constructor.
   * Because this method is used by readObject, all non-default initialization needs to occur here.
   * @param lsid
   * @throws MalformedLSIDException
   */
  private void initializeFromString( String lsid ) throws MalformedLSIDException {
      if(lsid.toString().indexOf("kepler-project.org") != -1 || lsid.indexOf("localhost") != -1)
      {
          isResolved = false;
          lsidStr = lsid;
      }
      else
      {
          isResolved = true;
          this.lsid = new com.ibm.lsid.LSID(lsid);
      }

      parseLSID();
      
  }
  
  /**
   * Define custom metadata representation consisting of a single String named "lsidString"
   */
  private static final ObjectStreamField[] serialPersistentFields = {
      new ObjectStreamField("lsidString",String.class)
  };
  
  /**
   * Custom deserialization method.  The serial representation of the KeplerLSID object
   * is simply the toString() representation.  This string representation is passed into
   * the initializeFromString method.
   * @param ois
   * @throws IOException
   */
  private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException {
      ObjectInputStream.GetField fields = ois.readFields();
      String lsidString = (String)fields.get("lsidString",null);
      try {
          initializeFromString(lsidString);
      }
      catch( MalformedLSIDException e ) {
          throw new IOException("Malformed LSID in serialized representation");
      }
  }
  
  /**
   * Custom serialization method.  The serial representation of the KeplerLSID object
   * is simply the toString() representation.
   * @param oos
   * @throws IOException
   */
  private void writeObject(ObjectOutputStream oos ) throws IOException {
      ObjectOutputStream.PutField fields = oos.putFields();
      fields.put("lsidString",toString());
      oos.writeFields();
  }
}
