/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.awt.event.KeyEvent;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import org.w3c.rdf.model.Model;

import com.interdataworking.mm.alg.MapPair;
import com.interdataworking.mm.alg.SFMatcher;


/**
 * The main class of the schema mapping system SCIA. It is responsible for
 * setting up a source schema, a target schema, an optinal domain, coefficients
 * for different aspects of schema information, thresholds, and running modes
 * and strategies. It also provides schema matching methods and supporting
 * methods. It may have a graphical user interface
 * @see MainUI.
 *
 *@author Guilian Wang
 */
public class SCIA {
    public static String SCIA_WORK_DIR = System.getProperty("env.SCIA_WORK_DIR");
    public static String LIB_JAR_DIR = System.getProperty(
            "env.SCIA_OLD_JAR_DIR");

    // main user interface
    public static MainUI mainUI;

    // user defined functions, key = function name, content = import sentence
    static Hashtable udfImportTable = new Hashtable();

    // user defined functions, key = function name, content = definition
    static Hashtable udfDefTable = new Hashtable();
    public static boolean bothXMLSchemas = false;

    // match the opposite direction or not
    public static boolean bothDirection = false;
    public static boolean debug_on = false;

    // the number of top matches to output
    public static int NUMBER_TOP_MATCHES_TO_OUTPUT = 3;
    public static String domain;
    public static Domain appDomain = null;
    public static boolean interactive_mode = true;
    public static boolean context_check = false;
    public static boolean reuseMap_on;
    public static String sourceSchema;
    public static String targetSchema;
    public static String sourceSchemaType;
    public static String targetSchemaType;
    public static SchemaTree schemaJTree1;
    public static SchemaTree schemaJTree2;
    public static String view;

    // the threshold for differentiating potentially trustable initial mappings
    // from previous matching or user input mappings from other mappins
    // these mappings won't be changed in later matching steps
    public static double GOOD_MAPPING_THRESHOLD = 2.0 / 3.0;

    // the threshold for selecting initial mappings from linguistic matching
    // for similarity flooding graph matching 
    public static double LINGUISTIC_THRESHOLD = 0.50;

    // the threshold for detecting possibly multiple matches
    // e.g. fullname should match to both lastname and firstname, not one of them
    public static double DIFFERENCE_THRESHOLD = 0.10;

    // threshold for deciding whether detecting possibly multiple matches or not
    // if second max sim > this threshold, it means it is significant match, 
    // should be considered, otherwise don't bother to pay attention to it 
    public static double SIGNIFICANT_SIM_THRESHOLD = 0.20;

    // be considered, otherwise don't bother to pay attention to it 
    public static double LOWEST_SIGNIFICANT_SIM_THRESHOLD = 0.10;

    // input matches with similarity beyond this threshold are regarded as user
    // input matches and are really reused in the further matching process 
    public static double REUSE_SIM_THRESHOLD = 0.6;

    // if the match sim is very high, don't care whether it is consistent with 
    // the other direction match
    public static double NO_CARE_DIRECTION_THRESHOLD = 0.8;

    // if multiple good linguistic and type matches have high enough sim, means
    // there might be really multiple good matches
    public static double MULTIPLE_LING_MATCHES_THRESHOLD = 0.7;

    // the weight of name similarity at linguistic matching step
    public static double NAME_WEIGHT_LINGUISTIC = 0.5;

    // the weight of path similarity at linguistic matching step
    public static double PATH_WEIGHT_LINGUISTIC = 0.5;

    // the weight of name, path data type and graph  similarity 
    //at combination of linguistic and structural matching step
    public static double NAME_WEIGHT_COMB = 0.25; // for book 0.1, bio/eco 0.3
    public static double PATH_WEIGHT_COMB = 0.25; // for book 0.4, bio/eco 0.3
    public static double GRAPH_WEIGHT_COMB = 0.3; // for book 0.5, bio/eco 0.4
    public static double DATATYPE_WEIGHT_COMB = 0.05; //for book, bio/eco 0.0
    public static double DESCRIPTION_WEIGHT_COMB = 0.05; //for book, bio/eco 0.0

    // for name, path & data type
    public static double LINGUITIC_WEIGHT_COMB = 0.5; // bio/eco 0.6 

    /* the weight for update the ancestors' match similarity values during
       context check, when users confirm or edit a match */
    public static double STRUCT_WEIGHT_UPDATE_ANCESTOR = 0.5;

    // the similarity threshold for deciding learnt mappings for core elements
    // good enough or not, if not, user input is required in order to improve 
    // the whole subtree matching
    public static double CORE_ACCEPT_THRESHOLD = 0.50; // 0.4 for bio dtds
    public static double CORE_LEAF_ACCEPT_THRESHOLD = 0.50;
    public static double OUTPUT_ACCEPT_THRESHOLD = 0.20;

    // if a node got multiple low similarity similar matches, it might mean
    // it does not have good matches at all, throw them away
    public static double MULTI_BAD_MATCHES_UPPER_SIM = 0.40;

    // if context node's best match sim is not high enough, but their data type match
    // for only  both XML schemas
    public static double TYPE_MATCH_CORE_LEAF_THRESHOLD = 0.35;

    // the threshold for selecting initial mappings from combination of name,
    // path, and graph matching for core elements's subtree mapping refinement
    public static double SUBTREE_COMB_THRESHOLD = 0.20;

    // the similarity given to hypernym
    public static double HYPERNYM_SIMILARITY = 0.8;

    // the coefficient for reducing the similaty of paths normalized with hyponym
    public static double PATH_HYPERNYM_COEFF = 0.8;

    /* the coefficient for reducing the similaty of names and paths if they
       both end with a digit, but the digits are different,
       e.g street1 vs. street2. Or lower, even close to 0 ???
    */
    public static double DIFF_LAST_DIGIT_COEFF = 0.3;

    // increasing the linguistic similarity with same data type
    public static double SAME_DATA_TYPE_SIM = 0.20;

    // define big subtree size lower limit
    public static long BIG_SUBTREE_LOWER_LIMIT = 4;

    //build data type similarity table
    public static Hashtable dataTypeSimTable = buildDataTypeSimTable();

    // target schemas of this application
    HashSet targetSchemas = new HashSet();

    // all source schema of this application
    HashSet allSourceSchemas = new HashSet();

    // table of schema's document name, virtual or real
    // key = schema file name
    // content = document name
    Hashtable docTable = new Hashtable();

    // table of target schema's corresponding source schema list
    // key = target schema file name, content = list of source schemas
    Hashtable sourceSchemas = new Hashtable();

    // table of individual target-source pair views
    // key = target schema file name + source schema file name
    // content = view string
    Hashtable individualViews = new Hashtable();

    // table of target views, key = virtual target document name
    // content = integrated view (union of the corresponding target's 
    // individual views
    Hashtable targetViews = new Hashtable();
    PrintWriter cout = new PrintWriter(System.err, true);

    // constructor
    public SCIA() throws Exception {
    }

    public String generateViewForSeqAndBSML() throws Exception {
        PrintWriter cout = new PrintWriter(System.err, true);

        // target schema tree
        SchemaTree tmpTree = new SchemaTree("/schemas/sequence.dtd", "DTD");
        SchemaTree targetTree = new SchemaTree(tmpTree);

        //targetTree.write(cout, true);
        // source schema tree
        tmpTree = new SchemaTree("/schemas/bsml2_4.dtd", "DTD");

        SchemaTree sourceTree = new SchemaTree(tmpTree);

        //targetTree.write(cout, true);
        //sourceTree.write(cout, true);
        // set up matches manually
        // for root sequences = Bsml/Definitions/Sequences
        TNode sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences");
        targetTree.rootNode.fromComb = new Matching(sourceNode, 1.0);

        // sequences/sequence = Bsml/Definitions/Sequences/Sequence
        TNode targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence");

        if (targetNode == null) {
            if (debug_on) {
                System.err.println("target is null");
            }
        }

        if (sourceNode == null) {
            if (debug_on) {
                System.err.println("source is null");
            }
        }

        targetNode.fromComb = new Matching(sourceNode, 1.0);

        // Sequences/Sequence@name = Bsml/Definitions/Sequences/Sequence@locus
        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/@name");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@locus");

        if (sourceNode == null) {
            if (debug_on) {
                System.err.println("source locus = null");
            }
        }

        if (targetNode == null) {
            if (debug_on) {
                System.err.println("target name = null");
            }
        }

        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/@id");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@ic-acckey");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/@length");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@length");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/@type");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@molecule");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/@genomeref");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@genomeref");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/@topology");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@topology");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/@alignment-sequence");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@alignment-sequence");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/@alignment-position");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@alignment-position");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/@alignment-inverted");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/@alignment-inverted");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/Organism");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Genomes/Genome/Organism");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        // condition ???
        targetNode.fromComb.condition = new Condition(sourceTree.getTNodeFromPath(
                    "/Bsml/Definitions/Genomes/Genome/@id"), "=",
                sourceTree.getTNodeFromPath(
                    "/Bsml/Definitions/Sequences/Sequence/@genomeref"));

        if (debug_on) {
            System.err.println("condition = " +
                targetNode.fromComb.condition.toString());
        }

        //targetNode.fromComb.condition = 
        //    new String("$Genome@id = $Sequence/genomeref");
        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/Seq-data");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/Seq-data");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        // independent (global) union 1:2 matches: segment and feature
        targetNode = targetTree.getTNodeFromPath("/Sequences/Sequence/Segment");
        targetNode.numOfIndependentUnionMatches = 2;
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/Segment");
        targetNode.independentUnionMatches = new ArrayList(2);
        targetNode.independentUnionMatches.add(new Matching(sourceNode, 1.0));
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/Feature-tables/Feature-table/Feature");
        targetNode.independentUnionMatches.add(new Matching(sourceNode, 1.0));

        //dependent union 1:2 matches
        targetNode = targetTree.getTNodeFromPath(
                "/Sequences/Sequence/Segment/@seg-start");
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/Segment/@seg-start");
        targetNode.dependentUnionMatches = new ArrayList(2);
        targetNode.dependentUnionMatches.add(new Matching(sourceNode, 1.0));
        sourceNode = sourceTree.getTNodeFromPath(
                "/Bsml/Definitions/Sequences/Sequence/Feature-tables/Feature-table/Feature/Location/Interval-loc/@startpos");
        targetNode.dependentUnionMatches.add(new Matching(sourceNode, 1.0));

        //targetTree.write(cout, true);
        String view = targetTree.generateView("bsml.xml");

        if (debug_on) {
            cout.println(view);
        }

        cout.flush();

        return view;
    }

    public String generateViewForObsAndStudyA() throws Exception {
        PrintWriter cout = new PrintWriter(System.err, true);

        // target schema tree
        SchemaTree tmpTree = new SchemaTree("observations.xsd", "XMLSchema");
        SchemaTree targetTree = new SchemaTree(tmpTree);

        //targetTree.write(cout, true);
        // source schema tree
        tmpTree = new SchemaTree("studyA.xsd", "XMLSchema");

        SchemaTree sourceTree = new SchemaTree(tmpTree);

        //sourceTree.write(cout, true);
        // set up matches manually
        // for root observations = studyA
        targetTree.rootNode.fromComb = new Matching(sourceTree.rootNode, 1.0);

        // observations/observation = studyA/obs
        TNode targetNode = targetTree.getTNodeFromPath(
                "/Observations/Observation");
        TNode sourceNode = sourceTree.getTNodeFromPath("/StudyA/Obs");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Observations/Observation/Date");
        sourceNode = sourceTree.getTNodeFromPath("/StudyA/Obs/Date");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Observations/Observation/Site");
        sourceNode = sourceTree.getTNodeFromPath("/StudyA/Obs/Location");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/Observations/Observation/Density");
        sourceNode = sourceTree.getTNodeFromPath("/StudyA/Obs/Count");

        MatchingUnit mu = new MatchingUnit(sourceNode, null, "DIV");
        java.util.List muList = new ArrayList();
        muList.add(mu);
        sourceNode = sourceTree.getTNodeFromPath("/StudyA/Obs/Area");
        mu = new MatchingUnit(sourceNode, "DIV (1000000)", null);
        muList.add(mu);
        targetNode.fromComb = new Matching(muList, 1.0);

        //targetTree.write(cout, true);
        String view = targetTree.generateView("studyA.xml");

        if (debug_on) {
            cout.println(view);
        }

        cout.flush();

        return view;
    }

    public String generateViewForLTEROverGCE() throws Exception {
        PrintWriter cout = new PrintWriter(System.err, true);

        // target schema tree
        SchemaTree tmpTree = new SchemaTree("/schemas/LTER-standard.xml",
                "XMLSchema");
        SchemaTree targetTree = new SchemaTree(tmpTree);

        //targetTree.write(cout, true);
        // source schema tree
        tmpTree = new SchemaTree("/schemas/GCE-0409.xml", "XMLSchema");

        SchemaTree sourceTree = new SchemaTree(tmpTree);

        //sourceTree.write(cout, true);
        // set up matches manually
        targetTree.rootNode.fromComb = new Matching(sourceTree.rootNode, 1.0);

        TNode targetNode = targetTree.getTNodeFromPath(
                "/LTER-Observation/Tuple");
        TNode sourceNode = sourceTree.getTNodeFromPath("/GCE-ObsTable/Row");
        Vector grpAttrs = new Vector();
        TNode grpAttrNode = sourceTree.getTNodeFromPath(
                "/GCE-ObsTable/Row/Site");
        grpAttrs.add(grpAttrNode);
        grpAttrNode = sourceTree.getTNodeFromPath("/GCE-ObsTable/Row/Treatment");
        grpAttrs.add(grpAttrNode);
        grpAttrNode = sourceTree.getTNodeFromPath("/GCE-ObsTable/Row/Replicate");
        grpAttrs.add(grpAttrNode);

        targetNode.fromComb = new Matching(sourceNode, 1.0, grpAttrs);

        targetNode = targetTree.getTNodeFromPath(
                "/LTER-Observation/Tuple/Block");
        sourceNode = sourceTree.getTNodeFromPath("/GCE-ObsTable/Row/Site");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/LTER-Observation/Tuple/Plot");
        sourceNode = sourceTree.getTNodeFromPath("/GCE-ObsTable/Row/Replicate");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/LTER-Observation/Tuple/Biodiversity");
        sourceNode = sourceTree.getTNodeFromPath(
                "/GCE-ObsTable/Row/Species_Code");
        targetNode.fromComb = new Matching(sourceNode, 1.0, "count");

        targetNode = targetTree.getTNodeFromPath(
                "/LTER-Observation/Tuple/Productivity");
        sourceNode = sourceTree.getTNodeFromPath(
                "/GCE-ObsTable/Row/Plant_Mass_m2");
        targetNode.fromComb = new Matching(sourceNode, 1.0, "sum");

        //targetTree.write(cout, true);
        String view = targetTree.generateView("studyA.xml");

        if (debug_on) {
            cout.println(view);
        }

        cout.flush();

        return view;
    }

    public void generateViewExampleForBooks() throws Exception {
        PrintWriter cout = new PrintWriter(System.err, true);

        // target schema tree
        SchemaTree tmpTree = new SchemaTree("book.dtd", "DTD");
        SchemaTree targetTree = new SchemaTree(tmpTree);

        //targetTree.write(cout, true);
        // source schema tree
        tmpTree = new SchemaTree("book1.dtd", "DTD");

        SchemaTree sourceTree = new SchemaTree(tmpTree);

        // set up matches manually
        // for root books = bib
        targetTree.rootNode.fromComb = new Matching(sourceTree.rootNode, 1.0);

        // books/book = bib/book
        TNode targetNode = targetTree.getTNodeFromPath("/books/book");
        TNode sourceNode = sourceTree.getTNodeFromPath("/bib/book");

        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/books/book/year");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/@year");

        if (sourceNode == null) {
            if (debug_on) {
                System.err.println("########sourceNode year = null");
            }
        }

        //System.exit(0);
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/books/book/title");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/title");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/books/book/author");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/author");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        //targetNode = targetTree.getTNodeFromPath("/books/book/editor");
        //sourceNode = sourceTree.getTNodeFromPath("/bib/book/editor");
        //targetNode.fromComb = new Matching(sourceNode, 1.0);
        targetNode = targetTree.getTNodeFromPath(
                "/books/book/author/full_name/first_name");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/author/first");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath(
                "/books/book/author/full_name/last_name");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/author/last");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        targetNode = targetTree.getTNodeFromPath("/books/book/price");
        sourceNode = sourceTree.getTNodeFromPath("/bib/book/price");
        targetNode.fromComb = new Matching(sourceNode, 1.0);

        //targetTree.write(cout, true);
        String view = targetTree.generateView("book1.xml");

        if (debug_on) {
            cout.println(view);
        }

        cout.flush();
    }

    /**
     * Build the mappings among common data types in schema languages,
     including XML Schema, DTDs, relational schemas and ontologies.
     @param sourceTree The source SchemaTree
     @param targetTree The target SchemaTree
     */
    public static java.util.List buildTypeMap(SchemaTree sourceTree,
        SchemaTree targetTree) {
        java.util.List typeMap = new ArrayList();
        typeMap.add(new MapPair(targetTree.parentElem, sourceTree.parentElem,
                1.0));
        typeMap.add(new MapPair(targetTree.leafElem, sourceTree.leafElem, 1.0));
        typeMap.add(new MapPair(targetTree.rootElem, sourceTree.rootElem, 1.0));
        typeMap.add(new MapPair(targetTree.CDATA, sourceTree.CDATA, 1.0));
        typeMap.add(new MapPair(targetTree.PCDATA, sourceTree.PCDATA, 1.0));
        typeMap.add(new MapPair(targetTree.INT, sourceTree.INT, 1.0));
        typeMap.add(new MapPair(targetTree.FLOAT, sourceTree.FLOAT, 1.0));
        typeMap.add(new MapPair(targetTree.DATE, sourceTree.DATE, 1.0));
        typeMap.add(new MapPair(targetTree.STRING, sourceTree.STRING, 1.0));
        typeMap.add(new MapPair(targetTree.NUMBER, sourceTree.NUMBER, 1.0));

        return typeMap;
    }

    /**
     * Build the common data type compatibility table. Key = (datatype1,datatype2)
     */
    public static Hashtable buildDataTypeSimTable() {
        Hashtable dtSimTable = new Hashtable();
        dtSimTable.put(new String("STRING" + "," + "STRING)"), new Double(1.0));
        dtSimTable.put(new String("INT" + "," + "INT)"), new Double(1.0));
        dtSimTable.put(new String("FLOAT" + "," + "FLOAT)"), new Double(1.0));
        dtSimTable.put(new String("NUMBER" + "," + "NUMBER)"), new Double(1.0));
        dtSimTable.put(new String("INTEGER" + "," + "INTEGER)"), new Double(1.0));
        dtSimTable.put(new String("INTEGER" + "," + "INT)"), new Double(1.0));
        dtSimTable.put(new String("INT" + "," + "INTEGER)"), new Double(1.0));
        dtSimTable.put(new String("DATE" + "," + "DATE)"), new Double(1.0));
        dtSimTable.put(new String("DOUBLE" + "," + "DOUBLE)"), new Double(1.0));
        dtSimTable.put(new String("CDATA" + "," + "CDATA)"), new Double(1.0));
        dtSimTable.put(new String("PCDATA" + "," + "PCDATA)"), new Double(1.0));

        return dtSimTable;
    }

    public static String schemaTreeMatch(boolean interactive_on,
        Domain appDomain, SchemaTree targetTree, SchemaTree sourceTree,
        boolean bothXMLSchemas, boolean bothDirection)
        throws Exception {
        //if(debug_on) System.err.println("\n\n^^^^^^ In schemaTreeMatch");
        PrintWriter cout = new PrintWriter(System.err, true);

        TNode targetTreeRoot = targetTree.rootNode;

        //System.gc();
        // source tree
        TNode sourceTreeRoot = sourceTree.rootNode;

        // test treeMatch
        //Hashtable simHash = new Hashtable (20);
        //targetTree.treeMatch(sourceTreeRoot, targetTreeRoot, simHash);
        //cout.println(simHash);
        //cout.println ("simHash contains " + simHash.size() + "keys");
        //test SchemaTree::getBestNode()
        //cout.println();
        //cout.println("==== Get best node from only node name similarity ====");
        //targetTree.getBestNode(targetTreeRoot, sourceTreeRoot);
        // test SchemaTree::getBestPath()
        //cout.println();
        //cout.println("=== Get best path from name similarity and hierarchy");
        //targetTree.getBestPath();
        //test SchemaTree::getBestPurePath()
        //cout.println();
        //cout.println("=== Get best path from only path string similarity ===");
        //targetTree.getBestPurePath(targetTreeRoot, sourceTreeRoot);
        //cout.flush();
        if (debug_on) {
            System.err.println(
                "====== Get description similarity tables ======");
        }

        Hashtable descSimTable = new Hashtable();

        DescriptionMatcher dm = new DescriptionMatcher(targetTree, sourceTree,
                descSimTable);

        // test SchemaTree: getNameAndPathSimTables()
        if (debug_on) {
            System.err.println(
                "====== Get name and path similarity tables ======");
        }

        Hashtable nameSimTable = new Hashtable();
        Hashtable pathSimTable = new Hashtable();

        targetTree.getNameAndPathSimTables(nameSimTable, pathSimTable,
            sourceTreeRoot, appDomain);

        //cout.flush();

        // test SchemaTree:getFromNameAndPath()
        //cout.println();

        if (debug_on) {
            System.err.println("=== Get mappings from name and path matching");
        }

        targetTree.getFromNameAndPath(sourceTreeRoot, nameSimTable, pathSimTable);

        if (interactive_on) {
            // get linguistic matchings
            targetTree.getLingMatchings(sourceTreeRoot, nameSimTable,
                pathSimTable);

            //compute subtree size for all nodes
            targetTreeRoot.getSubTreeSize();

            // get the number of potential matches for each node
            targetTree.getNumOfPotentialMatches(sourceTree, nameSimTable,
                pathSimTable, descSimTable, dataTypeSimTable, interactive_on);
        }

        //cout.flush();

        //test SchemaTree:toRDFModel(), getInitMap() and SFMatcher
        //cout.println();

        if (debug_on) {
            System.err.println(
                "====== Get mappings from structural similarity with initial mappings from name & path matching");
        }

        Hashtable tHash = new Hashtable();
        Hashtable sHash = new Hashtable();
        Model m1 = targetTree.toRDFModel(tHash);
        //cout.flush();

        Model m2 = sourceTree.toRDFModel(sHash);

        //sfm.formula = FORMULA_FFT;
        //sfm.FLOW_GRAPH_TYPE = FG_PRODUCT;
        java.util.List initMap = targetTree.getInitMapFromLinguistic(tHash,
                sHash);

        if (reuseMap_on) {
            /* Set input map with similarity beyond threshold as user input
               matches and get them into a list */
            java.util.List goodInputMap = targetTree.getGoodInputMap(tHash,
                    sHash);

            /* Combine the user input map and the initial map from linguistic
               matching, remove repeated ones */
            initMap = combineMaps(goodInputMap, initMap);
        }

        if (debug_on) {
            System.err.println("Init map before adding type = " + initMap);
        }

        //cout.flush();

        java.util.List typeMap = buildTypeMap(sourceTree, targetTree);
        initMap.addAll(typeMap);

        SFMatcher sfm = new SFMatcher();
        MapPair[] result = sfm.getMatch(m1, m2, initMap);
        MapPair.sortGroup(result);

        //sfm.dump(result);
        //test SchemaTree::getFromComb()
        if (debug_on) {
            System.err.println();
        }

        if (debug_on) {
            System.err.println(
                "====== Get mappings from graph, path, name similarity ===");
        }

        Map mapFromGraph = MapPair.sortedCandidates(result, false);
        //cout.flush();
        targetTree.getFromComb(mapFromGraph, nameSimTable, pathSimTable,
            descSimTable, dataTypeSimTable, tHash, sourceTree, false, false,
            interactive_on);

        if (debug_on) {
            System.err.println("after first combination");
        }

        //targetTree.write(cout, false);
        //cout.flush();

        if (interactive_on) {
            //draw mapping in the gui
            mainUI.addMatchedLines();
            mainUI.treesSplitPane.drawMapping(mainUI.treesSplitPane.getGraphics());
        }

        //test SchemaTree:adjustMatchWithHier()
        if (debug_on) {
            System.err.println();
        }

        if (debug_on) {
            System.err.println();
        }

        //set nodes' realParent and realChildren for structual matching
        targetTree.setRealParentChildren();

        //cout.println("====== Get mappings from graph, path, name, " +
        //  and hierarchy similarity ===");
        //targetTree.adjustMatchWithHier();
        // do context consistency enforcement step or not
        String msg = "Go to the context check step ?";

        String contextCheck; // = BR.readLine();

        int keyEntered = 0;
        final Interactive inter = new Interactive(msg, keyEntered);

        inter.show();

        Thread.sleep(1000);
        mainUI.treesSplitPane.paintImmediately(0, 0,
            mainUI.treesSplitPane.getWidth(), mainUI.treesSplitPane.getHeight());
        mainUI.treesSplitPane.drawMapping(mainUI.treesSplitPane.getGraphics());

        while (inter.isVisible()) {
            Thread.sleep(100);
        }

        ;

        if (inter.keyEntered == KeyEvent.VK_Y) {
            context_check = true;

            if (debug_on) {
                System.err.println(
                    "Finding context-holding elements and enforcing " +
                    "context consistency of their subtree matches");
            }

            //test SchemaTree::getCoreLevels()
 
            if (debug_on) {
               System.err.println();
               System.err.println(); 
               System.err.println("====== test getCoreLevels  =====");
            }

            long[] coreLevels = targetTree.getCoreLevels();

            //get rid of the effect of compositor nodes for computing core levels
            if (targetTree.schemaType.equals("XMLSchema")) {
                SchemaTree cleanTargetTree = targetTree.removeCompositorNodes();
                coreLevels = cleanTargetTree.getCoreLevels();
            }

            if (debug_on) {
                System.err.print("Core levels are: " + coreLevels[0] + " and " +
                    coreLevels[1]);
            }

            //test SchemaTree:adjustMatchWithCoreLevel()
            if (debug_on) {
                System.err.println();
            }

            if (debug_on) {
                System.err.println();
            }

            if (debug_on) {
                System.err.println("==== test adjustMatchWithCoreLevel:" +
                    " the main entity level");
            }

            targetTree.adjustMatchWithCoreLevel(coreLevels[0] - 1, sourceTree,
                nameSimTable, pathSimTable, descSimTable, dataTypeSimTable,
                false, interactive_on);

            if (debug_on) {
                System.err.println("After adjusting main entity level");
            }

            //cout.flush();

            //targetTree.write(cout, false);
            if (debug_on) {
                System.err.println();
            }

            if (debug_on) {
                System.err.println();
            }

            if (debug_on) {
                System.err.println(
                    "=== test adjustMatchWithCoreLevel: main aspect level");
            }

            targetTree.adjustMatchWithCoreLevel(coreLevels[0], sourceTree,
                nameSimTable, pathSimTable, descSimTable, dataTypeSimTable,
                false, interactive_on);

            if (!interactive_on) {
                sourceTree.clearMatchedTimes();

                if (debug_on) {
                    System.err.println("non-interact");
                }
            }

            if ((coreLevels[1] > 2) && interactive_on) {
                if (debug_on) {
                    System.err.println("inside");
                }

                int level = (int) coreLevels[1] - 1;

                msg = new String(
                        "Check the context for whole main aspect level" + " " +
                        level + "?");

                final Interactive inter1 = new Interactive(msg, keyEntered);

                inter1.show();

                if (debug_on) {
                    System.err.println("after show");
                }

                Thread.sleep(1000);
                mainUI.treesSplitPane.paintImmediately(0, 0,
                    mainUI.treesSplitPane.getWidth(),
                    mainUI.treesSplitPane.getHeight());
                mainUI.treesSplitPane.drawMapping(mainUI.treesSplitPane.getGraphics());

                while (inter1.isVisible()) {
                    Thread.sleep(100);
                }

                ;

                if (keyEntered == KeyEvent.VK_Y) {
                    targetTree.adjustMatchWithCoreLevel(coreLevels[1] - 1,
                        sourceTree, nameSimTable, pathSimTable, descSimTable,
                        dataTypeSimTable, false, interactive_on);
                } else {
                    msg = "Check the context for element with sub-tree at main" +
                        "aspect level " + level + "?";

                    final Interactive inter2 = new Interactive(msg, keyEntered);

                    inter2.show();

                    Thread.sleep(1000);
                    mainUI.treesSplitPane.paintImmediately(0, 0,
                        mainUI.treesSplitPane.getWidth(),
                        mainUI.treesSplitPane.getHeight());
                    mainUI.treesSplitPane.drawMapping(mainUI.treesSplitPane.getGraphics());

                    while (inter2.isVisible()) {
                        Thread.sleep(100);
                    }

                    ;

                    if (keyEntered == KeyEvent.VK_Y) {
                        targetTree.adjustMatchWithCoreLevel(coreLevels[1] - 1,
                            sourceTree, nameSimTable, pathSimTable,
                            descSimTable, dataTypeSimTable, true, interactive_on);
                    }
                }
            }
        }

        //if(debug_on) System.err.println("before both dir, interactive_mode = " + SchemaTree.interactive_on);
        // to match the opposite direction?
        if (bothDirection) {
            schemaTreeMatch(interactive_on, appDomain, sourceTree, targetTree,
                bothXMLSchemas, false);
            targetTree.writeBoth(cout, true, sourceTree);
        }

        //draw mapping in the gui
        mainUI.addMatchedLines();
        mainUI.treesSplitPane.drawMapping(mainUI.treesSplitPane.getGraphics());

        //cout.flush();
        //cout.close();

        return "Done";
    }

    public static Domain loadDomainThesauri(String domainName) {
        StringTokenizer st;
        String line;
        String line1;
        String domainFileName = null;

        if (domainName.equalsIgnoreCase("Ecology")) {
            domainFileName = "ecology.thes";
        } else if (domainName.equalsIgnoreCase("Business")) {
            domainFileName = "purchaseOrder.thes";
        } else if (domainName.equalsIgnoreCase("Biology")) {
            domainFileName = "biology.thes";
        } else if (domainName.equalsIgnoreCase("Bibliography")) {
            domainFileName = "bibliography.thes";
        } else if (domainName.equalsIgnoreCase("Other")) {
        }

        if (domainFileName == null) {
            return null;
        }

        // initialize domain tables
        Hashtable synonyms = new Hashtable();
        Hashtable abbrevs = new Hashtable();
        Hashtable hypernyms = new Hashtable();

        Vector vec = new Vector();

        try {
            String filePath = SCIA_WORK_DIR + "/auxil/" + domainFileName;

            // For reading file from jar file, only using relative path from 
            //CLASSPATH
            //InputStream is = this.getClass().getResourceAsStream(filePath);
            InputStream is = new FileInputStream(filePath);

            if (is == null) {
                if (debug_on) {
                    System.err.println(filePath + " InputStream = null");
                }
            } else {
                if (debug_on) {
                    System.err.println("InputStream of " + domainFileName +
                        " = " + is.available());
                }
            }

            BufferedReader br = new BufferedReader(new InputStreamReader(is));

            // load abbreviations
            line1 = br.readLine();
            line = br.readLine();

            if (line1.equalsIgnoreCase("FULL FORM & ABBREVIATIONS") &&
                    line.startsWith("==", 0) // line contains "========"
                     &&line.endsWith("==")) {
                line = br.readLine();

                do {
                    st = new StringTokenizer(line);

                    if (st.countTokens() >= 3) { //at least one abbrev for a term
                        vec = new Vector();

                        String fullForm = (String) st.nextElement();
                        st.nextElement(); //throw "="

                        String abbrev = (String) st.nextElement();

                        do {
                            if (abbrev.endsWith(",")) {
                                abbrev = abbrev.substring(0, abbrev.length() -
                                        1);
                            }

                            vec.addElement(abbrev);
                        } while (st.hasMoreElements() &&
                                ((abbrev = (String) st.nextElement()) != null));

                        //if(debug_on) System.err.println("fullForm = " + fullForm +
                        //		   "abbrevs = " + vec);
                        abbrevs.put((Object) fullForm, (Object) vec);
                    }
                } while (((line = br.readLine()) != null) &&
                        !(line.equalsIgnoreCase("SYNONYMS")));
            }

            if ((line = br.readLine()).startsWith("==", 0) &&
                    line.endsWith("==")) { // line contains "========"
                line = br.readLine();

                do {
                    st = new StringTokenizer(line);

                    if (st.countTokens() >= 2) { //at least 2 synonyms

                        String syn = (String) st.nextElement();
                        vec = new Vector();

                        do {
                            if (syn.endsWith(",")) {
                                syn = syn.substring(0, syn.length() - 1);
                            }

                            vec.addElement(syn);
                        } while (st.hasMoreElements() &&
                                ((syn = (String) st.nextElement()) != null));

                        //if(debug_on) System.err.println("all syn vec = " + vec);
                        for (int i = 0; i < vec.size(); i++) {
                            Vector synVec = new Vector();
                            Object keySyn = vec.elementAt(i);

                            for (int j = 0; j < vec.size(); j++) {
                                if (j != i) {
                                    synVec.addElement(vec.elementAt(j));
                                }
                            }

                            synonyms.put(keySyn, (Object) synVec);

                            //if(debug_on) System.err.println("key = " + keySyn +
                            //		   "    synVec = " + synVec);
                            //if(debug_on) System.err.println("syn table = " + synonyms);
                        }
                    }
                } while (((line = br.readLine()) != null) &&
                        !(line.equalsIgnoreCase("HYPERNYMS")));
            }

            // load hypernyms
            if ((line = br.readLine()).startsWith("==", 0) &&
                    line.endsWith("==")) { // line contains "========"
                line = br.readLine();

                do {
                    //if(debug_on) System.err.println("******line = " + line);
                    st = new StringTokenizer(line);

                    if (st.countTokens() >= 3) { //at least one entry

                        String big = (String) st.nextElement();
                        vec = new Vector();

                        Vector bigVec = new Vector();
                        bigVec.addElement(big);
                        st.nextElement(); // throw ">"

                        String small = (String) st.nextElement();

                        do {
                            if (small.endsWith(",")) {
                                small = small.substring(0, small.length() - 1);
                            }

                            vec.addElement(small);
                        } while (st.hasMoreElements() &&
                                ((small = (String) st.nextElement()) != null));

                        if (debug_on) {
                            System.err.println("small vec = " + vec +
                                "   big vec = " + bigVec);
                        }

                        //for big --> small ones
                        // check big has an entry already or not
                        Vector exist = (Vector) hypernyms.get((Object) big);

                        if (exist != null) {
                            if (debug_on) {
                                System.err.println("for big = " + big +
                                    "   exist vec = " + exist);
                            }

                            // ensure big's hypernym list contains all small
                            for (int i = 0; i < vec.size(); i++) {
                                Object elem = vec.elementAt(i);

                                if (!exist.contains(elem)) {
                                    //add it in
                                    if (debug_on) {
                                        System.err.println(
                                            "  before adding elem = " +
                                            (String) elem + "  hyper table = " +
                                            hypernyms);
                                    }

                                    exist.addElement(elem);

                                    /* ????? when changing y's entry,
                                    why the above exist.addElement() change z's
                                    entry to z=[x,z] from z=[x]in hypernym table?

                                    */
                                    //hypernyms.put((Object)big, exist);
                                    System.err.print("  after adding elem=" +
                                        (String) elem + ", exist = " + exist +
                                        "  hyper table = " + hypernyms + "\n");
                                }
                            }
                        } else { // no hypernym list for big yet
                            hypernyms.put((Object) big, (Object) vec);
                        }
                    }

                    if (debug_on) {
                        System.err.println("hypernym table = " + hypernyms);
                    }
                } while ((line = br.readLine()) != null);
            }

            br.close();
        } catch (IOException e) {
            if (debug_on) {
                System.err.println(e.getMessage());
            }
        }

        Domain domain = new Domain(synonyms, abbrevs, hypernyms);

        if (debug_on) {
            System.err.println("synonyms = " + synonyms);
        }

        if (debug_on) {
            System.err.println("abbrevs = " + abbrevs);
        }

        if (debug_on) {
            System.err.println("hypernyms = " + hypernyms);
        }

        return domain;
    }

    public void commandLineSchemaMatch(String[] args) throws Exception {
        // get application domain
        BufferedReader BR = new BufferedReader(new InputStreamReader(System.in));
        PrintWriter cout = new PrintWriter(System.err, true);
        cout.println(
            "Please specify your application domain, choose one from: ");
        cout.println("Bibliograghy (1)");
        cout.println("Biology (2)");
        cout.println("Business (3)");
        cout.println("Ecology (4)");
        cout.println("Other (5)");
        cout.flush();

        char dm = (char) BR.read();
        String domainName = "Other";

        if (dm == '2') {
            domainName = "Biology";

            if (debug_on) {
                System.err.println("Application Domain is Biology.");
            }

            if (debug_on) {
                System.err.println(
                    "Biological thesauri are used in linguistic matching.");
            }
        } else if (dm == '3') {
            domainName = "Business";

            if (debug_on) {
                System.err.println("Application Domain is Business.");
            }

            if (debug_on) {
                System.err.println(
                    "Purchase Order thesauri are used in linguistic matching.");
            }
        } else if (dm == '4') {
            domainName = "Ecology";

            if (debug_on) {
                System.err.println("Application Domain is Ecology.");
            }

            if (debug_on) {
                System.err.println(
                    "Ecological thesauri are used in linguistic matching.");
            }
        }

        Domain domain = loadDomainThesauri(domainName);

        // get the target schema
        BR = new BufferedReader(new InputStreamReader(System.in));
        cout.print("Please input the Target schema file name: ");
        cout.flush();

        String targetSchemaFileName = BR.readLine();
        String targetSchemaType = "OTHER";

        if (targetSchemaFileName.endsWith("dtd") ||
                targetSchemaFileName.endsWith("DTD")) {
            targetSchemaType = "DTD";
        } else if (targetSchemaFileName.endsWith("xml") ||
                targetSchemaFileName.endsWith("XML") ||
                targetSchemaFileName.endsWith("xsd") ||
                targetSchemaFileName.endsWith("XSD")) {
            targetSchemaType = "XMLSchema";
        }

        BR = new BufferedReader(new InputStreamReader(System.in));
        cout.print("Please input the Source schema file name: ");
        cout.flush();

        String sourceSchemaFileName = BR.readLine();
        cout.println();

        String sourceSchemaType = "OTHER";

        if (sourceSchemaFileName.endsWith("dtd") ||
                sourceSchemaFileName.endsWith("DTD")) {
            sourceSchemaType = "DTD";
        } else if (sourceSchemaFileName.endsWith("xml") ||
                sourceSchemaFileName.endsWith("XML") ||
                sourceSchemaFileName.endsWith("xsd") ||
                sourceSchemaFileName.endsWith("XSD")) {
            sourceSchemaType = "XMLSchema";
        }

        SchemaTree targetTree = new SchemaTree();
        SchemaTree sourceTree = new SchemaTree();

        targetTree = new SchemaTree(targetSchemaFileName, targetSchemaType);

        if (targetSchemaType.equals("DTD")) {
            bothXMLSchemas = false;
        } else if (targetSchemaType.equals("OTHER")) {
            if (debug_on) {
                System.err.println(
                    "Please make sure your schema file name having suffix of .xsd or .xml for XML Schemas, or .dtd for DTDs. Other type of schemas, e.g., relational schemas are not handled yet.");
            }

            System.exit(0);
        }

        cout.print("Target Schema Tree:\n");
        targetTree.write(cout, false);
        cout.flush();

        sourceTree = new SchemaTree(sourceSchemaFileName, sourceSchemaType);

        if (sourceSchemaType.equals("DTD")) {
            bothXMLSchemas = false;
        } else if (sourceSchemaType.equals("OTHER")) {
            if (debug_on) {
                System.err.println(
                    "Please make sure your schema file name having suffix of .xsd or .xml for XML Schemas, or .dtd for DTDs. Other type of schemas, e.g., relational schemas are not handled yet.");
            }

            System.exit(0);
        }

        if (sourceSchemaType.equals("XMLSchema") &&
                targetSchemaType.equals("XMLSchema")) {
            bothXMLSchemas = true;
        }

        cout.print("Source Schema Tree:\n");
        sourceTree.write(cout, false);
        cout.flush();

        boolean interactive_on = false;
        /*process options
        int the_option; // from the command line
        Getopt all_options = new Getopt("SCIA", args, "i");


        while ((the_option = all_options.getopt()) != -1) {
            switch (the_option) {
            //case 'd': SchemaTree.debug_on = true;
            //break;
            case 'i':
                interactive_on = true;

                break;
            }
        }
        */
        schemaTreeMatch(interactive_on, domain, targetTree, sourceTree,
            bothXMLSchemas, bothDirection);
    }

    /**
       Combine the second map list into the first one, get rid of repeated
       ones.
       @param map1 The first Map list
       @param map1 The second Map list
    */
    public static java.util.List combineMaps(java.util.List map1,
        java.util.List map2) throws Exception {
        java.util.List resMap = null;

        if ((map1 == null) && (map2 != null)) {
            resMap = map2;
        } else if ((map1 != null) && (map2 == null)) {
            resMap = map1;
        } else if ((map1 != null) && (map2 != null)) {
            for (int i = 0; i < map2.size(); i++) {
                MapPair mp2 = (MapPair) map2.get(i);

                for (int j = 0; j < map1.size(); j++) {
                    MapPair mp1 = (MapPair) map1.get(j);

                    // add the ones in map2 but not in map1 originally into map1
                    if (!mp1.r1.getLabel().equalsIgnoreCase(mp2.r1.getLabel()) &&
                            !mp1.r2.getLabel().equalsIgnoreCase(mp2.r2.getLabel())) {
                        map1.add(mp2);
                    }
                }

                resMap = map1;
            }
        }

        return resMap;
    }

    public static void main(String[] args) throws Exception {
        String debug = System.getProperty("debug");

        if ((debug != null) && (debug.equals("true"))) {
            debug_on = true;
        }

        //populate user defined function table	
        String fName = "sf";
        String fImport = "import skolem as UDF_skolem;\n";
        String fDef = "FUNCTION sf($str)\n" + "{ skolem($str)[1] }\n";
        udfImportTable.put(fName, fImport);
        udfDefTable.put(fName, fDef);
        fName = "firstStr";
        fImport = "import split as UDF_split;\n";
        fDef = "FUNCTION firstStr($str)\n" + "{ split(\" \",$str)[1] }\n";
        udfImportTable.put(fName, fImport);
        udfDefTable.put(fName, fDef);
        fName = "secondStr";
        fImport = "import split as UDF_split;\n";
        fDef = "FUNCTION secondStr($str)\n" + "{ split(\" \",$str)[2] }\n";
        udfImportTable.put(fName, fImport);
        udfDefTable.put(fName, fDef);

        if (debug_on) {
            System.err.println(" udfImportTable = " + udfImportTable +
                " udfDefTable = " + udfDefTable);
        }

        //SCIA scia = new SCIA();
        //String sourceSchema = "studyA.xsd";
        //String sourceSchema = "./schemas/eml.xsd";
        //String targetSchema = "/schemas/observations.xml";
        //String targetSchema = "./schemas/adn_record.xsd";
        //String targetSchema = "./schemas/studyB.xsd";
        sourceSchema = new String(SCIA_WORK_DIR + "/schemas/book2.dtd");
        targetSchema = new String(SCIA_WORK_DIR + "/schemas/book3.dtd");

        SchemaTree schemaTree1 = new SchemaTree(sourceSchema, "DTD");

        //SchemaTree schemaTree1 = new SchemaTree(sourceSchema, "XMLSchema", 
        //						gui); 
        //SchemaTree schemaTree1 = (new XMLSchemaImporter(sourceSchema)).tree;
        schemaJTree1 = new SchemaTree(schemaTree1);

        SchemaTree schemaTree2 = new SchemaTree(targetSchema, "DTD");

        //SchemaTree schemaTree2 = new SchemaTree(targetSchema, "XMLSchema"); 
        //SchemaTree schemaTree2 = (new XMLSchemaImporter(targetSchema)).tree;  
        schemaJTree2 = new SchemaTree(schemaTree2);

        // create a GUI
        MainUI gui = new MainUI();

        SCIA.mainUI = gui;
        schemaJTree1.mainUI = gui;
        schemaJTree2.mainUI = gui;
        gui.pack();
        gui.setVisible(true);

        //String view = mainFrame.generateViewForObsAndStudyA();
        //String view = mainFrame.generateViewForLTEROverGCE();
        //String seqOverBSMLView = mainFrame.generateViewForSeqAndBSML();
        //mainFrame.generateViewExampleForBooks();

        /*PrintWriter pw =
            new PrintWriter(new FileOutputStream("view.qlt"), true);
        pw.println(view);
        pw.flush();*/
    }

    public static void matchActors(SchemaTree sTr, SchemaTree tTr) {
        if (debug_on) {
            System.err.println("SCIA_WORK_DIR = " + SCIA_WORK_DIR);
        }

        if (debug_on) {
            System.err.println("LIB_JAR_DIR = " + LIB_JAR_DIR);
        }

        schemaJTree1 = sTr;
        schemaJTree2 = tTr;

        schemaJTree1.schemaType = "XMLSchema";
        schemaJTree2.schemaType = "XMLSchema";
        bothXMLSchemas = true;

        schemaJTree1.treeName = "sourceSchemaTree";
        schemaJTree2.schemaType = "targetSchemaTree";

        // create a GUI
        try {
            MainUI gui = new MainUI();
            SCIA.mainUI = gui;
            schemaJTree1.mainUI = gui;
            schemaJTree2.mainUI = gui;
            gui.pack();
            gui.setVisible(true);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
