/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.awt.Graphics;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.Icon;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.JViewport;
import javax.swing.tree.DefaultTreeCellRenderer;

import org.jdom.Document;
import org.jdom.Element;
import org.semanticweb.owl.model.OWLClass;
import org.semanticweb.owl.model.OWLException;
import org.w3c.dom.ls.LSParser;
import org.w3c.rdf.model.Model;
import org.w3c.rdf.model.ModelException;
import org.w3c.rdf.model.NodeFactory;
import org.w3c.rdf.model.Resource;
import org.w3c.rdf.util.RDFFactory;
import org.w3c.rdf.util.RDFFactoryImpl;
import org.xmlmiddleware.schemas.dtds.ElementType;
import org.xmlmiddleware.schemas.dtds.Group;
import org.xmlmiddleware.schemas.dtds.Particle;
import org.xmlmiddleware.schemas.dtds.Reference;

import com.interdataworking.mm.alg.StringMatcher;


/**
 * The SCIA internal tree representation of schemas. The SchemaTree holds all
 * schema elements (attributes and elements in XML DTD and Schemas and fields
 * in relational schemas), represent subelements and attributes as children
 * nodes. It also provides methods for manipulating the tree. Each node
 * @see TNode holds the schema element information including name, data
 * type, node type (parentElem, leafElem, attribute), description, parent,
 * children vector cardinality and matching results.
 *
 * @author Guilian Wang
 */
public class SchemaTree extends JTree implements AbstractSchema {
    public static int currentRowNum = 0;
    public static boolean debug_on = SCIA.debug_on;

    // for use of XML Schema APIs

    /** Default namespaces support (true). */
    protected static final boolean DEFAULT_NAMESPACES = true;

    /** Default validation support (false). */
    protected static final boolean DEFAULT_VALIDATION = false;

    /** Default Schema validation support (false). */
    protected static final boolean DEFAULT_SCHEMA_VALIDATION = false;
    static LSParser builder;

    //create RDFFactory and NodeFactory
    public static RDFFactory rf = new RDFFactoryImpl();
    public static NodeFactory nf = rf.getNodeFactory();

    //create nodes for node types
    public static Resource rootElem;
    public static Resource parentElem;
    public static Resource leafElem;
    public static Resource attr;

    //create nodes for data types
    public static Resource CDATA;
    public static Resource ID;
    public static Resource PCDATA;
    public static Resource STRING;
    public static Resource INT;
    public static Resource FLOAT;
    public static Resource DATE;
    public static Resource NUMBER;

    //create edge labels
    protected static final short MAX_HEIGHT = 20;
    public static Resource[] parent;
    public static Resource[] child;
    public static Resource dataType;
    public static Resource nodeType;

    // messages
    private static String ALLOCATE = " - Allocating]\n";
    private static String TREE = "[Tree ";
    private static String COST_INSERT = " - Cost Increment: Inserting ";
    public String schemaType;
    public Type topType;
    public TNode rootNode;
    public String treeName;
    public long occupancy;

    //public static boolean interactive_on = true;
    // need the tester so that we can access it from TNodes
    public MainUI mainUI;
    public boolean inSmallWindow = false;

    /**
     * The constructor. It is to create an empty SchemaTree.
     */
    public SchemaTree() {
    }

    /**
     * The constructor. It is to create a new SchemaTree and initialize
     * its name and common RDF resources for further graph matching from an
     * existing schema tree in order to follow the JTree model (create tree
     * from root).
     */
    public SchemaTree(SchemaTree tree) {
        super(new SchemaTreeModel(tree.rootNode));
        rootNode = tree.rootNode;
        updateTreeField();

        schemaType = tree.schemaType;

        //initialize name 
        treeName = tree.treeName;

        // get occupancy
        occupancy = tree.occupancy;

        try {
            // initialize common RDF resources for graph matching 
            rootElem = nf.createResource("rootElem");
            parentElem = nf.createResource("parentElem");
            leafElem = nf.createResource("leafElem");
            attr = nf.createResource("attribute");

            //create nodes for data types
            STRING = nf.createResource("STRING");
            INT = nf.createResource("INT");
            DATE = nf.createResource("DATE");
            CDATA = nf.createResource("CDATA");
            ID = nf.createResource("ID");
            FLOAT = nf.createResource("FLOAT");
            PCDATA = nf.createResource("PCDATA");
            NUMBER = nf.createResource("NUMBER");

            //create edge labels
            parent = new Resource[MAX_HEIGHT];
            child = new Resource[MAX_HEIGHT];

            for (int i = 0; i < MAX_HEIGHT; i++) {
                parent[i] = nf.createResource("lev" + i + "Parent");
                child[i] = nf.createResource("lev" + i + "Child");
            }

            dataType = nf.createResource("dataType");
            nodeType = nf.createResource("nodeType");
        } catch (ModelException me) {
        }

        if (debug_on) {
            System.err.print(TREE + treeName + ALLOCATE);
        }

        /*
        getSelectionModel().setSelectionMode
            (TreeSelectionModel.SINGLE_TREE_SELECTION);
        getSelectionModel().setSelectionMode
            (TreeSelectionModel.CONTIGUOUS_TREE_SELECTION);
        */
        DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();

        //renderer.setText("XYZ");
        Icon personIcon = null;
        renderer.setLeafIcon(personIcon);
        renderer.setClosedIcon(personIcon);
        renderer.setOpenIcon(personIcon);
        setCellRenderer(renderer);
    }

    /**
     * The constructor. It is to create a SchemaTree and initialize
     * its name from an input schema file.
     *
     * @param inputFile The name of the input file for a schema (a dtd file, an
     * XML Schema file or an OWL file).
     * @param scheType The type of the input schema file (DTD, XMLSchema or
     * OWL).
     */
    public SchemaTree(String inputFile, String scheType) {
        schemaType = scheType;

        //initialize tree name 
        int j = inputFile.lastIndexOf(".");
        int i = inputFile.lastIndexOf("\\");

        if (i == -1) {
            i = inputFile.lastIndexOf("/");
        }

        treeName = inputFile.substring(i + 1, j);

        //initialize currentRowNum for a new SchemaTree
        currentRowNum = 0;

        if (schemaType.equalsIgnoreCase("DTD")) {
            // parse dtd and populate the root node and other nodes
            MyDTDParser dtdParser = new MyDTDParser(inputFile);
            rootNode = insertRoot(dtdParser.schemaRootTag);

            Hashtable cardiInfo = new Hashtable();

            if (dtdParser.schemaRootElem == null) {
                if (debug_on) {
                    System.err.println("schem root elem is null");
                }
            }

            /* record cardinality info of all elements except the root in a
               hashtable
            */
            setChildCardinality(dtdParser.schemaRootElem.content, cardiInfo);

            insertChildren(dtdParser.elemHash, dtdParser.attrHash, rootNode,
                "", cardiInfo);
        } else if (schemaType.equalsIgnoreCase("XMLSchema")) {
            XMLSchemaParser xmlSchemaParser = new XMLSchemaParser(inputFile);
            rootNode = insertRoot(xmlSchemaParser.schemaRootTag);

            // hash the type definitions directly as child of the doc root
            Hashtable rootChildrenTypeHash = new Hashtable();

            if (xmlSchemaParser.schemaRootElem == null) {
                if (debug_on) {
                    System.err.println("schemaRootElem == null");
                }
            }

            insertChildren(xmlSchemaParser.eleAttrHash, rootNode,
                xmlSchemaParser.schemaRootElem, xmlSchemaParser.schemaRootTag,
                true, rootChildrenTypeHash);

            // added 3/20/2005
            //rootNode = importXMLSchema(inputFile);
        } else if (schemaType.equalsIgnoreCase("Ontology")) {
            MyOntologyParser ontoParser = new MyOntologyParser();
            ontoParser.parse(inputFile);

            rootNode = insertRoot(ontoParser.ontologyRootTag);

            insertChildren(ontoParser.subClassHash, rootNode, null, //root class
                true, ontoParser.rootKey, "");
        }

        // set row number for all nodes in this tree
        setRowNum();
    }

    public Type getTypeFromTag(String tag) {
        if (topType != null) {
            return topType.getTypeFromTagForAll(tag);
        }

        return null;
    }

    public List listOf(Type elem) {
        List res = new ArrayList();

        // construct a real list, or just type: 
        // list of Type elem not something else?
        return res;
    }

    public Set setOf(Type elem) {
        Set res = new HashSet();

        return res;
    }

    public TupleList product(List l) {
        TupleList res = new TupleList();

        return res;
    }

    public String append(String str1, String str2) {
        String res = new String();

        return res;
    }

    public List cons(Type elem, List l) {
        List res = new ArrayList();

        return res;
    }

    public Type head(List l) {
        Type res = new Type("", "");

        return res;
    }

    public List tail(List l) {
        List res = new ArrayList();

        return res;
    }

    public void paintComponent(Graphics g) {
        super.paintComponent(g);

        if (!inSmallWindow) {
            JScrollPane spane = (JScrollPane) ((JViewport) getParent()).getParent();
            MyJSplitPane treesPane = (MyJSplitPane) spane.getParent();

            treesPane.drawMapping(treesPane.getGraphics());
        }

        /*        else {
            JComponent treesPane = (JComponent)getParent();

            treesPane.drawMapping(treesPane.getGraphics());
            }

        if(debug_on) System.err.println("###overwritten JTree.paintComponent is called####");
        */
    }

    public void changeRoot(TNode newRoot) {
        ((SchemaTreeModel) getModel()).changeRoot(newRoot);
    }

    /**
     * Refine the matches of schema nodes at a specific level.
     *
     * @param level The level at which the nodes to be refined.
     * @param anotherTree The SchemaTree to be matched with.
     * @param nameSimTable The Hashtable that stores the computed similarity
     * for any pair of a node of this SchemaTree and a node of anotherTree.
     * @param pathSimTable The Hashtable that stores the computed similarity
     * for any pair of path of this SchemaTree and a path of anotherTree.
     * @param skipLeaf The boolean value indicating skipping leaf nodes or not,
     * if true, don't refine leaf nodes.
     * @interactive_on The boolean value indicating whether user interaction is
     * expected.
     */
    public void adjustMatchWithCoreLevel(long level, SchemaTree anotherTree,
        Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean skipLeaf, boolean interactive_on) {
        //get all TNodes at height = level
        Vector tnodesAtCoreLevel = new Vector();

        getTNodesAtLevel(level, tnodesAtCoreLevel, rootNode);

        if (!tnodesAtCoreLevel.isEmpty()) {
            for (int i = 0; i < tnodesAtCoreLevel.size(); i++) {
                TNode thisTNode = (TNode) tnodesAtCoreLevel.get(i);

                if (thisTNode != null) {
                    if (!skipLeaf ||
                            ((thisTNode.children != null) &&
                            !thisTNode.children.isEmpty())) {
                        try {
                            thisTNode.adjustMatchWithCoreLevel(anotherTree,
                                nameSimTable, pathSimTable, descSimTable,
                                dataTypeSimTable, interactive_on);
                        } catch (Exception e) {
                            if (debug_on) {
                                System.err.println("Exception from TNode." +
                                    "adjustMatchWithCoreLevel: ");
                            }

                            e.printStackTrace();
                            System.exit(0);
                        }
                    }
                }
            }
        }
    }

    public void setRealParentChildren() {
        if ((rootNode != null) && (rootNode.children != null) &&
                !rootNode.children.isEmpty()) {
            for (int i = 0; i < rootNode.children.size(); i++) {
                TNode child = (TNode) rootNode.children.get(i);
                child.setRealParentChildrenForAll(rootNode);
            }
        }
    }

    public SchemaTree removeCompositorNodes() {
        SchemaTree tr = new SchemaTree();

        if (rootNode != null) {
            tr.rootNode = new TNode(rootNode, null);
            rootNode.removeCompositorNodesInSubTree(tr.rootNode);
        }

        return tr;
    }

    public long[] getCoreLevels() {
        long level = 0;

        // total number of children for all tnodes at a level
        long numChildren = 0;

        // the average number of children for all the tnodes at a level 
        float avgNumChildren = 0;
        float max = 0;
        float secMax = 0;

        // the most important level where the main aspects of the entity 
        // that this schema is describing are located
        long mostCoreLevel = 0;

        // the second most important level
        long moreCoreLevel = 0;

        do {
            Vector tnodesAtThisLevel = new Vector();
            getTNodesAtLevel(level, tnodesAtThisLevel, rootNode);

            if (!tnodesAtThisLevel.isEmpty()) {
                for (int i = 0; i < tnodesAtThisLevel.size(); i++) {
                    TNode tnode = (TNode) tnodesAtThisLevel.get(i);

                    if (tnode.children != null) {
                        numChildren = numChildren + tnode.children.size();
                    }
                }

                avgNumChildren = numChildren / (tnodesAtThisLevel.size());

                if (debug_on) {
                    System.err.println("average # of children at level " +
                        level + " = " + avgNumChildren);
                }

                if (avgNumChildren > max) {
                    secMax = max;
                    moreCoreLevel = mostCoreLevel;
                    max = avgNumChildren;
                    mostCoreLevel = level + 1;
                    
                } else if (avgNumChildren > secMax) {
                    secMax = avgNumChildren;
                    moreCoreLevel = level + 1;
                }

                if (debug_on) {
                    System.err.println("level = " + level + " secMax = " +
                        secMax + " moreCoreLevel = " + moreCoreLevel);
                }

                level++;
                numChildren = 0;
            } else {
                break;
            }
        } while (true);

        long[] coreLevels = new long[2];

        
        if (secMax >= 2) {
            coreLevels[0] = mostCoreLevel;
            coreLevels[1] = moreCoreLevel;
        } else {
            coreLevels[0] = mostCoreLevel;
            coreLevels[1] = -1;
        }
        

        // make sure context check top-down
        /*
        if (mostCoreLevel > moreCoreLevel) {            
            coreLevels[0] = moreCoreLevel;
            coreLevels[1] = mostCoreLevel;
        } else {
            coreLevels[1] = moreCoreLevel;
            coreLevels[0] = mostCoreLevel;
        }
        */
        return coreLevels;
    }

    public void getLeaves(Vector leaves, TNode working) {
        if ((working.children != null) && !working.children.isEmpty()) {
            for (int i = 0; i < working.children.size(); i++) {
                getLeaves(leaves, (TNode) working.children.get(i));
            }
        } else {
            leaves.addElement(working);
        }
    }

    public void getTNodesAtLevel(long level, Vector tnodesAtLevel, TNode working) {
        if ((working.height < level) && (working.children != null) &&
                !working.children.isEmpty()) {
            for (int i = 0; i < working.children.size(); i++) {
                getTNodesAtLevel(level, tnodesAtLevel,
                    (TNode) working.children.get(i));
            }
        } else if (working.height == level) {
            tnodesAtLevel.addElement(working);
        }
    }

    /**
     * Create and insert the root node into a SchemaTree.
     * @param rootTag The name of the root node of the schema tree.
     *
     * @return a TNode
     * @see TNode, the root node of the SchemaTree.
     */
    public TNode insertRoot(String rootTag) {
        //increase the tree occupancy
        occupancy++;

        String index = "0";
        SchemaNode root = new SchemaNode(rootTag, "rootElem", "");
        rootNode = new TNode(root);
        rootNode.tree = this;

        if (debug_on) {
            System.err.print(TREE + treeName + COST_INSERT + rootTag + "]\n");
        }

        return rootNode;
    }

    /**
     * Create and insert the subelements in elemHash and the attributes in the
     * input attHash into the tree as children of the parentNode, update the
     * height for each affected node and assign index label for each node.
     * This is for DTD.
     *
     * @param elemHash The hashtable that holds the subelements for all elements
     * @param attrHash The hashtable that holds the attributes for all elements
     * @param parentNode The TNode to wich the children nodes to be attached
     * @param index The index string assigned to parentNode, this iss for
     * connecting with DDXMI system that assumes
     * manual match specification by indexing the same meaning nodes with the
     * same index in DDXMI
     *
     * @return true if succeeds, otherwise false.
     */
    public boolean insertChildren(Hashtable elemHash, Hashtable attrHash,
        TNode parentNode, String index, Hashtable cardinalityHash) {
        // index offset 
        int offset = 0;

        // handle attributes
        if (attrHash != null) {
            List attrs = (List) attrHash.get(parentNode.data.getTag());

            if ((attrs != null) && !attrs.isEmpty() &&
                    (parentNode.data.nodeType.equals("leafElem") ||
                    parentNode.data.nodeType.equals(""))) {
                parentNode.data.nodeType = "parentElem";

                // add each attribute as a child node to the parentNode
                for (int i = 0; i < attrs.size(); i++) {
                    org.xmlmiddleware.schemas.dtds.Attribute attr = (org.xmlmiddleware.schemas.dtds.Attribute) attrs.get(i);
                    String attrTag = attr.name.getLocalName();

                    if (attr != null) {
                        offset++;

                        String indexLabel = new String();

                        if (index.length() == 0) {
                            indexLabel = index + offset;
                        } else {
                            indexLabel = index + "." + offset;
                        }

                        String dataType = null;

                        if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_CDATA) {
                            dataType = "CDATA";
                        } else if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_ID) {
                            dataType = "ID";
                        } else if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_IDREF) {
                            dataType = "IDREF";
                        }

                        String minOc = "0";
                        String maxOc = "1";

                        if ((attr.required == org.xmlmiddleware.schemas.dtds.Attribute.REQUIRED_FIXED) ||
                                (attr.required == org.xmlmiddleware.schemas.dtds.Attribute.REQUIRED_REQUIRED)) {
                            minOc = "1";
                        }

                        if ((attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_IDREFS) ||
                                (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_ENTITIES) ||
                                (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_NMTOKENS)) {
                            maxOc = "unbounded";
                        }

                        SchemaNode attrSchemaNode = new SchemaNode(attrTag,
                                "attribute", dataType, minOc, maxOc);

                        TNode attrTNode = new TNode(attrSchemaNode, parentNode);

                        occupancy++;

                        if (debug_on) {
                            System.err.print(TREE + treeName + COST_INSERT +
                                attr + "]\n");
                        }
                    }
                }
            }
        }

        // handle elements
        List childElements = (List) elemHash.get(parentNode.data.getTag());

        if ((childElements != null) && (!childElements.isEmpty())) {
            if (parentNode.data.nodeType.equals("leafElem") ||
                    parentNode.data.nodeType.equals("")) {
                parentNode.data.nodeType = "parentElem";
            }

            for (int i = 0; i < childElements.size(); i++) {
                ElementType childET = (ElementType) childElements.get(i);

                if (childET != null) {
                    String childElemTag = childET.name.getLocalName();
                    offset++;

                    String indexLabel = new String();

                    if (index.length() == 0) {
                        indexLabel = index + offset;
                    } else {
                        indexLabel = index + "." + offset;
                    }

                    String dataType = null;
                    String minOc = null;
                    String maxOc = null;

                    /*
                    switch (childET.contentType) {
                    case ElementType.CONTENT_ANY:
                    case ElementType.CONTENT_MIXED:
                        throw new Exception("Can't process element types with " +
                                            "mixed or ANY content: " +
                                            childET.name.getUniversalName());

                    case ElementType.CONTENT_ELEMENT:
                        if (childET.content.isRepeatable) {
                           ????   maxOc = "unbounded";
                        }

                        Cardinality card = (Cardinality)cardinalityHash.
                            get(childET.name.getLocalName());
                        if (card != null) {
                            minOc = card.minOccurs;
                            maxOc = card.maxOccurs;
                        }
                        if(debug_on) System.err.println("in insertchildren(), " +
                                           "case CONTENT_ELEMENT: from hash " +
                                           childET.name.getLocalName() +
                                           " minOc = " + minOc +
                                           " maxOc = " + maxOc);
                        break;

                    case ElementType.CONTENT_PCDATA:
                        dataType = "PCDATA";
                        break;
                    case ElementType.CONTENT_EMPTY:
                        No content to process.
                        break;
                    }
                    */
                    if (childET.contentType == ElementType.CONTENT_PCDATA) {
                        dataType = "PCDATA";
                    }

                    if ((childET.contentType == ElementType.CONTENT_PCDATA) ||
                            (childET.contentType == ElementType.CONTENT_ELEMENT)) {
                        Cardinality card = (Cardinality) cardinalityHash.get(childET.name.getLocalName());

                        if (card != null) {
                            minOc = card.minOccurs;
                            maxOc = card.maxOccurs;
                        }

                        //if(debug_on) System.err.println("in insertchildren(), " + 
                        //	   "case PCDATA ||_ELEMENT: from hash " +
                        //	   childET.name.getLocalName() +
                        //	   " minOc = " + minOc +
                        //	   " maxOc = " + maxOc + "\n");
                    }

                    //if(debug_on) System.err.println("In insertChildren(): deal with " +
                    //		      childET.name.getLocalName() +
                    //	       " minOc = " + minOc +
                    //	       " maxOc = " + maxOc);
                    SchemaNode childSchemaNode = new SchemaNode(childElemTag,
                            "", dataType, minOc, maxOc);

                    //PrintWriter cout = new PrintWriter (if(debug_on) System.err, true);
                    //childSchemaNode.write(cout, "  ");
                    //cout.println();
                    //cout.flush();
                    TNode childTNode = new TNode(childSchemaNode, parentNode);

                    occupancy++;

                    if (debug_on) {
                        System.err.print(TREE + treeName + COST_INSERT +
                            childElemTag + "]\n");
                    }

                    // insert this child's chidren
                    insertChildren(elemHash, attrHash, childTNode, indexLabel,
                        cardinalityHash);
                }
            }
        } else {
            if (!parentNode.data.nodeType.equals("attribute")) {
                parentNode.data.nodeType = "leafElem";
            }
        }

        return true;
    }

    /**
     * Define the input elementType et's type in abstract schema using its
     * child elements in the input elemHash and attributes in the input attHash.
     * This is for DTD.
     */
    public void defineTypes(Hashtable elemHash, Hashtable attrHash,
        ElementType et, Type type, // the type of et in Abstract Schema
        boolean isHandlingSchemaRoot, Hashtable cardinalityHash) {
        String tag = et.name.getLocalName(); // tag of input ElementType et

        // initialize the type for input elem if not initialized yet
        if (type == null) {
            type = new Type(tag);
        }

        // handle attributes
        if (attrHash != null) {
            List attrs = (List) attrHash.get(tag);

            if ((attrs != null) && !attrs.isEmpty()) {
                // add each attribute as a type component of the parentType
                for (int i = 0; i < attrs.size(); i++) {
                    org.xmlmiddleware.schemas.dtds.Attribute attr = (org.xmlmiddleware.schemas.dtds.Attribute) attrs.get(i);
                    String attrTag = attr.name.getLocalName();

                    if (attr != null) {
                        String dataType = null;

                        if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_CDATA) {
                            dataType = "CDATA";
                        } else if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_ID) {
                            dataType = "ID";
                        } else if (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_IDREF) {
                            dataType = "IDREF";
                        }

                        String minOc = "0";
                        String maxOc = "1";

                        if ((attr.required == org.xmlmiddleware.schemas.dtds.Attribute.REQUIRED_FIXED) ||
                                (attr.required == org.xmlmiddleware.schemas.dtds.Attribute.REQUIRED_REQUIRED)) {
                            minOc = "1";
                        }

                        if ((attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_IDREFS) ||
                                (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_ENTITIES) ||
                                (attr.type == org.xmlmiddleware.schemas.dtds.Attribute.TYPE_NMTOKENS)) {
                            maxOc = "unbounded";
                        }

                        Type attrType = new Type(attrTag, true, dataType,
                                minOc, maxOc);

                        // add this new attrType as one type component of the 
                        // current element type 
                        if (type.components == null) {
                            type.components = new ArrayList();
                        }

                        type.components.add(attrType);
                    }
                }
            }
        }

        // handle elements
        List childElements = (List) elemHash.get(tag);

        if ((childElements != null) && (!childElements.isEmpty())) {
            for (int i = 0; i < childElements.size(); i++) {
                ElementType childET = (ElementType) childElements.get(i);

                if (childET != null) {
                    String childElemTag = childET.name.getLocalName();
                    String dataType = null;
                    String minOc = null;
                    String maxOc = null;

                    if (childET.contentType == ElementType.CONTENT_PCDATA) {
                        dataType = "PCDATA";
                    }

                    if ((childET.contentType == ElementType.CONTENT_PCDATA) ||
                            (childET.contentType == ElementType.CONTENT_ELEMENT)) {
                        Cardinality card = (Cardinality) cardinalityHash.get(childET.name.getLocalName());

                        if (card != null) {
                            minOc = card.minOccurs;
                            maxOc = card.maxOccurs;
                        }

                        //if(debug_on) System.err.println("in insertchildren(), " + 
                        //	   "case PCDATA ||_ELEMENT: from hash " +
                        //	   childET.name.getLocalName() +
                        //	   " minOc = " + minOc +
                        //	   " maxOc = " + maxOc + "\n");
                    }

                    //if(debug_on) System.err.println("In insertChildren(): deal with " +
                    //		      childET.name.getLocalName() +
                    //	       " minOc = " + minOc +
                    //	       " maxOc = " + maxOc);
                    Type childType = new Type(childElemTag, false, dataType,
                            minOc, maxOc);

                    //PrintWriter cout = new PrintWriter (if(debug_on) System.err, true);
                    //childSchemaNode.write(cout, "  ");
                    //cout.println();
                    //cout.flush();
                    //attach this new childType as one child of the current type
                    if (type.components == null) {
                        type.components = new ArrayList();
                    }

                    type.components.add(childType);

                    occupancy++;

                    if (debug_on) {
                        System.err.print(TREE + treeName + COST_INSERT +
                            childElemTag + "]\n");
                    }

                    // handle this child's chidren
                    if ((childType.typeType != null) &&
                            childType.typeType.equals("List") &&
                            (childType.element != null)) {
                        /*update list type's element, instead of the type itself
                           e.g., s(bib) = Book,
                                 s(Book) = ListOf(book),
                                 s(book) = title x year x Author,
                                 s(Author) = ListOf(author),
                                 s(author) = first x last,
                                 s(first) = String,
                                 s(last) = String
                        */
                        childType = childType.element;
                    }

                    defineTypes(elemHash, attrHash, childET, childType, false,
                        cardinalityHash);
                }
            }
        } else { // no child elements

            if (type.components != null) { // has child elements or attributes

                //how about the case of having a single non-repeatable 
                //attribute or element
            }
        }

        if (type.components != null) {
            type.typeType = "Product";
        }

        if (isHandlingSchemaRoot) {
            topType = type; //  set topType
        }
    }

    /**
     * Define the input element elem's type in abstract schema using its child
     * elements and attributes in the input elemAttrHash. Determine topType.
     * This is for XML Schema.
     */
    public void defineTypes(Hashtable elemAttrHash, Element elem,
        String schemaRootTag, 
    // the type of input elem in Abstract Schema
    Type type, boolean isHandlingRoot, Hashtable docRootChildTypeHash) {
        List children = new ArrayList();
        String key;
        String elemType;
        String tag = elem.getAttributeValue("name");

        // initialize the type for input elem in Abstract Schema 
        // if not initialized yet
        if (type == null) {
            type = new Type(tag);
        }

        if (isHandlingRoot) {
            if (debug_on) {
                System.err.println("is handling root");
            }

            // assume schema root element type defined at "schema name + Type"
            key = schemaRootTag + "." + schemaRootTag + "Type";

            children = (List) elemAttrHash.get(key);

            // hash the type definition as direct child of the doc root
            if (elem == null) {
                if (debug_on) {
                    System.err.println("elem = null");
                }
            }

            Document doc = elem.getDocument();

            if (doc != null) {
                if (debug_on) {
                    System.err.println("doc = " + doc);
                }

                Element docRoot = doc.getRootElement();

                if (docRoot != null) {
                    List docRootChildren = docRoot.getChildren();
                    Iterator iterator = docRootChildren.iterator();

                    while (iterator.hasNext()) {
                        Element docRootChild = (Element) iterator.next();
                        String localName = docRootChild.getName();

                        if (localName.equalsIgnoreCase("ElementType") ||
                                localName.equalsIgnoreCase("AttributeType")) {
                            tag = docRootChild.getAttributeValue("name");
                            elemType = docRootChild.getAttributeValue("type");

                            if ((tag != null) && (elemType != null)) {
                                docRootChildTypeHash.put(tag, elemType);
                            }
                        }
                    }
                } else {
                    if (debug_on) {
                        System.err.println("docRoot == null");
                    }
                }
            } else {
                if (debug_on) {
                    System.err.println("doc == null");
                }
            }
        } else { // not handling root
            tag = elem.getAttributeValue("name"); // tag of input Element elem
            elemType = elem.getAttributeValue("type");

            if (tag != null) {
                key = tag;

                Element ancestor = (Element) elem.getParent();

                if (ancestor != null) {
                    String aName = ancestor.getAttributeValue("name");

                    while (aName == null) {
                        ancestor = (Element) ancestor.getParent();
                        aName = ancestor.getAttributeValue("name");
                    }

                    key = aName + "." + tag;
                }

                //if(debug_on) System.err.println("key = " + key);
                children = (List) elemAttrHash.get(key);
            }

            if ((children == null) || children.isEmpty()) {
                // no direct children, go to type definition to get children
                key = schemaRootTag + "." + elemType;
                children = (List) elemAttrHash.get(key);
            }
        }

        if (debug_on) {
            System.err.println("schemaRootTag = " + schemaRootTag +
                " Dealing with " + tag + " children = " + children);
        }

        boolean isRealParentElem = false;

        if ((children != null) && !children.isEmpty()) {
            // add  each child type in input elem's type components
            Iterator iterator = children.iterator();
            Hashtable typeHash = new Hashtable();

            while (iterator.hasNext()) {
                Element child = (Element) iterator.next();
                String childTag = child.getAttributeValue("name");
                String childLocalName = child.getName();
                String childDataType = child.getAttributeValue("type");

                //if(debug_on) System.err.println("localName= " + childLocalName);
                //filter out any child that does not have tag and does not have
                // type, i.e., description
                if ((childTag != null) || (childDataType != null)) {
                    isRealParentElem = true;

                    //if (childDataType == null) {
                    //	childDataType = "";
                    //
                    // hash type definition
                    if ((childLocalName.equalsIgnoreCase("AttributeType") ||
                            childLocalName.equalsIgnoreCase("ElementType")) &&
                            (childTag != null)) {
                        typeHash.put(childTag, childDataType);
                    }

                    // get tag for child element or attribute
                    if ((childTag == null) &&
                            (child.getName().equalsIgnoreCase("Element") ||
                            child.getName().equalsIgnoreCase("Attribute"))) {
                        childTag = childDataType;
                    }

                    //get cardinality information
                    String minOccur = child.getAttributeValue("minOccurs");
                    String maxOccur = child.getAttributeValue("maxOccurs");

                    /* if original parent in XML Schema is "Sequence" which is
                      skipped when hashing the parent-chilren relationship.
                      The following is ok with a sequence of single element,
                      but there might be a sequence of multiple elements.
                      E.g., <Sequence>
                               <Element time>
                               <Element location>
                               <Element Species>
                               <Element Count>
                            </Sequence>
                      This is the same as (time, location, species, count)*
                      in DTD
                    */
                    Element parent = (Element) child.getParent();

                    if (parent.getName().equalsIgnoreCase("Sequence")) {
                        if ((parent.getAttributeValue("minOccurs") != null) ||
                                (parent.getAttributeValue("maxOccurs") != null)) {
                            minOccur = parent.getAttributeValue("minOccurs");
                            maxOccur = parent.getAttributeValue("minOccurs");
                        } else {
                            minOccur = "0";
                            maxOccur = "*";
                        }
                    }

                    // get child element or attribute data type if it is atomic
                    // check whether it is user defined data type
                    // if so, get its real data type
                    // assume it is defined locally or as a child of the schema
                    if ((childDataType != null) &&
                            (child.getAttributeValue("name") == null) &&
                            (childLocalName.equalsIgnoreCase("Attribute") ||
                            childLocalName.equalsIgnoreCase("Element"))) {
                        // check whether defined locally as AttributeType
                        String tempType = (String) typeHash.get(childDataType);

                        if (tempType != null) {
                            childDataType = tempType;
                        } else { // not defined locally, check from the doc root

                            if (debug_on) {
                                System.err.println(
                                    "type is not defined locally");
                            }

                            tempType = (String) docRootChildTypeHash.get(childDataType);

                            if (tempType != null) {
                                childDataType = tempType;
                            }
                        }
                    }

                    if (!(child.getName().equalsIgnoreCase("AttributeType")) &&
                            !(child.getName().equalsIgnoreCase("ElementType"))) {
                        boolean isAttr = child.getName().equalsIgnoreCase("Attribute");
                        Type childType = new Type(childTag, isAttr,
                                childDataType, minOccur, maxOccur);

                        // attach this new childType as one child of the 
                        // current type 
                        if (type.components == null) {
                            type.components = new ArrayList();
                        }

                        type.components.add(childType);

                        occupancy++;

                        // handle this child's chidren 
                        if (childType.typeType.equals("List") &&
                                (childType.element != null)) {
                            /* update list type's element, instead of the type
                               itself
                               e.g., s(bib) = Book,
                               s(Book) = ListOf(book),
                               s(book) = title x year x Author,
                               s(Author) = ListOf(author),
                               s(author) = first x last,
                               s(first) = String,
                               s(last) = String
                            */
                            childType = childType.element;
                        }

                        defineTypes(elemAttrHash, child, schemaRootTag,
                            childType, false, docRootChildTypeHash);
                    }
                }
            }
        } else { // no child elements

            if (type.components != null) { // has child elements or attributes

                /*how about the case of having a single non-repeatable attribute
                  or element */
            }
        }

        if (type.components != null) {
            type.typeType = "Product";
        }

        if (isHandlingRoot) {
            topType = type; //  set topType
        }
    }

    /**
      * Create and insert the subelements and attributes in the
      * input hashtable elemAttrHash into the tree as children of the parent
      * element, update the
      * height for each affected node and assign index label for each node.
      * This is for creating schema trees from XML Schema files, named as xxx.xml
      *  in SCIA_WORK_DIR/schemas/, not for those named as xxx.xsd.
      *
      * @param elemHash The hashtable that holds the subelements and attributes
      * for all elements
      * @param parent The parent Element to wich the children nodes belong
      * @param  SchemaRootTag The name of the schema root.
      * @param isHandlingRoot The boolean value indicating whether working on
      * the schema tree root or not
      * @param docRootChildTypeHash The hashtable that holds the type information
      * of the document root children.
      *
      * @return true if succeeds, otherwise false.
      */
    public boolean insertChildren(Hashtable elemAttrHash, TNode parentNode,
        Element parent, String schemaRootTag, boolean isHandlingRoot,
        Hashtable docRootChildTypeHash) {
        List children = new ArrayList();
        String key;

        if (isHandlingRoot) {
            // assume schema root element type defined at "schema name + Type"
            key = schemaRootTag + "." + schemaRootTag + "Type";

            children = (List) elemAttrHash.get(key);

            // hash the type definition as direct child of the doc root
            if (parent == null) {
                if (debug_on) {
                    System.err.println("parent = null");
                }
            }

            Document doc = parent.getDocument();

            if (doc != null) {
                //if(debug_on) System.err.println("doc = " + doc);
                Element docRoot = doc.getRootElement();

                if (docRoot != null) {
                    List docRootChildren = docRoot.getChildren();
                    Iterator iterator = docRootChildren.iterator();

                    while (iterator.hasNext()) {
                        Element docRootChild = (Element) iterator.next();
                        String localName = docRootChild.getName();

                        if (localName.equalsIgnoreCase("ElementType") ||
                                localName.equalsIgnoreCase("AttributeType")) {
                            String tag = docRootChild.getAttributeValue("name");
                            String type = docRootChild.getAttributeValue("type");

                            if ((tag != null) && (type != null)) {
                                docRootChildTypeHash.put(tag, type);
                            }
                        }
                    }
                } else {
                    if (debug_on) {
                        System.err.println("doc == null");
                    }
                }
            } else {
                if (debug_on) {
                    System.err.println("doc == null");
                }
            }
        } else {
            String parentTag = parent.getAttributeValue("name");
            String parentType = parent.getAttributeValue("type");

            if (parentTag != null) {
                key = parentTag;

                Element ancestor = (Element) parent.getParent();

                if (ancestor != null) {
                    String aName = ancestor.getAttributeValue("name");

                    while (aName == null) {
                        ancestor = (Element) ancestor.getParent();
                        aName = ancestor.getAttributeValue("name");
                    }

                    key = aName + "." + parentTag;
                }

                //if(debug_on) System.err.println("key = " + key);
                children = (List) elemAttrHash.get(key);
            }

            if ((children == null) || children.isEmpty()) {
                // no direct children, go to type definition to get children
                key = schemaRootTag + "." + parentType;
                children = (List) elemAttrHash.get(key);
            }
        }

        boolean isRealParentElem = false;

        if ((children != null) && !children.isEmpty()) {
            // insert each child as parentNode's child node
            Iterator iterator = children.iterator();
            Hashtable typeHash = new Hashtable();

            while (iterator.hasNext()) {
                Element child = (Element) iterator.next();
                String childTag = child.getAttributeValue("name");
                String childLocalName = child.getName();
                String childDataType = child.getAttributeValue("type");

                //if(debug_on) System.err.println("localName= " + childLocalName);
                //filter out any child that does not have tag and does not have 
                // type, i.e. description
                if ((childTag != null) || (childDataType != null)) {
                    isRealParentElem = true;

                    if (childDataType == null) {
                        childDataType = "";
                    }

                    // hash type definition
                    if ((childLocalName.equalsIgnoreCase("AttributeType") ||
                            childLocalName.equalsIgnoreCase("ElementType")) &&
                            (childTag != null)) {
                        typeHash.put(childTag, childDataType);
                    }

                    // get tag for child element or attribute
                    if ((childTag == null) &&
                            (child.getName().equalsIgnoreCase("Element") ||
                            child.getName().equalsIgnoreCase("Attribute"))) {
                        childTag = childDataType;
                    }

                    //get cardinality information
                    String minOccur = child.getAttributeValue("minOccurs");
                    String maxOccur = child.getAttributeValue("maxOccurs");

                    // get child element or attribute data type if it is atomic
                    // check whether it is user defined data type
                    // if so, get its real data type
                    // assume it is defined locally or as a child of the schema
                    if ((childDataType != null) &&
                            (child.getAttributeValue("name") == null) &&
                            (childLocalName.equalsIgnoreCase("Attribute") ||
                            childLocalName.equalsIgnoreCase("Element"))) {
                        // check whether defined locally as AttributeType
                        String tempType = (String) typeHash.get(childDataType);

                        if (tempType != null) {
                            childDataType = tempType;
                        } else { // not defined locally, check from the doc root

                            if (debug_on) {
                                System.err.println(
                                    "type is not defined locally");
                            }

                            tempType = (String) docRootChildTypeHash.get(childDataType);

                            if (tempType != null) {
                                childDataType = tempType;
                            }
                        }
                    }

                    String childNodeType = child.getName();

                    if (childNodeType.equalsIgnoreCase("Attribute")) {
                        childNodeType = "attribute";
                    } else if (childNodeType.equalsIgnoreCase("Element")) {
                        childNodeType = "parentElem";
                    }

                    if (!(child.getName().equalsIgnoreCase("AttributeType")) &&
                            !(child.getName().equalsIgnoreCase("ElementType"))) {
                        SchemaNode childSchemaNode = new SchemaNode(childTag,
                                childNodeType, childDataType, minOccur, maxOccur);
                        TNode childTNode = new TNode(childSchemaNode, parentNode);

                        occupancy++;

                        if (debug_on) {
                            System.err.print(TREE + treeName + COST_INSERT +
                                childTag + "]\n");
                        }

                        // insert this child's chidren
                        insertChildren(elemAttrHash, childTNode, child,
                            schemaRootTag, false, docRootChildTypeHash);
                    }
                }
            }
        }

        if (!isRealParentElem) { // no meaningful children

            if (!(parentNode.data.nodeType.equalsIgnoreCase("attribute"))) {
                parentNode.data.nodeType = "leafElem";
            }
        }

        return true;
    }

    /**
     * Insert the children of the SchemaNode in input class uri --> subclasses
     * hashtable update the height for each affected node and assign index
     * label for each node.
     * This is for OWL Ontology.
     * @param subClassHash The hashtable that holds the sub-classes for all
     * classes
     * @param parentNode The parent TNode to wich the children nodes belong
     * @param parentNode The parent class being handled
     * @param  rootUri The URI for owl:thing
     * @param isHandlingRoot The boolean value indicating whether working on
     * the schema tree root or not
     * @param index The index string assigned to the parent node, for connecting
     * with DDXMI system.
     *
     * @return true if succeeds, otherwise false
     */
    public boolean insertChildren(Hashtable subClassHash, TNode parentNode, //the current node
        OWLClass parentCls, //the current class
        boolean isHandlingRoot, String rootUri, // uri for owl:Thing
        String index) {
        // index offset 
        int offset = 0;
        Set children = new HashSet();
        String key = null;

        if (isHandlingRoot) {
            if (debug_on) {
                System.err.println("is handling root");
            }

            // assumed that ontology root is owl:Thing
            key = rootUri;
        } else {
            try {
                key = ((URI) parentCls.getURI()).toString();
            } catch (OWLException e) {
                e.printStackTrace();
            }
        }

        if (key != null) {
            children = (Set) subClassHash.get(key);
        }

        boolean isRealParentElem = false;

        if ((children != null) && !children.isEmpty()) {
            // insert each child as parentNode's child node
            Iterator iterator = children.iterator();

            while (iterator.hasNext()) {
                OWLClass child = (OWLClass) iterator.next();
                String childTag = null;

                try {
                    childTag = ((URI) child.getURI()).getFragment();
                } catch (OWLException e) {
                    e.printStackTrace();
                }

                String childDataType = new String();

                if (childTag != null) {
                    isRealParentElem = true;

                    if (childDataType == null) {
                        childDataType = "";
                    }

                    //TODO get cardinality information, only for property, not class
                    String childNodeType = "parentElem";
                    String minOccur = new String();
                    String maxOccur = new String();

                    offset++;

                    String indexLabel = new String();

                    if (index.length() == 0) {
                        indexLabel = index + offset;
                    } else {
                        indexLabel = index + "." + offset;
                    }

                    SchemaNode childSchemaNode = new SchemaNode(childTag,
                            childNodeType, childDataType, minOccur, maxOccur);
                    TNode childTNode = new TNode(childSchemaNode, parentNode);

                    occupancy++;

                    if (debug_on) {
                        System.err.print(TREE + treeName + COST_INSERT +
                            childTag + "]\n");
                    }

                    // insert this child class' subclasses
                    insertChildren(subClassHash, childTNode, child, false,
                        rootUri, indexLabel);
                }
            }
        }

        if (!isRealParentElem) { // no meaningful children
            parentNode.data.nodeType = "leafElem";
        }

        return true;
    }

    /**
     * Compute the linguistic matchings for all nodes in the tree.
     *
     * @param working1 The working TNode in another tree
     * @param nameSimTable The Hashtable holding the similarity values between
     * any pair of one node in this tree and one node in another tree
     * @param The Hashtable holding the similarity values between
     * any pair of one path in this tree and one path in another tree
     */
    public void getLingMatchings(TNode working1, Hashtable nameSimTable, //name sim table
        Hashtable pathSimTable) //path sim table
        throws Exception {
        if (rootNode != null) {
            rootNode.getLingMatchingsForAll(working1, nameSimTable, pathSimTable);
        }
    }

    public void getNumOfPotentialMatches(SchemaTree anotherTree,
        Hashtable nameSimTable, Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, boolean interactive_on)
        throws Exception {
        if (rootNode != null) {
            rootNode.getNumOfPotentialMatchesForAll(anotherTree, nameSimTable,
                pathSimTable, descSimTable, dataTypeSimTable, interactive_on);
        }
    }

    public void writeMapToFile(PrintWriter stream) {
        //if(debug_on) System.err.println("In Tree writeToFile()");
        if (rootNode != null) {
            rootNode.writeAllMapToFile(stream);
        }
    }

    /**
     * Write the mapping between this schema tree and the source schema tree
     * into an XML document with structure specified in scia_mapping.xsd.
     @param stream The name of the ProntWriter to the mapping XML document
     @param sourceSchema The name of the source schema tree, i.e., schemaJTree1
     @param contextCheck The boolean value whether the mapping is computed
     using context check or not
     @param k The number of matches to be written
     */
    public void writeMapToXMLDoc(PrintWriter stream, String sourceSchema,
        boolean contextCheck, int k) {
        stream.println("<?xml version=\"1.0\"?>");
        stream.println("<mapping>");
        stream.println("<targetSchema> " + treeName + " </targetSchema>");
        stream.println("<sourceSchema> " + sourceSchema + " </sourceSchema>");

        if (rootNode != null) {
            rootNode.writeAllMapToXMLDoc(stream, contextCheck, k);
        }

        stream.println("</mapping>");
    }

    public PrintWriter write(PrintWriter stream, boolean writeMapping) {
        //stream.println("In Tree Write()");
        stream.println("Tree " + treeName + ":\noccupancy is " + occupancy +
            " elements.");

        return (rootNode != null)
        ? rootNode.write_AllTNodes(stream, "", writeMapping) : stream;
    }

    /**
     * Write out matches considering both directions. E.g., (S1:e1, S2:e2) is
     * accepted only if S2:e2 can not find better ei in S1 from the other
     * direction of matching.
     *
     * @param stream The PrintWriter of the outpu to be sent
     * @param writeMapping The boolean value indicating whether to write
     * final mapping or not
     * @param tree1 The SchemaTree to which this SchemaTree is matched with
     *
     * @return the PrintWriter
     */
    public PrintWriter writeBoth(PrintWriter stream, boolean writeMapping,
        SchemaTree tree1) {
        //stream.println("In Tree Write()");
        stream.println("Tree " + treeName + "(occupancy: " + occupancy +
            " paths)" + " <-> Tree " + tree1.treeName + "(occupancy: " +
            tree1.occupancy + " paths)");

        return (rootNode != null)
        ? rootNode.writeBothForAll(stream, "", writeMapping, tree1) : stream;
    }

    public void treeMatch(TNode working, // this dtd tree working node 
        TNode working1, // another dtd tree working node
        Hashtable simHash) // a table to hold similarity 
                           // between any two 
                           // nodes of two trees
     {
        StringMatcher sm = new StringMatcher();

        String path = working.getPath();
        String path1 = working1.getPath();

        String tag = working.data.getTag();
        String tag1 = working1.data.getTag();

        Double sim = new Double(sm.computeStringSimilarity(tag, tag1));

        //System.err.print("\nThe similarity is ");
        //System.err.print(sim);
        String key = "(" + path + "," + path1 + ")";
        simHash.put((Object) key, (Object) sim);

        if ((working.children != null) && !working.children.isEmpty()) {
            for (int i = 0; i < working.children.size(); i++) {
                treeMatch(working1, (TNode) working.children.get(i), simHash);
            }
        }

        if ((working1.children != null) && !working1.children.isEmpty()) {
            for (int i = 0; i < working1.children.size(); i++) {
                treeMatch((TNode) working1.children.get(i), working, simHash);
            }
        }
    }

    public void getBestNode(TNode working, // this dtd tree working node 
        TNode working1) // another dtd tree working node
     {
        working.getBestNodeForAll(working1);
    }

    public void getBestPath() {
        if (rootNode != null) {
            rootNode.getBestPathForAll();
        }
    }

    public void getBestPurePath(TNode working, // this dtd tree working node 
        TNode working1) //another dtd tree working node
     {
        working.getBestPurePathForAll(working1);
    }

    public void getFromNameAndPath(TNode working1, Hashtable nameSimTable, //name sim table
        Hashtable pathSimTable) //path sim table
        throws Exception {
        if (rootNode != null) {
            rootNode.getFromNameAndPathForAll(working1, nameSimTable,
                pathSimTable);
        }
    }

    public void getNameAndPathSimTables(Hashtable nameSimTable,
        Hashtable pathSimTable, TNode working1, Domain dm) {
        if (rootNode != null) {
            rootNode.getNameAndPathSimTablesForAll(nameSimTable, pathSimTable,
                working1, dm);
        }
    }

    public TNode getTNodeFromRowNum(int num) {
        TNode tnode;

        if (rootNode != null) {
            tnode = rootNode.getTNodeFromRowNumForAll(num);

            return tnode;
        }

        return null;
    }

    public Model toRDFModel(Hashtable resHash) {
        Model model = SchemaTree.rf.createModel();

        if (rootNode != null) {
            try {
                rootNode.toRDFModelForAll(null, model, resHash);
            } catch (java.lang.Exception e) {
                if (debug_on) {
                    System.err.println("From TNode.toRDFModelForAll(): " + e);
                }

                e.printStackTrace();
            }
        }

        return model;
    }

    public List getInitMap(Hashtable thisResHash, Hashtable resHash)
        throws Exception {
        List initMap = new ArrayList();

        if (rootNode != null) {
            rootNode.getInitMapForAll(thisResHash, resHash, initMap);
        }

        return initMap;
    }

    public List getInitMapFromLinguistic(Hashtable thisResHash,
        Hashtable resHash) throws Exception {
        List initMap = new ArrayList();

        if (rootNode != null) {
            rootNode.getInitMapFromLinguisticForAll(thisResHash, resHash,
                initMap);
        }

        return initMap;
    }

    public TNode getTNodeFromPath(String path) {
        if (rootNode != null) {
            TNode tnode = rootNode.getTNodeFromPathForAll(path);

            if (tnode != null) {
                return tnode;
            }
        }

        return null;
    }

    /**
     * Compute the best matching candidates for all nodes in this tree by
     * combining all the matching results of linguistic, structural and
     * optional context check. Asks user input if
     * decision cannot be made at critical points in interactive mode.
     *
     * @param mapFromGraph The output from graph matching
     * @param nameSimTable The Hashtable that stores the computed similarity
     * for any pair of a node of this SchemaTree and a node of anotherTree
     * @param pathSimTable The Hashtable that stores the computed similarity
     * for any pair of path of this SchemaTree and a path of anotherTree
     * @param descSimTable The Hashtable that stores the computed similarity
     * for any pair of node description of this SchemaTree and a node description
     * of anotherTree
     * @param dataTypeSimTable The Hashtable that stores the similarity
     * for any pair of common data types
     * @param resHash The Hashtable holding the resources for paths
     * @param tree The SchemaTree is being matched with this SchemaTree
     * @param fromContextCheck The boolean value indicating this method is
     * called from context check process or not
     * @param fromUserInputSubTreeMatch The boolean value indicating this method
     * is called in the process of matching subtrees of user input match or not
     * @param interactive_on The boolean value indicating user interaction is
     * expected or not, i.e., in interactive mode or non-intercative mode
     */
    public void getFromComb(Map mapFromGraph, Hashtable nameSimTable,
        Hashtable pathSimTable, Hashtable descSimTable,
        Hashtable dataTypeSimTable, Hashtable resHash, SchemaTree tree,
        boolean fromContextCheck, boolean fromUserInputSubTreeMatch,
        boolean interactive_on) throws Exception {
        if (rootNode != null) {
            rootNode.getFromCombForAll(mapFromGraph, nameSimTable,
                pathSimTable, descSimTable, dataTypeSimTable, resHash, tree,
                fromContextCheck, fromUserInputSubTreeMatch, interactive_on);
        }
    }

    public void adjustMatchWithHier() {
        if (rootNode != null) {
            rootNode.adjustMatchWithHierForAll();
        }
    }

    public void setRowNum() {
        if (rootNode != null) {
            currentRowNum = 0;
            rootNode.setRowNumForAll();
        }
    }

    public void clearMatchedTimes() {
        if (rootNode != null) {
            rootNode.clearMatchedTimesForAll();
        }
    }

    public String generateView(String documentFile) {
        String query = new String();
        StringBuffer queryHead = new StringBuffer();
        StringBuffer udfDefs = new StringBuffer();

        if (rootNode != null) {
            // assume only one root, isLastChild = true
            query = rootNode.generateView(documentFile, "", "", null, true,
                    null, queryHead, udfDefs);
        }

        query = queryHead.append(udfDefs.toString()).append(query).toString();

        return query;
    }

    public void updateTreeField() {
        if (rootNode != null) {
            rootNode.updateTreeFieldForAll(this);
        }
    }

    /**
     * Initialize mapping from an XML document, assume its root and schema names
     * are checked already.
     @param rootElem The root element of the XML document of mapping to reuse
     @param sTree The source SchemaTree
     */
    public void initMappingFromXML(Element rootElem, SchemaTree sTree) {
        try {
            List components = rootElem.getChildren("component");

            if ((components == null) || components.isEmpty()) {
                return;
            }

            for (int i = 0; i < components.size(); i++) {
                Element component = (Element) components.get(i);
                String tPath = component.getChildTextTrim("targetPath");
                TNode tTNode = getTNodeFromPath(tPath);

                // Import the best match
                // Import 1-to-1 match
                Element best = component.getChild("one-to-one");

                if ((tTNode != null) && (best != null)) {
                    tTNode.fromComb = init_1_to_1(best, sTree);

                    if (tTNode.fromComb != null) {
                        if (tTNode.matchings == null) {
                            tTNode.matchings = new Vector();
                        }

                        tTNode.matchings.add(tTNode.fromComb);
                    }
                }

                // Import constant
                Element consElem = component.getChild("constant");

                if ((tTNode != null) && (consElem != null)) {
                    tTNode.constantValue = consElem.getTextTrim();
                }

                // Import function output
                Element funElem = component.getChild("functionOutput");

                if ((tTNode != null) && (funElem != null)) {
                    tTNode.functionOutput = funElem.getTextTrim();
                }

                // Import local n:1 matches
                Element localElem = component.getChild("local-n-to-1");

                if ((tTNode != null) && (localElem != null)) {
                    tTNode.fromComb = init_local_n_to_1(localElem, sTree);
                }

                // Import global n:1 matches
                Element globalElem = component.getChild("global-n-to-1");

                if ((tTNode != null) && (globalElem != null)) {
                    init_global_n_to_1(globalElem, sTree, tTNode);
                }

                // import the second, third, ...
                for (int j = 2; j <= SCIA.NUMBER_TOP_MATCHES_TO_OUTPUT; j++) {
                    String name = "match_" + j;
                    Element otherMatchElem = component.getChild(name);

                    if ((tTNode != null) && (otherMatchElem != null)) {
                        Matching match = init_1_to_1(otherMatchElem, sTree);

                        if (match != null) {
                            if (debug_on) {
                                System.err.println("match_" + j + " = " +
                                    match.tnode.getPath());
                            }

                            tTNode.matchings.add(match);
                        }
                    }
                }
            }
        } catch (Exception e) {
            if (debug_on) {
                System.err.println("SchemaTree::initMappingFromXML()");
            }

            if (debug_on) {
                System.err.println(e.getMessage());
            }

            e.printStackTrace();
        }
    }

    /**
     * Initialize a global n-to-1 mapping from an element in xml document.
     *
     * @param globalElem The gloal n-to-1 element in the mapping document to
     * reuse
     * @param sourceTree The source SchemaTree
     * @param node The schema tree node with which the globalElem is associated
     */
    public void init_global_n_to_1(Element globalElem, SchemaTree sourceTree,
        TNode node) {
        if (globalElem == null) {
            return;
        }

        List unionMatches = new ArrayList();
        List oneToOneChildren = globalElem.getChildren("1-to-1");

        if ((oneToOneChildren != null) && !oneToOneChildren.isEmpty()) {
            for (int i = 0; i < oneToOneChildren.size(); i++) {
                Element child = (Element) oneToOneChildren.get(i);
                unionMatches.add(init_1_to_1(child, sourceTree));
            }
        }

        List local_n_1Children = globalElem.getChildren("local-n-to-1");

        if ((local_n_1Children != null) && !local_n_1Children.isEmpty()) {
            for (int i = 0; i < local_n_1Children.size(); i++) {
                Element child = (Element) local_n_1Children.get(i);
                unionMatches.add(init_local_n_to_1(child, sourceTree));
            }
        }

        // Check whether parent has union matches or not
        if ((node.parent != null) &&
                (node.parent.independentUnionMatches != null) &&
                !node.parent.independentUnionMatches.isEmpty()) {
            node.dependentUnionMatches = unionMatches;
        } else if ((node.parent != null) &&
                (node.parent.independentUnionMatches == null)) {
            node.independentUnionMatches = unionMatches;
        }
    }

    /**
     * Initialize a local n-to-1 mapping from an element in xml document.
     @param localElem The local n-to-1 element in the mapping document to reuse
     @param sTree The source SchemaTree
     */
    public Matching init_local_n_to_1(Element localElem, SchemaTree sourceTree) {
        Matching m = new Matching();
        Element condElem = localElem.getChild("condition");
        Element simElem = localElem.getChild("similarity");
        int numOfChildren = localElem.getChildren().size();
        int numOfUnits = 0;

        if ((condElem != null) && (simElem != null)) {
            numOfUnits = numOfChildren - 2; //condElem and simElem
            m.condition = initCondition(condElem, sourceTree);
            m.sim = new Double(simElem.getTextTrim()).doubleValue();
        } else if ((condElem == null) && (simElem != null)) {
            numOfUnits = numOfChildren - 1; //simElem
            m.sim = new Double(simElem.getTextTrim()).doubleValue();
            ;
        } else if (simElem == null) {
            numOfUnits = numOfChildren; // all children are units
            m.sim = 1.0;
        }

        List mUnits = new ArrayList();

        for (int j = 1; j <= numOfUnits; j++) {
            String name = "unit" + j;
            Element unit = localElem.getChild(name);
            MatchingUnit mUnit = new MatchingUnit(sourceTree.getTNodeFromPath(
                        unit.getChildTextTrim("sourcePath")),
                    unit.getChildTextTrim("operations"),
                    unit.getChildTextTrim("combOperation"));
            mUnits.add(mUnit);
        }

        m.matchingUnits = mUnits;

        return m;
    }

    /**
     * Initialize a condition from an condition element in xml document.
     @param condElem The condition element of a mapping in the mapping document
     to reuse
     @param sourceTree The source SchemaTree
     @param targetTree The target SchemaTree
     */
    public Condition initCondition(Element condElem, SchemaTree sourceTree) {
        Condition cond = null;
        String path1;
        String path2;

        if (condElem != null) {
            String value = condElem.getChildTextTrim("constant");
            List sPaths = condElem.getChildren("sourcePath");
            List tPaths = condElem.getChildren("targetPath");
            String symbol = condElem.getChildTextTrim("compareSymbol");

            if (sPaths.size() == 2) {
                path1 = (String) sPaths.get(0);
                path2 = (String) sPaths.get(1);
                cond = new Condition(sourceTree.getTNodeFromPath(path1),
                        symbol, sourceTree.getTNodeFromPath(path2));
            } else if (tPaths.size() == 2) {
                path1 = (String) tPaths.get(0);
                path2 = (String) tPaths.get(1);
                cond = new Condition(getTNodeFromPath(path1), symbol,
                        getTNodeFromPath(path2));
            } else if ((sPaths.size() == 1) && (sPaths.size() == 1)) {
                path1 = (String) sPaths.get(0);
                path2 = (String) tPaths.get(0);
                cond = new Condition(sourceTree.getTNodeFromPath(path1),
                        symbol, getTNodeFromPath(path2));
            } else if ((sPaths.size() == 1) && (value != null)) {
                path1 = (String) sPaths.get(0);
                cond = new Condition(sourceTree.getTNodeFromPath(path1),
                        symbol, value);
            } else if ((tPaths.size() == 1) && (value != null)) {
                path1 = (String) tPaths.get(0);
                cond = new Condition(getTNodeFromPath(path1), symbol, value);
            }
        }

        return cond;
    }

    /**
     * Initialize an 1-to-1 mapping from an mapping element in xml document.
     @param localElem The 1-to-1 element in the mapping document to reus
     @param sourceTree The source SchemaTree
     */
    public Matching init_1_to_1(Element mElem, SchemaTree sourceTree) {
        String sPath = mElem.getChildTextTrim("sourcePath");
        TNode sTNode = sourceTree.getTNodeFromPath(sPath);

        Double sim = new Double(mElem.getChildTextTrim("similarity"));

        // Give implicit perfect match similarity value 1.0
        if (sim == null) {
            sim = new Double(1.0);
        }

        //Get operations
        String op = mElem.getChildTextTrim("operations");

        // Get condition
        Element condElem = mElem.getChild("condition");
        Condition cond = initCondition(condElem, sourceTree);

        // get grouping attributes
        Vector grpAttrs = null;
        Element grpElem = mElem.getChild("groupAttrs");

        if (grpElem != null) {
            List grpAttrList = grpElem.getChildren("groupAttr");

            if (debug_on) {
                System.err.println("grpAttrList.size() = " +
                    grpAttrList.size());
            }

            grpAttrs = new Vector();

            for (int i = 0; i < grpAttrList.size(); i++) {
                String path = ((Element) grpAttrList.get(i)).getTextTrim();
                TNode node = sourceTree.getTNodeFromPath(path);
                grpAttrs.add(node);
            }
        }

        // get Aggregate operation
        String aggregator = null;

        if (mElem.getChild("aggregator") != null) {
            aggregator = mElem.getChildTextTrim("aggregator");
        }

        // create a new match and return it
        if (sim != null) {
            Matching match = new Matching(sTNode, sim.doubleValue(), op,
                    grpAttrs, aggregator, cond);

            return match;
        }

        return null;
    }

    public void initMappingFromFile(BufferedReader r, //reader for the existing 
                                                      //mapping file
        SchemaTree sTree) { //another tree to match

        // the first line of the mapping file is read and checked already
        if (debug_on) {
            System.err.println("init mapping from file!");
        }

        try {
            String line;
            StringTokenizer st;
            String sPath; // source path
            String tPath; // target path
            Double sim;
            TNode sTNode;
            TNode tTNode;

            while ((line = r.readLine()) != null) {
                st = new StringTokenizer(line);

                if (debug_on) {
                    System.err.println();
                }

                if (debug_on) {
                    System.err.print("line = " + line);
                }

                if (debug_on) {
                    System.err.print("# token = " + st.countTokens());
                }

                if (st.countTokens() >= 3) {
                    if (debug_on) {
                        System.err.print(" Valid match!");
                    }

                    sPath = (String) st.nextElement();

                    if (debug_on) {
                        System.err.println(" after taking 1st, # token = " +
                            st.countTokens());
                    }

                    if (debug_on) {
                        System.err.print(" sPath = " + sPath);
                    }

                    st.nextElement(); // throw " --> "
                    tPath = (String) st.nextElement();

                    if (debug_on) {
                        System.err.print("  tPath = " + tPath);
                    }

                    sTNode = sTree.getTNodeFromPath(sPath);
                    tTNode = this.getTNodeFromPath(tPath);

                    if ((sTNode != null) && (tTNode != null)) {
                        if (debug_on) {
                            System.err.println(
                                " sTNode and tTNode are not null");
                        }

                        //sim =  (Double)st.nextElement();
                        sim = new Double(st.nextToken());

                        if (sim != null) {
                            tTNode.fromComb = new Matching(sTNode,
                                    sim.doubleValue());

                            //System.err.println("sim value exists!");
                            if (st.hasMoreTokens()) {
                                tTNode.fromComb.conditionStr = st.nextToken();

                                if (debug_on) {
                                    System.err.println("conditionStr = " +
                                        tTNode.fromComb.conditionStr);
                                }
                            }
                        } else {
                            tTNode.fromComb = new Matching(sTNode, 1.0);
                        }
                    }
                }
            }
        } catch (IOException e) {
            if (debug_on) {
                System.err.println(e.getMessage());
            }
        }
    }

    /**
     * Record cardinality info of the input content's all elements and their
       all child elements in a hashtable. Key is the element's tag.
       In insertChildren(), the hashtable is accessed to get cardinality info
       for each element.

       Note: How to deal with choice group and sequence (?),
       e.g. (author+ | editor+) is a problem!
       Currently, the "OR" case is not considered yet.
    */
    public void setChildCardinality(Group content, Hashtable cardinalityHash) {
        Particle particle;
        boolean repeatable;
        ElementType child;
        Cardinality cardi;
        String minOc = null;
        String maxOc = null;

        if (content == null) {
            if (debug_on) {
                System.err.println("content is null");
            }

            return;
        }

        for (int i = 0; i < content.members.size(); i++) {
            // Get the content particle and determine if its cardinality info.
            particle = (Particle) content.members.elementAt(i);

            // Process the content particle.
            //
            // If the content particle is a reference to an element type,
            // save information about whether the property is repeatable.
            // If the content particle is a group, process it recursively.
            if (particle.type == Particle.TYPE_ELEMENTTYPEREF) {
                Reference ref = (Reference) particle;
                child = ref.elementType;

                if (particle.isRepeatable && particle.isRequired) {
                    minOc = "1";
                    maxOc = "*";
                    cardi = new Cardinality(minOc, maxOc);
                    cardinalityHash.put(child.name.getLocalName(), cardi);
                } else if (!particle.isRepeatable && particle.isRequired) {
                    minOc = "1";
                    maxOc = "1";
                    cardi = new Cardinality(minOc, maxOc);
                    cardinalityHash.put(child.name.getLocalName(), cardi);
                } else if (particle.isRepeatable && !particle.isRequired) {
                    minOc = "0";
                    maxOc = "*";
                    cardi = new Cardinality(minOc, maxOc);
                    cardinalityHash.put(child.name.getLocalName(), cardi);
                } else if (!particle.isRepeatable && !particle.isRequired) {
                    minOc = "0";
                    maxOc = "1";
                    cardi = new Cardinality(minOc, maxOc);
                    cardinalityHash.put(child.name.getLocalName(), cardi);
                }

                //if(debug_on) System.err.println("in setCar(), deal with " + 
                //			   child.name.getLocalName());
                //if(debug_on) System.err.print(" is ELEMENTTYPEREF" + "  minOc = " + minOc +
                //	 "  maxOc = " + maxOc + "\n");
                //recursion, handling children
                if ((child.children != null) && !child.children.isEmpty()) {
                    //if(debug_on) System.err.println("deal with " + child.name.
                    //getLocalName() + "'s children");
                    setChildCardinality(child.content, cardinalityHash);
                }
            } else {
                //particle.type = Particle.TYPE_CHOICE || Particle.TYPE_SEQUENCE
                //if(debug_on) System.err.println("TYPE_CHOICE||TYPE_SEQUENCE");
                setChildCardinality((Group) particle, cardinalityHash);
            }
        }
    }

    public PrintWriter write_ToAbstractSchema(PrintWriter stream) {
        //stream.println("In Tree Write_ToAbstractSchema()");
        stream.println("Tree " + treeName + ":\noccupancy is " + occupancy +
            " elements.");

        return (topType != null) ? topType.write_All(stream) : stream;
    }

    public Type getTopType() {
        return topType;
    }

    public String getSchemaType() {
        return schemaType;
    }

    /**
     * Add this schema tree node's existing match and all decendent nodes'
     * matches with higher similarity values into
     * the input List goodInputMap for reuse in the further matching process.
     *
     * @param thisResHash The Hashtable for retrieving this node's corresponding
     * resource from its path
     * @param reHash The Hashtable for retrieving this node's matching nodes'
     * resources from their paths
     * @goodInputMap The list of good matches to reuse
     *
     * @return The good input map list for reuse in the matching process
     */
    public List getGoodInputMap(Hashtable thisResHash, Hashtable resHash) {
        List goodInputMap = new ArrayList();

        if (rootNode != null) {
            rootNode.addGoodInputMapForAll(thisResHash, resHash, goodInputMap);
        }

        return goodInputMap;
    }

    public static void main(String[] args) throws Exception {
        PrintWriter cout = new PrintWriter(System.err, true);

        /* test abstract schema
        SchemaTree tr = new SchemaTree("book1.dtd", "DTD");
        if(debug_on) System.err.println("topType's name= " + tr.topType.typeName +
                           " type = " + tr.topType.typeType);

        tr.write_ToAbstractSchema(cout);

        tr = new SchemaTree("book1.dtd", "DTD");
        if(debug_on) System.err.println("topType's name= " + tr.topType.typeName +
                           " type = " + tr.topType.typeType);
        tr.write_ToAbstractSchema(cout);
        */
        /*test shema tree from owl ontology*/
        SchemaTree tr1 = new SchemaTree("./schemas/food_m.owl", "Ontology");
        tr1.write(cout, false);

        cout.flush();
        cout.close();
    }
}
