/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.io.PrintWriter;
import java.util.Vector;

import org.apache.xerces.dom3.bootstrap.DOMImplementationRegistry;
import org.apache.xerces.xs.XSAttributeDeclaration;
import org.apache.xerces.xs.XSAttributeUse;
import org.apache.xerces.xs.XSComplexTypeDefinition;
import org.apache.xerces.xs.XSConstants;
import org.apache.xerces.xs.XSElementDeclaration;
import org.apache.xerces.xs.XSImplementation;
import org.apache.xerces.xs.XSLoader;
import org.apache.xerces.xs.XSModel;
import org.apache.xerces.xs.XSModelGroup;
import org.apache.xerces.xs.XSNamedMap;
import org.apache.xerces.xs.XSObject;
import org.apache.xerces.xs.XSObjectList;
import org.apache.xerces.xs.XSParticle;
import org.apache.xerces.xs.XSSimpleTypeDefinition;
import org.apache.xerces.xs.XSTerm;
import org.apache.xerces.xs.XSTypeDefinition;


/**
 * The parser for XML Schema files based on Apache XML Schema APIs. The
 * XMLSchemaImporter imports an XML Schema file into a SchemaTree in SCIA.
 *
 *@author Guilian Wang
 */
public class XMLSchemaImporter {
    XSModel xsmodel;
    XSElementDeclaration rootElem;
    SchemaTree tree = new SchemaTree();

    /**
     * Constructor. It is to create an XSModel in APache XML Schema package
     * included in the xerces package from the input xmlSchemaFile,
     * and build a SchemaTree from the XSModel.
     *
     * @param xmlSchemaFile The name of the input XML Schema file
     */
    public XMLSchemaImporter(String xmlSchemaFile) {
        try {
            // get DOM Implementation using DOM Registry
            System.setProperty(DOMImplementationRegistry.PROPERTY,
                "org.apache.xerces.dom.DOMXSImplementationSourceImpl");

            DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();
            XSImplementation impl = (XSImplementation) registry.getDOMImplementation(
                    "XS-Loader");
            XSLoader schemaLoader = impl.createXSLoader(null);

            //DOMConfiguration config = schemaLoader.getConfig();
            // create Error Handler
            //DOMErrorHandler errorHandler = new SchemaTree();
            // set error handler
            //config.setParameter("error-handler", errorHandler);
            // set validation feature
            //config.setParameter("validate", Boolean.TRUE);
            // parse document
            if (SCIA.debug_on) {
                System.err.println("Parsing " + xmlSchemaFile + "...");
            }

            xsmodel = schemaLoader.loadURI(xmlSchemaFile);

            buildSchemaTree();

            //initialize tree name 
            int j = xmlSchemaFile.lastIndexOf(".");
            int i = xmlSchemaFile.lastIndexOf("/");
            tree.treeName = xmlSchemaFile.substring(i + 1, j);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * Find the root element in the XSModel created from the input XML Schema
     * file of the XMLSchemaImporter.
     */
    public void getRootElem() {
        if (xsmodel != null) {
            // global element declarations
            XSNamedMap nameMap = xsmodel.getComponents(XSConstants.ELEMENT_DECLARATION);

            if (nameMap.getLength() == 0) {
                System.err.println("The schema has no element declaration!");
                System.exit(0);
            }

            if (SCIA.debug_on) {
                System.err.println("**************************************");
            }

            if (SCIA.debug_on) {
                System.err.println("* Global elements: {namespace} name");
            }

            if (SCIA.debug_on) {
                System.err.println("**************************************");
            }

            Vector pRootElems = new Vector(); // potential root elements

            for (int i = 0; i < nameMap.getLength(); i++) {
                XSObject item = nameMap.item(i);

                if (SCIA.debug_on) {
                    System.err.println("{" + item.getNamespace() + "}" +
                        item.getName());
                }

                XSElementDeclaration elDec = (XSElementDeclaration) item;
                XSTypeDefinition typeDef = elDec.getTypeDefinition();

                if (typeDef instanceof XSComplexTypeDefinition) {
                    short contentType = ((XSComplexTypeDefinition) typeDef).getContentType();

                    if (contentType == XSComplexTypeDefinition.CONTENTTYPE_ELEMENT) {
                        //to do: currently take the first complex elem as root
                        if (rootElem == null) {
                            rootElem = elDec;

                            if (SCIA.debug_on) {
                                System.err.println(
                                    "1st top level complex elem _elem = " +
                                    rootElem.getName());
                            }

                            return;
                        }

                        //System.out.println("content_element");
                        pRootElems.add(elDec);
                    } else if (contentType == XSComplexTypeDefinition.CONTENTTYPE_MIXED) {
                        //to do: currently take the first complex elem as root
                        if (rootElem != null) {
                            rootElem = elDec;

                            if (SCIA.debug_on) {
                                System.err.println(
                                    "1st top level complex elem _mixed = " +
                                    rootElem.getName());
                            }

                            return;
                        }

                        //System.out.println("content_mixed");
                        pRootElems.add(elDec);
                    } /*else if (contentType ==
                         XSComplexTypeDefinition.CONTENTTYPE_EMPTY){
                    if(SCIA.debug_on) System.err.println("content_empty");
                    } else if (contentType ==
                         XSComplexTypeDefinition.CONTENTTYPE_SIMPLE){
                    if(SCIA.debug_on) System.err.println("content_simple");
                    }*/}
            }

            for (int h = 0; h < pRootElems.size(); h++) {
                if (SCIA.debug_on) {
                    System.err.println(h + "th possible root = " +
                        ((XSObject) pRootElems.get(h)).getName());
                }
            }

            if (pRootElems.size() > 1) {
                //to do: debug isParentOf() that causes outOfMemoryError
                rootElem = getRootOf(pRootElems);

                //rootElem = (XSElementDeclaration)pRootElems.get(0);
            }
        }

        //System.out.println("root elem = " + rootElem.getName());
    }

    public XSElementDeclaration getRootOf(Vector possibleRoots) {
        XSElementDeclaration root = (XSElementDeclaration) possibleRoots.get(0);
        boolean noRoot = false; // single rooted

        for (int i = 1; i < possibleRoots.size(); i++) {
            XSElementDeclaration elem = (XSElementDeclaration) possibleRoots.get(i);

            /*
            if (isParentOf(elem, root)) {
                root = elem;
                noRoot = false;
            } else if (isParentOf(root, elem)) {
                noRoot = false;
            } else if (!isParentOf(root, elem) && !isParentOf(elem, root) ) {
                noRoot = true;
            }
            */
            if (isParentOf(root, elem)) {
                if (SCIA.debug_on) {
                    System.err.println(root.getName() + " isParentOf " +
                        elem.getName());
                }

                noRoot = false;
            } else if (isParentOf(elem, root)) {
                root = elem;
                noRoot = false;
            }
        }

        return root;
    }

    public boolean isParentOf(XSObject ob1, XSObject ob2) {
        if (SCIA.debug_on) {
            System.err.println("in isParentOf(): " + "ob1 = " + ob1.getName() +
                " ob2 = " + ob2.getName());
        }

        /* breadth first search for ob2 from ob1 tree
           maintain a list of ob1 nodes to check itself or has child == ob2
        */
        Vector toCheck = new Vector();

        if (ob1 instanceof XSElementDeclaration) {
            XSElementDeclaration ed1 = (XSElementDeclaration) ob1;
            XSTypeDefinition type1 = ed1.getTypeDefinition();

            //??? to be parent, ob1 has to be complex type or model group 
            if (!(type1 instanceof XSComplexTypeDefinition) &&
                    !(ob1 instanceof XSModelGroup)) {
                return false;
            } else if (type1 instanceof XSComplexTypeDefinition) {
                XSParticle p = ((XSComplexTypeDefinition) type1).getParticle();

                if (p == null) {
                    return false;
                }

                XSTerm term = p.getTerm();

                if (term instanceof XSModelGroup) {
                    XSModelGroup mg = (XSModelGroup) term;
                    XSObjectList ol = mg.getParticles();

                    if ((ol == null) || (ol.getLength() == 0)) {
                        return false;
                    }

                    for (int i = 0; i < ol.getLength(); i++) {
                        XSParticle subP = (XSParticle) ol.item(i);
                        toCheck.addElement(subP);
                    }
                } // else if instanceof XSElementDeclaration or XSWildCard ???
            }
        } else if (ob1 instanceof XSModelGroup) {
            XSModelGroup mg = (XSModelGroup) ob1;
            XSObjectList ol = mg.getParticles();

            if ((ol == null) || (ol.getLength() == 0)) {
                return false;
            }

            //System.out.println("ol.length = " + ol.getLength());

            /* breadth first search */

            // add particles into toCheck vector
            for (int i = 0; i < ol.getLength(); i++) {
                XSParticle subP = (XSParticle) ol.item(i);
                toCheck.addElement(subP);
            }

            /* depth first search, breadth first search is much better
            for (int i = 0; i < ol.getLength(); i++) {
                XSTerm subTerm = ((XSParticle)ol.item(i)).getTerm();
                if (subTerm instanceof XSElementDeclaration) {
                    XSElementDeclaration ed = (XSElementDeclaration)subTerm;
                    //System.out.println("subTerm is ED " + subTerm.getName());
                    XSTypeDefinition type = ed.getTypeDefinition();
                    if (ed.getName().equals(ob2.getName())) {
                        return true;
                    } else if (type instanceof XSComplexTypeDefinition) {
                        if (isParentOf(ed, ob2)) {
                            return true;
                        }
                    }
                } else if (subTerm instanceof XSModelGroup) {
                    XSModelGroup subMG = (XSModelGroup)subTerm;
                    if (isParentOf(subMG, ob2)) {
                        return true;
                    }
                }
            }
            */
        }

        // to check the children nodes in toCheck
        for (int i = 0; i < toCheck.size(); i++) {
            XSTerm subTerm = ((XSParticle) toCheck.get(i)).getTerm();

            if (subTerm instanceof XSElementDeclaration) {
                XSElementDeclaration ed = (XSElementDeclaration) subTerm;

                //System.out.println("subTerm is ED " + subTerm.getName());
                XSTypeDefinition type = ed.getTypeDefinition();

                if (ed.getName().equals(ob2.getName())) {
                    return true;
                } else if (type instanceof XSComplexTypeDefinition) {
                    // add children to the end of toCheck vector
                    XSParticle p = ((XSComplexTypeDefinition) type).getParticle();

                    if (p != null) {
                        XSTerm term = p.getTerm();

                        if (term instanceof XSModelGroup) {
                            XSObjectList ol = ((XSModelGroup) term).getParticles();

                            if ((ol != null) && (ol.getLength() != 0)) {
                                for (int j = 0; j < ol.getLength(); j++) {
                                    XSParticle subP = (XSParticle) ol.item(j);
                                    toCheck.addElement(subP);
                                }
                            }
                        }
                    }
                }
            } else if (subTerm instanceof XSModelGroup) {
                XSModelGroup subMG = (XSModelGroup) subTerm;

                // add children to the end of toCheck vector
                XSObjectList subOL = subMG.getParticles();

                if ((subOL != null) && (subOL.getLength() != 0)) {
                    for (int j = 0; j < subOL.getLength(); j++) {
                        XSParticle subP = (XSParticle) subOL.item(j);
                        toCheck.addElement(subP);
                    }
                }
            }
        }

        return false;
    }

    /** Build the SchemaTree tree in the XMLSchemaInporter using Apache XML
     * Schema APIs.
     */
    public void buildSchemaTree() {
        tree.schemaType = "XMLSchema";

        /*to do: now assume 1st complex type elem is the root
        */
        getRootElem();

        //System.out.println("root elem = " + rootElem.getName());
        String description = getDescription(rootElem);
        SchemaNode root = new SchemaNode(rootElem.getName(), "rootElem", "",
                description);
        tree.rootNode = new TNode(root);
        tree.rootNode.tree = tree;
        insertChildren(tree.rootNode, rootElem);

        //set row number for each node
        tree.setRowNum();
    }

    /**
     * Insert subelements and attributes of the input XSObject ob (an element,
     * or a model group) as children of the SchemaTree node.
     *
     * @param node The TNode to which the children nodes are to be attached
     * @param ob The XSObject whose subelements and attributes to be handled
     */
    public void insertChildren(TNode node, XSObject ob) {
        String tag = new String();
        TNode child;
        SchemaNode sn;

        if (ob instanceof XSElementDeclaration) {
            XSElementDeclaration ed = (XSElementDeclaration) ob;
            XSTypeDefinition type = ed.getTypeDefinition();

            //??? to be parent, ob1 has to be complex type or model group 
            if (!(type instanceof XSComplexTypeDefinition) &&
                    !(ob instanceof XSModelGroup)) {
                return;
            } else if (type instanceof XSComplexTypeDefinition) {
                // process attributes
                XSObjectList auOL = ((XSComplexTypeDefinition) type).getAttributeUses();

                if ((auOL != null) && (auOL.getLength() > 0)) {
                    for (int i = 0; i < auOL.getLength(); i++) {
                        XSAttributeUse au = (XSAttributeUse) auOL.item(i);
                        String maxOc = "1";
                        String minOc = "0";

                        if (au.getRequired()) {
                            minOc = "1";
                        }

                        XSAttributeDeclaration ad = au.getAttrDeclaration();
                        XSSimpleTypeDefinition at = ad.getTypeDefinition();

                        sn = new SchemaNode(ad.getName(), "attribute",
                                at.getName(), minOc, maxOc, getDescription(ad));

                        child = new TNode(sn, node);

                        tree.occupancy++;
                    }
                }

                //process subelements
                XSParticle p = ((XSComplexTypeDefinition) type).getParticle();

                if (p == null) {
                    return;
                }

                XSTerm term = p.getTerm();

                if ((term instanceof XSModelGroup) ||
                        (term instanceof XSElementDeclaration)) {
                    insertChildren(node, term);
                } //else if  XSWildCard ???
            } // simeple type doesn't allow attributes
        } else if (ob instanceof XSModelGroup) {
            XSModelGroup mg = (XSModelGroup) ob;
            short compos = mg.getCompositor();

            if (compos == XSModelGroup.COMPOSITOR_ALL) {
                //System.out.println("********** compositor_all");
                tag = ".all.";
            } else if (compos == XSModelGroup.COMPOSITOR_CHOICE) {
                //System.out.println("compositor_choice");
                // add a .choice. node
                tag = ".choice.";
            } else if (compos == XSModelGroup.COMPOSITOR_SEQUENCE) {
                //System.out.println("compositor_sequence");
                // add a .sequence. node
                tag = ".sequence.";
            }

            sn = new SchemaNode(tag, "compositor", "");

            TNode compositorNode = new TNode(sn, node);

            //compositor nodes don't affect the depth of the schema tree
            compositorNode.height = node.height;

            tree.occupancy++;

            XSObjectList ol = mg.getParticles();

            if ((ol == null) || (ol.getLength() == 0)) {
                return;
            }

            //System.out.println("ol.length = " + ol.getLength());

            /* insert all subelements as child of the compositor, then
               insertChildren for each of the subelements recursively,
               depth first */
            for (int i = 0; i < ol.getLength(); i++) {
                XSParticle subP = (XSParticle) ol.item(i);
                String maxOc = "1";

                if (subP.getMaxOccursUnbounded()) {
                    maxOc = "unbounded";
                }

                String minOc = maxOc.valueOf(subP.getMinOccurs());
                XSTerm subTerm = subP.getTerm();

                if (subTerm instanceof XSElementDeclaration) {
                    XSElementDeclaration subEd = (XSElementDeclaration) subTerm;

                    //System.out.println("subTerm is ED " + subTerm.getName());
                    tag = subEd.getName();

                    XSTypeDefinition subType = subEd.getTypeDefinition();
                    sn = new SchemaNode(tag, "leafElem", subType.getName(),
                            minOc, maxOc, getDescription(subEd));
                    child = new TNode(sn, compositorNode);

                    tree.occupancy++;

                    /* detect recursive reference, if this node appear twice
                       in the path to the root, then don't add its chidren
                    */
                    TNode ref = child.getRefNode(tag, subType.getName());

                    if (ref == null) {
                        insertChildren(child, subEd);
                    } else {
                        child.reference = ref;
                    }
                } else if (subTerm instanceof XSModelGroup) {
                    insertChildren(compositorNode, subTerm);
                }
            }
        }
    }

    public String getDescription(XSElementDeclaration ed) {
        String description = null;

        if (ed.getAnnotation() != null) {
            String annotation = ed.getAnnotation().getAnnotationString();
            int end = annotation.indexOf("</xs:documentation");

            if (end > 0) {
                String desc = annotation.substring(0, end);
                int start = desc.lastIndexOf(">");

                if (start > 0) {
                    description = desc.substring(start + 1, end);
                }
            }
        }

        return description;
    }

    public String getDescription(XSAttributeDeclaration ad) {
        String description = null;
        String annotation = ad.getAnnotation().getAnnotationString();
        int end = annotation.indexOf("</xs:documentation");

        if (end > 0) {
            String desc = annotation.substring(0, end);
            int start = desc.lastIndexOf(">");

            if (start > 0) {
                description = desc.substring(start + 1, end);
            }
        }

        return description;
    }

    public static void main(String[] args) throws Exception {
        String SCIA_WORK_DIR = System.getProperty("env.SCIA_WORK_DIR");
        XMLSchemaImporter imp = new XMLSchemaImporter(SCIA_WORK_DIR +
                "/schemas/adn_record.xsd");

        //imp.getRootElem();
        //imp.buildSchemaTree();
        PrintWriter cout = new PrintWriter(System.out, true);
        imp.tree.write(cout, true);

        /*
        TNode ds = imp.tree.
            getTNodeFromPath("/eml/dataset/methods/methodStep/dataSource");
        if (ds != null) {
            ds.write_AllTNodes(cout, "", true);
            ds.addRefSubTree();
            ds.write_AllTNodes(cout, "", true);
        } else {
            if(SCIA.debug_on) System.err.println("ds = null");
        }
        */
        cout.flush();
    }
}
