/**
 *    '$RCSfile: OntoClassSelectionJPanel.java,v $'
 *
 *     '$Author: bowers $'
 *       '$Date: 2006/02/28 19:22:09 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler-project.org
 *
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms.gui;

import org.kepler.sms.OntologyCatalog;
import org.kepler.sms.NamedOntModel;
import org.kepler.sms.NamedOntClass;
import java.awt.Component;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JTree;
import javax.swing.JList;
import javax.swing.DefaultListModel;
import javax.swing.JPanel;
import javax.swing.JFrame;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.BorderFactory;
import javax.swing.SpringLayout;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;



/**
 * This class implements a simple panel for selecting classes from a
 * tree widget and adding them to a target list.  The panel uses drag
 * and drop events as well as basic buttons for adding and removing
 * classes. This panel also provides a simple term search mechanism.
 * 
 * @author Shawn Bowers
 */
public class OntoClassSelectionJPanel extends JPanel {


    /**
     * default constructor that initializes the panel
     */
    public OntoClassSelectionJPanel() {
	JScrollPane treeView = createTreeView();
	JScrollPane listView = createListView();

	JPanel panel1 = new JPanel();
	panel1.setLayout(new BoxLayout(panel1, BoxLayout.LINE_AXIS));
	panel1.add(_searchTxt);
	panel1.add(_searchBtn);
 	_searchBtn.addActionListener(new ClassSearchButtonListener());
 	_searchTxt.addActionListener(new ClassSearchButtonListener());

	JPanel panel2 = new JPanel();
	panel2.setLayout(new BoxLayout(panel2, BoxLayout.PAGE_AXIS));
	panel2.add(treeView);
	panel2.add(Box.createRigidArea(new Dimension(0,5)));
	panel2.add(panel1);

	JPanel panel3 = new JPanel();
	panel3.setLayout(new BoxLayout(panel3, BoxLayout.LINE_AXIS));
	panel3.add(Box.createHorizontalGlue());
	panel3.add(_addBtn);
	panel3.add(Box.createRigidArea(new Dimension(5,0)));
	panel3.add(_removeBtn);
 	_addBtn.addActionListener(new ClassAddButtonListener());
 	_removeBtn.addActionListener(new ClassRemoveButtonListener());

	JPanel panel4 = new JPanel();
	panel4.setLayout(new BoxLayout(panel4, BoxLayout.PAGE_AXIS));
	panel4.add(listView);
	panel4.add(Box.createRigidArea(new Dimension(0,5)));
	panel4.add(panel3);

	JPanel panel5 = new JPanel();
	panel5.setLayout(new BoxLayout(panel5, BoxLayout.LINE_AXIS));
	panel5.add(panel2);
	panel5.add(Box.createRigidArea(new Dimension(15,0)));
	panel5.add(panel4);

	add(panel5);
    }
    

    /**
     * initiliazes and creates the tree view sub-panel
     */
    private JScrollPane createTreeView() {
	// create the default root node
	DefaultMutableTreeNode rootNode = new DefaultMutableTreeNode("");
	// get each ont model checked on for the library
	Iterator ontModels = OntologyCatalog.instance().getLibraryNamedOntModels();
	while(ontModels.hasNext()) {
	    // add ontologies to root
	    NamedOntModel m = (NamedOntModel)ontModels.next();
	    DefaultMutableTreeNode childNode = new DefaultMutableTreeNode(m);
	    rootNode.add(childNode);
	    // get each root class of the model
	    Iterator rootClasses = m.getRootClasses(true);
	    while(rootClasses.hasNext()) {
		// build tree from the root
		NamedOntClass root = (NamedOntClass) rootClasses.next();
		buildTree(root, childNode);
	    }
	}

	// assign root to tree
	_ontoTree = new JTree(rootNode);
	// configure tree
	_ontoTree.setRootVisible(false);
        _ontoTree.setCellRenderer(new MyRenderer());
	// wrap tree in scroll pane
	return new JScrollPane(_ontoTree, 
			       JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, 
			       JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);	
    }

    /**
     * recursively initializes the tree 
     */
    private void buildTree(NamedOntClass c, DefaultMutableTreeNode parentNode) {
	// add the class to the parent
	DefaultMutableTreeNode childNode = new DefaultMutableTreeNode(c); 
	parentNode.add(childNode);
	// get subclasses
	Iterator subclasses = c.getNamedSubClasses(true);
	while(subclasses.hasNext()) {
	    NamedOntClass subclass = (NamedOntClass)subclasses.next();
	    buildTree(subclass, childNode);
	}
    }

    /**
     * private class for rendering the ontology tree. Uses different
     * icons for ontology nodes and class nodes.
     */
    private class MyRenderer extends DefaultTreeCellRenderer {
	private ImageIcon _classIcon;
	private ImageIcon _ontoIcon;

	/** initializes the renderer */
	public MyRenderer() {
	    _classIcon = new ImageIcon(CLASS_ICON);	    
	    _ontoIcon = new ImageIcon(ONTO_ICON);	    
	}

	public Component getTreeCellRendererComponent(JTree tree, Object value,
						      boolean sel, boolean expanded,
						      boolean leaf, int row,
						      boolean hasFocus) 
	{
	    super.getTreeCellRendererComponent(tree, value, sel,
					       expanded, leaf, row,
					       hasFocus);
	    if (isClassNode(value)) {
		setIcon(_classIcon);
		setToolTipText("This is a class ... ");
	    } 
	    else if (isOntoNode(value)) {
		setIcon(_ontoIcon);
		setToolTipText("This is an ontology ... ");
	    } 
	    return this;
	}
	
	/** @return True if the given node object is a NamedOntClass */
	protected boolean isClassNode(Object value) {
	    DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
	    Object obj = node.getUserObject();
	    if(obj instanceof NamedOntClass)
		return true;
	    return false;
	}

	/** @return True if the given node object is a NamedOntModel */
	protected boolean isOntoNode(Object value) {
	    DefaultMutableTreeNode node = (DefaultMutableTreeNode)value;
	    Object obj = node.getUserObject();
	    if(obj instanceof NamedOntModel)
		return true;
	    return false;
	}

    }; // inner class


    /**
     * initiliazes and creates the list sub-panel
     */
    private JScrollPane createListView() {
	_classList = new JList(new DefaultListModel());
	_classList.setFixedCellWidth(175);
	// wrap list in scroll pane
	return new JScrollPane(_classList, 
			       JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, 
			       JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);	
    }


    /**
     * provides access to the selected ontology classes
     * 
     * @return the current set of selections in the panel
     */ 
    public Vector getNamedOntClasses() {
	return getListAsVector();
    }


    /**
     * adds the given NamedOntClass objects to the panel.
     */
    public void addNamedOntClasses(NamedOntClass ontClass) {
	addToList(ontClass);
    }



    ////////////////////////////////////////////////////////////////////////
    ////  PRIVATE METHODS
    

    /** helper function to add ont classes to the list */
    private void addToList(NamedOntClass ontClass) {
	DefaultListModel mdl = (DefaultListModel)_classList.getModel();
	if(!mdl.contains(ontClass)) {
	    //String txt = ontClass.getOntologyName();
	    mdl.addElement(ontClass);
	}
    }

    /** helper function to return all items in the class list */
    private Vector getListAsVector() {
	DefaultListModel mdl = (DefaultListModel)_classList.getModel();
	Vector result = new Vector();
	for(int i = 0; i < mdl.getSize(); i++ )
	    result.add(mdl.getElementAt(i));
	return result;
    }

    /** helper function to return all items in the class list */
    private void removeFromList(NamedOntClass ontClass) {
	DefaultListModel mdl = (DefaultListModel)_classList.getModel();
	mdl.removeElement(ontClass);
    }

    /** helper function that returns tree nodes containing OntTreeNodes */
    private Vector findMatchingClasses(String str) {
	DefaultMutableTreeNode root = (DefaultMutableTreeNode)_ontoTree.getModel().getRoot();
	return findMatchingClasses(root, str);
    }

    /** helper function that returns tree nodes containing OntTreeNodes */
    private Vector findMatchingClasses(DefaultMutableTreeNode root, String str) {
	Vector result = new Vector();
	Object obj = root.getUserObject();
	if(obj instanceof NamedOntClass) {
	    NamedOntClass cls = (NamedOntClass)obj;
	    if(approxMatch(cls.getName(), str)) {
		result.add(root);
		return result;
	    }
	}
	Enumeration children = root.children();
	while(children.hasMoreElements()) {
	    DefaultMutableTreeNode child = (DefaultMutableTreeNode)children.nextElement();
	    Iterator ancestors = findMatchingClasses(child, str).iterator();
	    while(ancestors.hasNext()) 
		result.add(ancestors.next());
	}
	return result;
    }


    /**
     *  helper function for search
     *
     *@param  val1  first string to compare
     *@param  val2  second string to compare
     *@return       true if strings approximately match
     */
    private boolean approxMatch(String val1, String val2) {
	val1 = val1.toLowerCase();
	val2 = val2.toLowerCase();
	if (val1.indexOf(val2) != -1 || val2.indexOf(val1) != -1)
	    return true;
	return false;
    }
    

    /**
     * collapse the ontology tree to just the ontology nodes
     */
    private void collapseTree() {
	int row = _ontoTree.getRowCount() - 1;
	while (row >= 0) {
	    _ontoTree.collapseRow(row);
	    row--;
	}
    }


    ////////////////////////////////////////////////////////////////////////
    ////  LISTENERS


    /** listener for search button */
    private class ClassSearchButtonListener implements ActionListener {
 	public void actionPerformed(ActionEvent ev) {
	    // reset the selections 
	    _ontoTree.clearSelection();
	    // collapse the tree
	    collapseTree();
	    // get the search string
 	    String searchStr = _searchTxt.getText();
	    // if empty return
 	    if(searchStr.trim().equals(""))
 		return;
	    // get all the matches
  	    Iterator results = findMatchingClasses(searchStr).iterator();
	    while(results.hasNext()) {
		// add selection for each match
		DefaultMutableTreeNode node = (DefaultMutableTreeNode)results.next();
		_ontoTree.addSelectionPath(new TreePath(node.getPath()));
	    }
 	}
     }

    /** listener for add button */
    private class ClassAddButtonListener implements ActionListener {
 	public void actionPerformed(ActionEvent ev) {
	    // get the tree selections
	    TreePath[] paths = _ontoTree.getSelectionPaths();
	    for(int i = 0; i < paths.length; i++) {
		TreePath p = paths[i];
		DefaultMutableTreeNode node = (DefaultMutableTreeNode)p.getLastPathComponent();
		Object obj = node.getUserObject();
		if(obj instanceof NamedOntClass) {
		    addToList((NamedOntClass)obj);
		}
	    }
 	}
     }

    /** listener for remove button */
    private class ClassRemoveButtonListener implements ActionListener {
 	public void actionPerformed(ActionEvent ev) {
	    // get the list selections
	    Object[] items = _classList.getSelectedValues();
	    for(int i = 0; i < items.length; i++) {
		NamedOntClass ontoClass = (NamedOntClass)items[i];
		removeFromList(ontoClass);
	    }
 	}
     }



    ////////////////////////////////////////////////////////////////////////
    ////  PRIVATE ATTRIBUTES


    private Vector _semTypes = new Vector();

    private JTree _ontoTree;
    private JList _classList;
    private JTextField _searchTxt = new JTextField(16); 
    private JButton _searchBtn = new JButton("search"); 
    private JButton _addBtn = new JButton("add"); 
    private JButton _removeBtn = new JButton("remove"); 

    private String KEPLER = System.getProperty("KEPLER");
    private String CLASS_ICON = KEPLER + "/configs/ptolemy/configs/kepler/sms/class.png"; 
    private String ONTO_ICON = KEPLER + "/configs/ptolemy/configs/kepler/sms/onto.png"; 
    




    ////////////////////////////////////////////////////////////////////////
    ////  FOR TESTING

    public static void main(String [] args) {
	try {
	    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
	} catch (Exception e) { }

	OntoClassSelectionJPanel pane = new OntoClassSelectionJPanel();

	JFrame frame = new JFrame();
	frame.getContentPane().add(new OntoClassSelectionJPanel());
	frame.setTitle("Test Frame");
	frame.pack();
	frame.show();
    }//testing



    // javax.swing.AbstractionAction

    // org.kepler.gui.ActorDialogAction (for example)


} // OntoClassSelectionJPanel