/**
 *    '$RCSfile: PortSemanticTypeEditorPane.java,v $
 *
 *     '$Author: bowers $'
 *       '$Date: 2006/02/28 01:16:01 $
 *   '$Revision: 1.11 $'
 *
 *  For Details: http://kepler-project.org
 *
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms.gui;



import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreeSelectionModel;
import javax.swing.tree.DefaultTreeCellRenderer;

import org.kepler.sms.KeplerCompositeIOPort;
import org.kepler.sms.KeplerIOPortReference;
import org.kepler.sms.KeplerRefinementIOPort;
import org.kepler.sms.KeplerVirtualIOPort;

import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.type.Type;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.RecordType;
import ptolemy.data.type.ArrayType;
import ptolemy.kernel.Entity;
import ptolemy.kernel.util.NamedObj;


/**
 * 
 * @author Shawn Bowers
 */
public class PortSemanticTypeEditorPane extends JPanel {

    /**
     * Constructor
     */
    public PortSemanticTypeEditorPane(Frame owner, NamedObj namedObj) {
	super();
	_owner = owner;
	_namedObj = namedObj;
	
	JPanel pane = new JPanel();
	pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));
	pane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

	// do the tree part
	pane.add(create_portTrees());

	// add the semantic type 
	_semTypeTable = new SemanticTypeTable(owner);
	pane.add(_semTypeTable);

	// add the semantic link
	_semLinkTable = new SemanticLinkTable(owner);
	pane.add(_semLinkTable);

	add(pane);
    }


    /**
     * Performs the commit button operation.
     */
    public void doCommit() {
	_semTypeTable.commitAnnotationObjects();
    }
    

    /**
     * Performs the cancel button operation.
     */
    public void doClose() {
	_semTypeTable.dispose();
    }


    /**
     * @return True if any semantic type annotations have been modified.
     */
    public boolean hasModifiedSemTypes() {
	return _semTypeTable.hasModifiedAnnotationObjects();
    }
    

    /**
     * @return The first error if semantic type annotations are not
     * well formed, and null otherwise.
     */
    public String wellFormedSemTypes() {
	return _semTypeTable.wellFormedSemTypes();
    }


    /** 
     * @return True if there are any semantic types that are not in
     * the catalog.
     */
    public boolean hasUnknownSemTypes() {
	return _semTypeTable.hasUnknownSemTypes();
    }



    /**
     * 
     */
    private JPanel create_portTrees() {
	// outer pane
	JPanel pane = new JPanel();
	pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));

	// tree pane
	JPanel treePane = new JPanel();
	treePane.setLayout(new BoxLayout(treePane, BoxLayout.X_AXIS));

	// create the input tree
	_inputPortTree = new JTree(_create_input_model());
	//DefaultTreeCellRenderer renderer = new DefaultTreeCellRenderer();
	//Icon openIcon = renderer.getLeafIcon();
	//renderer.setOpenIcon(openIcon);
	//renderer.setClosedIcon(openIcon);
	//renderer.setLeafIcon(null);
	//_inputPortTree.setCellRenderer(renderer);

	_inputPortTree.setRootVisible(true);
	_inputPortTree.setEditable(false);
	_inputPortTree.addTreeSelectionListener( new TreeSelectionListener() {
		public void valueChanged(TreeSelectionEvent e) {
		    _PortTreeNode node = (_PortTreeNode)
			_inputPortTree.getLastSelectedPathComponent();
		    if (node == null) 
			return;
		    _inputSingleClick((_PortTreeNode)node);
		}
	    });
	_inputPortTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
	
	JScrollPane inputTreeView = new JScrollPane(_inputPortTree);
	JPanel inputTreePane = _labelComponent("Input Ports: ", inputTreeView, 225, 175);

	// create the output tree
	_outputPortTree = new JTree(_create_output_model());
	//_outputPortTree.setCellRenderer(renderer);
	//_outputPortTree.setRootVisible(false);
	//_outputPortTree.setEditable(false);

	_outputPortTree.setRootVisible(false);
	_outputPortTree.setEditable(false);
	//Listen for when the selection changes.
	_outputPortTree.addTreeSelectionListener( new TreeSelectionListener() {
		public void valueChanged(TreeSelectionEvent e) {
		    _PortTreeNode node = (_PortTreeNode)
			_outputPortTree.getLastSelectedPathComponent();
		    if (node == null) 
			return;
		    _outputSingleClick((_PortTreeNode)node);
		}
	    });
	_outputPortTree.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

	JScrollPane outputTreeView = new JScrollPane(_outputPortTree);
	JPanel outputTreePane = _labelComponent("Output Ports: ", outputTreeView, 225, 175);

	// add input and output tree to treepane
	treePane.add(inputTreePane);
	treePane.add(Box.createRigidArea(new Dimension(10,0)));	
	treePane.add(outputTreePane);

	// button pane
	JPanel buttonPane = new JPanel();
	buttonPane.setLayout(new BoxLayout(buttonPane, BoxLayout.X_AXIS));
	buttonPane.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

	// create the buttons
	JButton generalizeBtn = new JButton("Create Composite Port");
	generalizeBtn.setActionCommand("generalize");
	generalizeBtn.addActionListener(_buttonListener);
	generalizeBtn.setToolTipText("Create a group of ports");

	JButton portRemoveBtn = new JButton("Remove Composite Port");
	portRemoveBtn.setEnabled(false);
	portRemoveBtn.setActionCommand("remove");
	portRemoveBtn.addActionListener(_buttonListener);
	portRemoveBtn.setToolTipText("Remove a virtual port");

	// add buttons
	buttonPane.add(Box.createHorizontalGlue());
	buttonPane.add(generalizeBtn);
	//buttonPane.add(Box.createRigidArea(new Dimension(5, 0)));
	//buttonPane.add(refineBtn);	
	buttonPane.add(Box.createRigidArea(new Dimension(5, 0)));
	buttonPane.add(portRemoveBtn);

	// add intermediate panes to outer pane
	pane.add(treePane);
	pane.add(Box.createRigidArea(new Dimension(0, 5)));
	pane.add(buttonPane);
	// return it
	return pane;
    }


    /**
     * This is called in the composite dialog
     */
    protected DefaultTreeModel _create_input_model() {
	_PortTreeNode root = new _PortTreeNode(null);
	_inputTreeModel = new DefaultTreeModel(root);
	
	for(Iterator iter = _namedObj.containedObjectsIterator(); iter.hasNext(); ) {
	    Object obj = iter.next();
	    if(obj instanceof TypedIOPort) {
		TypedIOPort port = (TypedIOPort)obj;
		if(port.isInput()) {
		    _PortTreeNode node = new _PortTreeNode(port);
		    root.addChild(node);
		    Type t = port.getType();
		    if(t instanceof RecordType) {
			//_addSubComponents(node, (RecordType)t);
		    }
		    else if(t instanceof ArrayType) {
		    }
		    
		}
	    }
	    else if(obj instanceof KeplerCompositeIOPort) {
		KeplerCompositeIOPort port = (KeplerCompositeIOPort)obj;
		if(port.isInput()) {
		    _PortTreeNode node = new _PortTreeNode(port);
		    root.addChild(node);
		    // get the "children" and add ... 
		    Iterator refs = port.getEncapsulatedPortReferences().iterator();
		    while(refs.hasNext()) {
			KeplerIOPortReference ref = (KeplerIOPortReference)refs.next();
			_PortTreeNode ref_node = new _PortTreeNode(ref.getPort());
			node.addChild(ref_node);
		    }
		}
	    } 
	}
	
	return _inputTreeModel;
    }


    /**
     * This is called in the composite dialog
     * Need to figure out how this should look ...
     */
    //   1. Need to list (alphabetically) ports first
    //   2. Then list (alphabetically) the generalized ports
    protected DefaultTreeModel _create_output_model() {
	_PortTreeNode root = new _PortTreeNode("");
	_outputTreeModel = new DefaultTreeModel(root);

	// get all the ports and refinement port pointers
	Vector ports = new Vector();
	Vector refinePortNames = new Vector();
	Iterator iter = _namedObj.containedObjectsIterator();
	while(iter.hasNext()) {
	    Object obj = iter.next();
	    if(obj instanceof KeplerRefinementIOPort)
		refinePortNames.add(((KeplerRefinementIOPort)obj).getPointer());
	    else
		ports.add(obj);
	}

	iter = ports.iterator();
	while(iter.hasNext()) {
	    Object obj = iter.next();
	    if(obj instanceof TypedIOPort) {
		TypedIOPort port = (TypedIOPort)obj;
		if(port.isOutput()) {
		    _PortTreeNode node = new _PortTreeNode(port);
		    root.addChild(node);
		    Type t = port.getType();
		    if(t instanceof RecordType) {
			_addRefinementPorts(node, (RecordType)t, refinePortNames);
		    }
		    else if(t instanceof ArrayType) {
		    }
		}
		// _create_sub_ports(_namedObj, node)
	    }
	    else if(obj instanceof KeplerCompositeIOPort) {
		KeplerCompositeIOPort port = (KeplerCompositeIOPort)obj;
		if(port.isOutput()) {
		    _PortTreeNode node = new _PortTreeNode(port);
		    root.addChild(node);
		    // get the "children" and add ... 
		    Iterator refs = port.getEncapsulatedPortReferences().iterator();
		    while(refs.hasNext()) {
			KeplerIOPortReference ref = (KeplerIOPortReference)refs.next();
			_PortTreeNode ref_node = new _PortTreeNode(ref.getPort());
			node.addChild(ref_node);
		    }
		}
	    } 
	}

	return _outputTreeModel;
    }


    /**
     *
     */
    private void _addRefinementPorts(_PortTreeNode parent, RecordType t, Vector refinePortNames) {
	Iterator iter = t.labelSet().iterator(); // get record component labels
	while(iter.hasNext()) {
	    String label = (String)iter.next();
	    Object obj = parent.getPort();
	    String portname = null;
	    if(obj instanceof IOPort)
		portname = ((IOPort)obj).getName();
	    else if(obj instanceof KeplerVirtualIOPort)
		portname = ((KeplerVirtualIOPort)obj).getName();
	    String pointer = portname + "/" + label;
	    // check if exists already as a port
	    if(!refinePortNames.contains(pointer)) {
		try {
		    KeplerRefinementIOPort p = new KeplerRefinementIOPort(_namedObj, pointer);
		    _PortTreeNode node = new _PortTreeNode(p);
		    parent.addChild(node);
		    if(p.getType() instanceof RecordType) {
			_addRefinementPorts(node, (RecordType)p.getType(), refinePortNames);
		    }
		} catch(Exception e) {
		    e.printStackTrace();
		}
	    }
	}
    }
    


    /**
     * 
     */
    private void _doGeneralize() {
	KeplerCompositeIOPort newPort;
	newPort = CompositePortDialog.showDialog(_owner, (Entity)_namedObj);
	if(newPort == null)
	    return;
	if(newPort.isInput()) {
	    _inputPortTree.setModel(_create_input_model());
	}
	else if(newPort.isOutput()) {
	    _outputPortTree.setModel(_create_output_model());
	}
    }


    /**
     * 
     */
    private void _doRefine() {
    }


    /**
     * 
     */    
    private void _doRemove() {
    }



    /**
     *
     */
    private void _inputSingleClick(_PortTreeNode node) {
	_outputPortTree.setSelectionRow(-1);
	_selectedNode = node;
	_updateGUI();
    }

    /**
     *
     */
    private void _outputSingleClick(_PortTreeNode node) {
	_inputPortTree.setSelectionRow(-1);
	_selectedNode = node;
	_updateGUI();
    }

    /**
     *
     */
    private void _updateGUI() {
	Object obj = _selectedNode.getPort();
	if(obj instanceof IOPort) {
	    IOPort port = (IOPort)obj;
	    _semTypeTable.addAnnotationObject(port);
	    _semTypeTable.setAnnotationObjectVisible(port);
	    _semLinkTable.addLinkDomainObject(port);
	    _semLinkTable.setLinkDomainObjectVisible(port);
	}
	else if(obj instanceof KeplerVirtualIOPort) {
	    KeplerVirtualIOPort port = (KeplerVirtualIOPort)obj;
	    _semTypeTable.addAnnotationObject(port);
	    _semTypeTable.setAnnotationObjectVisible(port);
	    _semLinkTable.addLinkDomainObject(port);
	    _semLinkTable.setLinkDomainObjectVisible(port);
	}
    }


    /**
     * Inner class for tree nodes
     */
    private class _PortTreeNode implements TreeNode {
	/** constructor */
	public _PortTreeNode(Object port) {
	    _port = port;
	}

	/** @return The port encapsulated in the node */
	public Object getPort() { 
	    return _port; 
	}

	/** Add given node as child */
	public void addChild(_PortTreeNode child) { 
	    if(!_children.contains(child)) {
		_children.add(child);
		child.removeFromParent();
		child.setParent(this);
	    }
	}

	/** Set the parent of this node */
	public void setParent(_PortTreeNode parent) {
	    _parent = parent;
	}

	/** Remove the parent of this node */
	public void removeFromParent() { 
	    _parent = null; 
	}

	/** @return The children of this node */
	public Enumeration children() {
	    return _children.elements();
	}

	/** @return Always true. */
	public boolean getAllowsChildren() { 
	    return true; 
	}

	/** @return The ith tree node */
	public TreeNode getChildAt(int index) {
	    if(index < _children.size() && index >= 0 ) 
		return (TreeNode)_children.elementAt(index);
	    return null;
	}

	/** @return the index of the child node */
	public int getIndex(TreeNode node) {
	    return _children.indexOf(node);
	}

	/** @return The number of children */
	public int getChildCount() {
	    return _children.size();
	}
	
	/** @return The parent of this node */
	public TreeNode getParent() { 
	    return _parent; 
	}

	/** @return True if no children */
	public boolean isLeaf() {
	    return _children.size() == 0;
	}
	
	public String toString() { 
	    String str = "";
	    if(_port instanceof IOPort)
		str += ((IOPort)_port).getName() + "  (";
	    else if(_port instanceof KeplerCompositeIOPort)
		str += ((KeplerCompositeIOPort)_port).getName() + "  (";
	    else if(_port instanceof KeplerRefinementIOPort)
		str += ((KeplerRefinementIOPort)_port).getName() + "  (";	    

	    if(_port instanceof TypedIOPort) {
		Type type = ((TypedIOPort)_port).getType();
		if(type instanceof BaseType)
		    str += type;
		else if(type instanceof RecordType)
		    str += "record";
		else if(type instanceof ArrayType)
		    str += "array";
		else
		    str += "other";
	    }
	    else if(_port instanceof KeplerVirtualIOPort) {
		Type type = ((KeplerVirtualIOPort)_port).getType();
		if(type instanceof BaseType)
		    str += type;
		else if(type instanceof RecordType)
		    str += "record";
		else if(type instanceof ArrayType)
		    str += "array";
		else
		    str += "other";
	    }
	    else
		str += "unknown";
	    return str + ")";
	}


	/* Private members */
	private Object _port;
	private Vector _children = new Vector();
	private _PortTreeNode _parent; 

    };



    /**
     * Given a label string, component, height, and width, creates a
     * new panel with a label and the component of size height and
     * width. The label is positioned above the component and is left
     * justified.
     */
    private JPanel _labelComponent(String str, Component component, int width, int height) {
	// output pane
	JPanel pane = new JPanel();
	pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));	
	// pane for label only
	JPanel labelPane = new JPanel();
	labelPane.setLayout(new BoxLayout(labelPane, BoxLayout.X_AXIS));
	labelPane.add(new JLabel(str, SwingConstants.LEFT));
	labelPane.add(Box.createHorizontalGlue());
	// add label
	pane.add(labelPane);
	// add space
	pane.add(Box.createRigidArea(new Dimension(0, 5)));
	// add component
	pane.add(component);
	// set sizes
	pane.setMaximumSize(new Dimension(width, height));
	pane.setMinimumSize(new Dimension(width, height));
	pane.setPreferredSize(new Dimension(width, height));
	// return outer pane
	return pane;
    }

    /**
     * anonymous class to handle button events
     */
    private ActionListener _buttonListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		if(e.getActionCommand().equals("generalize")) 
		    _doGeneralize();
		else if(e.getActionCommand().equals("refine")) 
		    _doRefine();
		else if(e.getActionCommand().equals("remove")) 
		    _doRemove();
	    }
	};



    /* Private Members */

    private JTree _inputPortTree;
    private JTree _outputPortTree;
    private DefaultTreeModel _inputTreeModel;
    private DefaultTreeModel _outputTreeModel;

    private _PortTreeNode _selectedNode;


    private SemanticTypeTable _semTypeTable;
    private SemanticLinkTable _semLinkTable;

    private Frame _owner;
    private NamedObj _namedObj;


} // PortSemanticTypeEditorPan
