/* Base class for collection types.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp;

import org.nddp.tokens.ClosingDelimiterToken;
import org.nddp.tokens.OpeningDelimiterToken;

/**
 * Class representing a collection of tokens.  In the current implementation,
 * Collection also serves as the abstract base class for collection types. 
 * <p>
 * Note that an instance of Collection does not contain references to every 
 * token contained by the corresponding token collection. It only contains 
 * references to the {@link org.nddp.tokens.OpeningDelimiterToken} and 
 * {@link org.nddp.tokens.ClosingDelimiterToken} objects that 
 * logically denote the collection in the token stream. Other tokens belong 
 * to the collection if and only if they occur in the sequence of tokens
 * between the collection's delimiter tokens.
 * This approach allows Collection objects to be immutable. Because 
 * Collection objects are immutable, they may be passed freely between 
 * and accessed concurrently by actors without danger.
 * <p>
 * Certain tokens store references to the Collection object corresponding to
 * the collection they are in, allowing the {@link org.nddp.AtomicCoactor} 
 * base class to verify that these tokens occur properly nested between the 
 * collection's opening and closing delimiters.  
 * <p>
 * In addition to references to the opening and closing delimiters, a 
 * Collection object stores the name of the collection it represents.  
 * The name need not be unique and may be set to the empty string if desired.
 * <p>
 * 
 *  @author Timothy M. McPhillips
 */
public class Collection {

    /** 
     * Constructs an instance of Collection with a name equal to the empty string. 
     * 
     * @param type The type of the collection.
     * not <code>null</code>.
     */
    public Collection() {

        // create opening and closing delimiters for the collection
		openingDelimiter = new OpeningDelimiterToken(this);
		closingDelimiter = new ClosingDelimiterToken(this);		
    }
    
    /** 
     * Constructs an instance of Collection with the specified name. 
     * 
     * @param collectionName Name of this collection.  If equal to the null string, 
     * an value equal to the empty string is assigned to name. 
     */
	public Collection(String collectionName) {
	    
		this();

		if (collectionName != null) {
			name = collectionName;
		} else {
			name = "";
		}
	}
	
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    /** 
     * Returns the closing delimiter for this collection.
     * 
     * @return The immutable {@link org.nddp.tokens.ClosingDelimiterToken}
     * object that represents the closing delimiter for this collection.
     */	
	public ClosingDelimiterToken closingDelimiter() {
		return closingDelimiter;
	}
	
	/** 
     * Returns the name of this collection.
     * 
     * @return The name of this collection as a String.
     */	
	public String name() {
		return name;
	}
	
	/** 
     * Returns the type of this collection.
     * 
     * @return The type of this collection as a String.
     */	
	public String type() {
	    return "Collection";
	}
	
    /** 
     * Returns the opening delimiter for this collection.
     * 
     * @return The immutable {@link org.nddp.tokens.OpeningDelimiterToken} 
     * object that represents the opening delimiter for this collection.
     */	
	public OpeningDelimiterToken openingDelimiter() {
		return openingDelimiter;
	}
	
	/** 
     * Returns a string representation of this collection.
     * 
     * @return A String comprising the type of this collection and the 
     * collection name if the latter is not an empty string.
     */	
	public String toString() {
		
	    // if the name is the empty string, return  just the name of the 
	    // collection type
		if ( name.equals("") ) {
			return type();
			
		// otherwise return a string containing both the collection type 
		// and name
		} else {
			return type() + " name=\"" + name + "\"";
		}
	}
		
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	/** The closing delimiter for this collection. */
	private final ClosingDelimiterToken closingDelimiter;
	
	/** The name of this collection. */
	private String name = "";
	
	/** The opening delimiter for this collection. */
	private final OpeningDelimiterToken openingDelimiter;
}
