/* Class representing a collection path.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ptolemy.data.StringToken;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.util.IllegalActionException;

/** 
 @author Timothy M. McPhillips
 */

public class CollectionPath {
    
    // add JUnit tests
    public CollectionPath(Parameter path) throws IllegalActionException {
        
        this(((StringToken)(path.getToken())).stringValue());
    }

    public CollectionPath(String path) throws IllegalActionException {
        
        	super();

        	// trim whitespace from both ends of path
        	path = path.trim();
        	
		Matcher matcher;

        // extract and remove options string if present
        matcher = OPTIONS_PATTERN.matcher(path);
        if (matcher.find()) {
            path = matcher.group(1);
            String options = matcher.group(3);
            
            if (options.indexOf("delete") != -1) {
                _deleteScope = true;
            }
        }
        
        // make sure path contains no white space
        	matcher = WHITE_SPACE_PATTERN.matcher(path);
        	if (matcher.find()) {
        	    throw new IllegalActionException("No white space is allowed " +
    	            "in collection path: " + path);
        	}
            
		// return in default state if null is empty string
		if (path.equals("")) {
		    return;
		}

		matcher = C_SLASH_SLASH_D_SLASH_SLASH_C.matcher(path);
		if (matcher.matches()) {
			parseCollectionSlashSlashDataSlashSlashCollection(matcher, path);
			return;
		}

		matcher = C_OR_D_SLASH_SLASH_C_OR_D.matcher(path);
		if (matcher.matches()) {
			parseCollectionOrDataSlashSlashCollectionOrData(matcher, path);
			return;
		}

		matcher = SLASH_SLASH_C_OR_D.matcher(path);
		if (matcher.matches()) {
			parseSlashSlashCollectionOrData(matcher, path);
			return;
		}

		matcher = SLASH_SLASH_D_SLASH_SLASH_C.matcher(path);
		if (matcher.matches()) {
			parseSlashSlashDataSlashSlashCollection(matcher, path);
			return;
		}
		
		matcher = C_SLASH_SLASH.matcher(path);
		if (matcher.matches()) {
			parseCollectionSlashSlash(matcher, path);
			return;
		}

		matcher = C_SLASH_D.matcher(path);
		if (matcher.matches()) {
			parseCollectionSlashData(matcher, path);
			return;
		}

		matcher = C_SLASH.matcher(path);
		if (matcher.matches()) {
			parseCollectionSlash(matcher, path);
			return;
		}

		matcher = C_OR_D.matcher(path);
		if (matcher.matches()) {
			parseCollectionOrData(matcher, path);
			return;
		}
		
		throw new IllegalActionException(
	        "Path " + path + " does not match expected format.");
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    ////

    public final boolean deleteScope() {
        return _deleteScope;
    }

    public final boolean exposeNonDataEvents() {
        return exposeNonDataEvents;
    }

	public final Class ignoreCollectionsInside() {
        return ignoreCollectionsInside;
    }

    public final Class ignoreCollectionsOutside() {
        return ignoreCollectionsOutside;
    }
    
    public final Class targetDataType() {
        return targetDataType;
    }
    
	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

    private void parseCollectionSlash(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    ignoreCollectionsOutside = CollectionTypes.valueOfString(matcher.group(1));
	    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(1));
	}

	private void parseCollectionSlashData(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    ignoreCollectionsOutside = CollectionTypes.valueOfString(matcher.group(1));
	    targetDataType = DataTypes.valueOfString(matcher.group(2));
	    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(1));
	}
	
	private void parseCollectionSlashSlash(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    ignoreCollectionsOutside = CollectionTypes.valueOfString(matcher.group(1));
	}
	
	private void parseCollectionSlashSlashDataSlashSlashCollection(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    ignoreCollectionsOutside = CollectionTypes.valueOfString(matcher.group(1));
	    targetDataType = DataTypes.valueOfString(matcher.group(2));
	    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(3));
	}

	private void parseSlashSlashDataSlashSlashCollection(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    targetDataType = DataTypes.valueOfString(matcher.group(1));
	    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(2));
	}
	
	private void parseCollectionOrDataSlashSlashCollectionOrData(
        Matcher matcher, String path) throws IllegalActionException {
	    	    
	    if (CollectionTypes.isDefined(matcher.group(1))) {
	        
	        parseCollectionSlashSlashCollectionOrData(matcher, path);
	    
	    } else if (DataTypes.isDefined(matcher.group(1))) {

	        parseDataSlashSlashCollection(matcher, path);
	        
	    } else {

	        throw new IllegalActionException(
    		        "No collection or data type " + matcher.group(1) + ".");
	    }
	}
	
	private void parseDataSlashSlashCollection(
        Matcher matcher, String path) throws IllegalActionException {

	    targetDataType = DataTypes.valueOfString(matcher.group(1));
	    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(2));
	    exposeNonDataEvents = false;
	}
	
	private void parseCollectionSlashSlashCollectionOrData(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    ignoreCollectionsOutside = CollectionTypes.valueOfString(matcher.group(1));
	    
	    if (CollectionTypes.isDefined(matcher.group(2))) {
	        
		    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(2));
		    
	    } else if (DataTypes.isDefined(matcher.group(2))) {
	        
		    targetDataType = DataTypes.valueOfString(matcher.group(2));
		    
	    } else {

	        throw new IllegalActionException(
    		        "No collection or data type " + matcher.group(2) + ".");
	    }
	}	
	
	private void parseSlashSlashCollectionOrData(
        Matcher matcher, String path) throws IllegalActionException {
	    
	    if (CollectionTypes.isDefined(matcher.group(1))) {
	        
		    ignoreCollectionsInside = CollectionTypes.valueOfString(matcher.group(1));
		    
	    } else if (DataTypes.isDefined(matcher.group(1))) {
	        
		    targetDataType = DataTypes.valueOfString(matcher.group(1));
		    
	    } else {

	        throw new IllegalActionException(
    		        "No collection or data type " + matcher.group(1) + ".");
	    }
	}
	
    private void parseCollectionOrData(Matcher matcher, String path) 
    		throws IllegalActionException {
    
    	    if (CollectionTypes.isDefined(matcher.group(1))) {
    	        
        	    ignoreCollectionsInside = 
    	        ignoreCollectionsOutside = 
        	        CollectionTypes.valueOfString(matcher.group(1));
        	    
        	    targetDataType = DataTypes.NoMatch.class;
        	    
    		} else if (DataTypes.isDefined(matcher.group(1))) {
    		    
    	    		targetDataType = DataTypes.valueOfString(matcher.group(1));
    	    		exposeNonDataEvents = false;
    	    		
    		} else {
    		    
	    		throw new IllegalActionException(
    		        "No collection or data type " + matcher.group(1) + ".");
    		}
    }

  	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
    
    private static final Pattern OPTIONS_PATTERN = Pattern.compile("\\s*([^\\[\\s]+)\\s*(\\[([^\\]]+)\\])\\s*");
    
    private static final Pattern C_OR_D = 	Pattern.compile("(\\w+)");

    private static final Pattern C_SLASH = Pattern.compile("(\\w+)/");

    private static final Pattern C_SLASH_D = Pattern.compile("(\\w+)/(\\w+)");

    private static final Pattern C_SLASH_SLASH = Pattern.compile("(\\w+)//");

    private static final Pattern C_OR_D_SLASH_SLASH_C_OR_D = 
        Pattern.compile("(\\w+)//(\\w+)");

    private static final Pattern C_SLASH_SLASH_D_SLASH_SLASH_C = 
        	Pattern.compile("(\\w+)//(\\w+)//(\\w+)");

    private static final Pattern SLASH_SLASH_D_SLASH_SLASH_C = 
        	Pattern.compile("//(\\w+)//(\\w+)");

    private static final Pattern SLASH_SLASH_C_OR_D = Pattern.compile("//(\\w+)");

    private static final Pattern WHITE_SPACE_PATTERN = Pattern.compile("\\s+");
    
    private boolean _deleteScope = false;
    private Class ignoreCollectionsInside = Object.class;
    private Class ignoreCollectionsOutside = Object.class;
    private Class targetDataType = Object.class;
    private boolean exposeNonDataEvents = true;
}
