/* A file-reading actor that reads file paths from its input port.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors;

import java.io.BufferedReader;
import java.io.IOException;

import org.nddp.util.FileOrUrl;
import org.nddp.util.Port;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="general"
 *
 *  @author Timothy M. McPhillips
 */

public class FileReader extends TypedAtomicActor {

    public FileReader(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an input port for receiving a reference to the process environment
        namePort = new TypedIOPort(this, "name", true, false);
        namePort.setTypeEquals(BaseType.STRING);

        // create an output port for forwarding a reference to the process environment
        contentPort = new TypedIOPort(this, "content", false, true);
        contentPort.setTypeEquals(BaseType.STRING);
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort namePort;
    public TypedIOPort contentPort;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    public void fire() throws IllegalActionException {
        
        super.fire();
        
        // read the next file name from the input port       
        String fileName = Port.getString(namePort, 0, false);

        if (fileName == null) {
            return;
        }
        
        // open the file
        FileOrUrl fileOrUrl;
        BufferedReader reader;
        
        try {

            fileOrUrl = new FileOrUrl(fileName);
            reader = fileOrUrl.open();
             
        } catch (NameDuplicationException exception) {
            throw new IllegalActionException(
                "Name duplication while opening file " + fileName + 
                ", " + exception.getMessage());
        }

        // TODO handle int overflow on conversion from long
        StringBuffer fileContents = new StringBuffer((int) fileOrUrl.length());
        
        try {
            // read file line by line, adding each line to the file collection 
            // as a data token
            String line;
            while ( (line = reader.readLine()) != null) {
                fileContents.append(line + "\n");
            }
    
            // close the file
            reader.close();
        
        } catch (IOException exception) {
        
            throw new IllegalActionException(
                "IOException while reading file " + fileName + 
                ", " + exception.getMessage());
        }    
        
        contentPort.broadcast(new StringToken(fileContents.toString()));  
 
    }
  
    private static final long serialVersionUID = 1L;
}
