/* Wrapper actor for Phylip CONSENSE.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
package org.nddp.actors;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.JFrame;
import javax.swing.JPanel;

import org.apache.batik.swing.JSVGCanvas;
import org.apache.batik.swing.gvt.GVTTreeRendererAdapter;
import org.apache.batik.swing.gvt.GVTTreeRendererEvent;

import org.nddp.util.Parameters;
import org.nddp.util.ProcessEnvironment;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="general"
 */
public class SvgViewer extends TypedAtomicActor {

    public SvgViewer(CompositeEntity container, String name)
        	throws NameDuplicationException, IllegalActionException {
        
        	super(container, name);
        
        // create an input port for receiving a reference to the process environment
        environmentIn = new TypedIOPort(this, "environmentIn", true, false);
        environmentIn.setTypeEquals(BaseType.OBJECT);

        // create an output port for forwarding a reference to the process environment
        environmentOut = new TypedIOPort(this, "environmentOut", false, true);
        environmentOut.setTypeEquals(BaseType.OBJECT);

        windowTitle = 
            Parameters.quotedStringParameter(this, "windowTitle", "Drawgram");
    }
    
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    public TypedIOPort environmentIn;
    public TypedIOPort environmentOut;

    public Parameter windowTitle;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void initialize() throws IllegalActionException {
    
        super.initialize();

        // start up the graphical user interface if it is not running
        if (gui.isRunning() == false) {
            gui.start();
        }
    }

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public void fire() throws IllegalActionException {
        
        // call superclass method
        super.fire();
        
        // receive one ObjectToken token on the environmentIn port
        ObjectToken environmentToken = (ObjectToken) environmentIn.get(0);
        
        // extract the reference to the process environment
        ProcessEnvironment processEnvironment = (ProcessEnvironment)environmentToken.getValue();

        // construct path to the svg file to draw
        String svgFileURI = "file:" + processEnvironment.workingDirectoryPath() + "/plotfile";
        
        gui.loadSvgFile(svgFileURI);

        // forward the process environment
        environmentOut.broadcast(environmentToken);
    } 

    public final void attributeChanged(Attribute attribute) 
        throws IllegalActionException {

        if (attribute == windowTitle) {
            _windowTitle = Parameters.stringValue(windowTitle);
            gui.resetTitle();
        } else {
            super.attributeChanged(attribute);
        }
    }    

    ///////////////////////////////////////////////////////////////////
	////                       protected methods                   ////

    private Gui gui = new Gui();
    private static final long serialVersionUID = 1L;
    private String _windowTitle;
	
	public class Gui {
	
	    public Gui() {
	    }
	    	
	    public boolean isRunning() {
		    return _isRunning;
		}
		
 		private boolean _isRunning = false;    
	   	private JFrame frame;
	    	private JSVGCanvas svgCanvas;
	
	    	public void start() {
	        
	        	frame = new JFrame(_windowTitle);
	    		svgCanvas = new JSVGCanvas();
	    	    JPanel panel = new JPanel(new BorderLayout());	
	        new JPanel(new FlowLayout(FlowLayout.LEFT));
	        panel.add("Center", svgCanvas);
	        	frame.getContentPane().add(panel);
	
	        	frame.addWindowListener(new WindowAdapter() {
	            public void windowClosing(WindowEvent e) {
	                _isRunning = false;
	            }
	        });

	        frame.setSize(400, 400);
	        frame.setVisible(true);
	        
            svgCanvas.addGVTTreeRendererListener(new GVTTreeRendererAdapter() {
	            public void gvtRenderingPrepare(GVTTreeRendererEvent e) {
	            }
	            public void gvtRenderingCompleted(GVTTreeRendererEvent e) {
	                synchronized(svgCanvas) {
	                    svgCanvas.notify();
	                }
	            }
	        });

            _isRunning = true;
	    	}

        public void resetTitle() {
            if (_isRunning) {
                frame.setTitle(_windowTitle);
            }
        }
            
	    	public void loadSvgFile(String path) {

	    	    svgCanvas.setURI(path);
			
			try {
		    	    synchronized(svgCanvas) {
					svgCanvas.wait();
				}
	        } catch (InterruptedException e) {
				// discard exceptions
	        }
	    	}
	}
}
