/* Write out a Phylip character weights file.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylip;

import java.io.IOException;

import org.nddp.actors.EnvironmentActor;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.WeightVector;
import org.nddp.phylogeny.phylip.PhylipFiles;
import org.nddp.util.Parameters;
import org.nddp.util.Port;
import org.nddp.util.ProcessEnvironment;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class WritePhylipWeights extends EnvironmentActor {

    public WritePhylipWeights(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an input port for receiving a weight vector
        weightsIn = new TypedIOPort(this, "weightsIn", true, false);
        weightsIn.setTypeEquals(BaseType.OBJECT);

        // create an input port for receiving a character matrix
        characterMatrix = new TypedIOPort(this, "characterMatrix", true, false);
        characterMatrix.setTypeEquals(BaseType.OBJECT);

        // create an input port for sending a weight vector
        weightsOut = new TypedIOPort(this, "weightsOut", false, true);
        weightsOut.setTypeEquals(BaseType.OBJECT);

        // create a parameter for optionally specifying the name of the trees file
        fileName = 
            Parameters.stringParameter(this, "fileName", "weights");
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort weightsIn;
    public TypedIOPort characterMatrix;
    public TypedIOPort weightsOut;
    public Parameter fileName;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public final void attributeChanged(Attribute attribute) 
        throws IllegalActionException {
        
        if (attribute == fileName) {
        
            _fileName = Parameters.stringValue(fileName);
    
        } else {
            super.attributeChanged(attribute);
        }
    }        
    public void fire() throws IllegalActionException {
 
        // call superclass method
        super.fire();
        
        // get a reference to the process environment
        ProcessEnvironment processEnvironment = receiveEnvironment();
    
        // read the weight vector from the input port
        WeightVector weightVector = null;
        
        // try to read an object token containing a weight vector from weightsIn
        ObjectToken weightToken = Port.getObjectToken(weightsIn, 0, false);
        
        // try to read an characterMatrix from the port of the same name
        CharacterMatrix matrix = (CharacterMatrix) Port.getObject(characterMatrix, 0, false);
        
        // create a new weight vector if none was received
        if (weightToken == null) {

            // require that a characterMatrix be received if no weight vector was received
            if (matrix == null) {
                throw new IllegalActionException("CharacterMatrix is required when no WeightVector is given.");
            }
            
            // derive the size of the weight vector from the character matrix column count
            weightVector = new WeightVector(matrix.characterCount());
            
        } else {

            // if a weight vector was recieved extract it from the object token
            weightVector = (WeightVector)weightToken.getValue();
            
            // make sure weight vector is correct size for character matrix (if present)
            if (matrix != null && weightVector.size() != matrix.characterCount()) {
                throw new IllegalActionException("WeightVector size does not match CharacterMatrix character count");
            }
        }
        
        // write the weights file for Phylip
        try {
            PhylipFiles.writeWeightsFile(
                weightVector, 
                processEnvironment.createInputFile(_fileName));
        } catch (IOException ex) {
            throw new IllegalActionException(
                    "Error writing weights file");
        } catch (ExternalEnvironmentException e) {
            throw new IllegalActionException("Error writing infile");
        }

        // send the weight vector if it was created de novo
        if (weightToken == null) {
            weightsOut.send(0, new ObjectToken(weightVector));
        }

        // forward the process environment
        sendEnvironment();
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                    private variables                      ////

    private String _fileName;
    private static final long serialVersionUID = 1L;
}
