/* Base class for actors that parse blocks of Nexus-formatted strings.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylogeny;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.nddp.actors.OptionalActor;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

public abstract class NexusParsingActor extends OptionalActor {

    public NexusParsingActor(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an input port for receiving the contents of a Nexus file
        nexusFilePort = new TypedIOPort(this, "nexusFile", true, false);
        nexusFilePort.setTypeEquals(BaseType.STRING);
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort nexusFilePort;

    ///////////////////////////////////////////////////////////////////
    ////                      protected methods                    ////
        
    protected String extractNexusBlock(String nexusFileString) {

        // look for a Nexus block matching the specified pattern
        Matcher matcher = _blockPattern.matcher(nexusFileString);

        if (matcher.find()) {
            
            // extract the requested block
            String blockString = matcher.group(2);
    
            // remove comment blocks
            blockString = 
                NEXUS_COMMENT_PATTERN.matcher(blockString).replaceAll("");
            
            // remove whitespace around = signs
            blockString = 
                NEXUS_EQUALS_PATTERN.matcher(blockString).replaceAll("=");
            
            // parse the characters block if found
            return blockString;
        
        } else {
            
            return null;
        }
    }

    protected void setBlockName(String blockName) {
        
        _blockPattern =  
            Pattern.compile("(BEGIN\\s+" + blockName + "(.*?)END\\s*?;)", 
                    Pattern.DOTALL | Pattern.CASE_INSENSITIVE);

    }

    protected static Map _parseCharacterPropertyList(String valueString) {
        
        // create a map to store properties in
            Map propertyMap = new HashMap();
        
            // remove whitespace around dashes
            valueString = 
                NEXUS_DASH_PATTERN.matcher(valueString).replaceAll("-");
    
            // find the property assignment clauses
            Matcher propertyMatcher = 
                NEXUS_CHARACTER_PROPERTY_PATTERN.matcher(valueString);
    
            // iterate over property assignment clauses
            while (propertyMatcher.find()) {
            
                // extract the property value 
                String propertyString = propertyMatcher.group(1).trim();
        
                // extract the character indices 
                List indexList = _parseCharacterGroup(
                        propertyMatcher.group(2));
    
                // iterator over the character indices in the list
                for (Iterator i = indexList.iterator(); i.hasNext(); ) {
            
                    // extract the index
                    Integer index = (Integer) i.next();
            
                    // map the index to the property value
                    propertyMap.put(index, propertyString);
                }
            }
    
            return propertyMap;
    }

    protected static List _parseCharacterGroup(String characterGroup) {
        
        // create a list to store the indices
        List indexList = new LinkedList();
        
        // break the index string into tokens
        StringTokenizer tokenizer = new StringTokenizer(characterGroup);
        
        // iterate over the tokens
        while (tokenizer.hasMoreTokens()) {
            
            // get the next token
            String token = tokenizer.nextToken();
            
            // find position of the dash character if present
            int dashPosition = token.indexOf('-');
            
            // if no dash then token represents just one index
            if (dashPosition == -1) {
                
                // parse the character index and add it to the index list
                indexList.add(new Integer(token.trim()));
            
            // otherwise the token represents a range of indices
            } else {
            
                // parse the index starting the range
                int startIndex = 
                    Integer.parseInt(token.substring(0, dashPosition).trim());
                
                // parse the index ending the range
                int endIndex = 
                    Integer.parseInt(token.substring(dashPosition + 1).trim());
                
                // iterate over the index range
                for (int i = startIndex; i <= endIndex; i++) {
                    
                    // add each index in the range to the index list
                    indexList.add(new Integer(i));
                }
            }
        }
        
        // return the index list
        return indexList;
    }

    ///////////////////////////////////////////////////////////////////
    ////                    protected variables                    ////

    protected static final Pattern NEXUS_UNDERSCORE_PATTERN = 
        Pattern.compile("_+");

    private static final Pattern NEXUS_CHARACTER_PROPERTY_PATTERN = 
        Pattern.compile("(.+?):(.+?)[,;]");

    private static final Pattern NEXUS_DASH_PATTERN = 
        Pattern.compile("\\s*-\\s*", Pattern.DOTALL);
    
    ///////////////////////////////////////////////////////////////////
    ////                    private variables                      ////

    private Pattern _blockPattern;
    
    private static final Pattern NEXUS_COMMENT_PATTERN = 
        Pattern.compile("\\[.*?\\]", Pattern.DOTALL);
    
    private static final Pattern NEXUS_EQUALS_PATTERN = 
        Pattern.compile("\\s*=\\s*", Pattern.DOTALL);

    ///////////////////////////////////////////////////////////////////
    ////                protected inner classes                    ////
        
    public static class NexusCommand {

        protected static NexusCommand extractNexusCommand(String commandName, String nexusBlockString) {
            
            Pattern NEXUS_COMMAND_PATTERN = Pattern.compile(commandName + "[^;]*?;", 
                Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
            
            Matcher commandMatcher = NEXUS_COMMAND_PATTERN.matcher(nexusBlockString);
            
            if (commandMatcher.find()) {

                return new NexusCommand(commandMatcher.group());
                
            } else {
                
                return null;
            }
        }

        
        protected static List extractNexusCommands(String commandName, String nexusBlockString) {
            
            Pattern NEXUS_COMMAND_PATTERN = Pattern.compile(commandName + "[^;]*?;", 
                Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
            
            Matcher commandMatcher = NEXUS_COMMAND_PATTERN.matcher(nexusBlockString);
            
            List commandList = new LinkedList();
            
            while (commandMatcher.find()) {

                commandList.add(new NexusCommand(commandMatcher.group()));
            }

            return commandList;
        }
        
        
        public NexusCommand(String originalCommandString) {
            
            _originalCommandString = originalCommandString;
            _orderedArguments = new TreeMap();
            _orderedPropertyKeys = new TreeMap();
            _propertyValues = new HashMap();
            _originalPropertyKeys = new HashMap();
            
            // remove terminal semicolon
            String commandString = 
                _originalCommandString.substring(0, 
                        _originalCommandString.length()-1);
            
            commandString = _parseArguments(commandString, 
                    NEXUS_QUOTED_ARGUMENT_PATTERN, false);
            
            commandString = _parseProperties(commandString, 
                    NEXUS_QUOTED_PROPERTY_PATTERN);
            
            commandString = _parseProperties(commandString, 
                    NEXUS_UNQUOTED_PROPERTY_PATTERN);
            
                commandString = _parseArguments(commandString, 
                        NEXUS_UNQUOTED_ARGUMENT_PATTERN, true);
        }

        ///////////////////////////////////////////////////////////////////
        ////                         public methods                    //// 
        
        public int argumentCount() {
            return _orderedArguments.size();
        }
        
        public Iterator argumentIterator() { 
            return new ArgumentIterator(); 
        }

        public String getProperty(String key) {
            
            return (String)_propertyValues.get(key);
        }
        
        public String name() { 
            return _commandName; 
        }
        
        public String originalPropertyKey(String allCapsKey) {
            
            return (String) _originalPropertyKeys.get(allCapsKey);
        }
        
        public Iterator propertyIterator() { 
            return new PropertyIterator(); 
        }
            
        public String toString() {
            return _originalCommandString;
        }

        ///////////////////////////////////////////////////////////////////
        ////                         private methods                   ////

        private String _parseArguments(String commandString, Pattern matchPattern, 
            boolean firstArgumentIsCommandName) {
            
            Matcher argumentMatcher = matchPattern.matcher(commandString);      
            
            if (firstArgumentIsCommandName) {
                argumentMatcher.find();
                    _commandName = argumentMatcher.group().toUpperCase();
            }
            
            int matchPosition = 0;
            while (argumentMatcher.find()) {
                String matchString = argumentMatcher.group(1);
                matchPosition = 
                    _originalCommandString.indexOf(matchString, matchPosition);
                _orderedArguments.put(new Integer(matchPosition), matchString);
                matchPosition += matchString.length();
            }
            
            return  matchPattern.matcher(commandString).replaceAll("");
        }
        
        private String _parseProperties(String commandString, 
            Pattern matchPattern) {
            
            Matcher propertyMatcher = matchPattern.matcher(commandString);
            
            int matchPosition = 0;
            while (propertyMatcher.find()) {
                String matchString = propertyMatcher.group(0);
                String originalKey = propertyMatcher.group(1);
                String allCapsKey = originalKey.toUpperCase();
                String value = propertyMatcher.group(2);
                matchPosition = 
                    _originalCommandString.indexOf(matchString, matchPosition);
                _orderedPropertyKeys.put(new Integer(matchPosition), allCapsKey);
                _propertyValues.put(allCapsKey,value);
                _originalPropertyKeys.put(allCapsKey, originalKey);
                matchPosition += matchString.length();
            }
            
            return matchPattern.matcher(commandString).replaceAll("");  
        }

        ///////////////////////////////////////////////////////////////////
        ////                         private variables                 ////
        
        private String _commandName;
        private final Map _orderedArguments;
        private final Map _orderedPropertyKeys;
        private final String _originalCommandString;
        private final Map _propertyValues;
        private final Map _originalPropertyKeys;
        private static final Pattern NEXUS_QUOTED_ARGUMENT_PATTERN = 
            Pattern.compile("\'(.*?)\'");
        private static final Pattern NEXUS_QUOTED_PROPERTY_PATTERN = 
            Pattern.compile("(\\S+)=\"(.*?)\"");
        private static final Pattern NEXUS_UNQUOTED_ARGUMENT_PATTERN = 
            Pattern.compile("([^,\\s]+)");  
        private static final Pattern NEXUS_UNQUOTED_PROPERTY_PATTERN = 
            Pattern.compile("(\\S+)=(\\S+)");

        ///////////////////////////////////////////////////////////////////
        ////                       private inner classes               ////
        
        private class ArgumentIterator implements Iterator {
            
            ///////////////////////////////////////////////////////////////////
            ////   public methods for NexusCommand.ArgumentIterator        ////

            public boolean hasNext() { 
                return _argumentKeyIterator.hasNext(); 
            }
            
                public Object next() { 
                    return _orderedArguments.get(_argumentKeyIterator.next()); 
                }
                
                public void remove() {
                    throw new UnsupportedOperationException(); 
                }

            ///////////////////////////////////////////////////////////////////
            ////   private variables for NexusCommand.ArgumentIterator     ////

                private Iterator _argumentKeyIterator = 
                    _orderedArguments.keySet().iterator();
        }
        
        private class PropertyIterator implements Iterator {
            
            ///////////////////////////////////////////////////////////////////
            ////   public methods for NexusCommand.PropertyIterator        ////

            public boolean hasNext() { 
                return _propertyMetaKeyIterator.hasNext(); 
            }
            
                public Object next() { 
                    return _orderedPropertyKeys.get(
                            _propertyMetaKeyIterator.next());
                }
                
                public void remove() { 
                    throw new UnsupportedOperationException(); 
                }

            ///////////////////////////////////////////////////////////////////
            ////   private variables for NexusCommand.PropertyIterator     ////

            private Iterator _propertyMetaKeyIterator = 
                _orderedPropertyKeys.keySet().iterator();
        }
    }
}
