/* Actor that parses Assumptions block of a Nexus-formatted string.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylogeny;

import java.util.Map;

import org.nddp.phylogeny.WeightVector;
import org.nddp.util.Port;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class ParseNexusAssumptionsBlock extends NexusParsingActor {

    public ParseNexusAssumptionsBlock(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an output port for sending the character matrix
        weightVectorPort = new TypedIOPort(this, "weightVector", false, true);
        weightVectorPort.setTypeEquals(BaseType.OBJECT);
        
        // precompile the pattern for finding the named Nexus block
        setBlockName("ASSUMPTIONS");
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort weightVectorPort;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public void fire() throws IllegalActionException {
        
        super.fire();
        
        if (! actorEnabled()) {
            return;
        }

        // read the nexus file content string from the input port       
        String nexusFileString = Port.getString(nexusFilePort, 0, true);

        // extract the characters block
        String assumptionsBlockString = extractNexusBlock(nexusFileString);
        
        // parse the characters block if it was found
        if (assumptionsBlockString != null) {
            
            NexusCommand weightSetCommand =            
                NexusCommand.extractNexusCommand("WTSET", assumptionsBlockString);
            
            if (weightSetCommand != null) {
    
                // get the unparsed version of the command
                String commandString = weightSetCommand.toString();
            
                // extract the value of the command which follows the equal sign
                int equalSignPosition = commandString.indexOf('=');
                String valueString = commandString.substring(equalSignPosition + 1);
            
                // get the mapping between character indices and property values
                Map weightMap = _parseCharacterPropertyList(valueString);
        
                // create a new weight set from the mapping
                WeightVector weightVector = new WeightVector(weightMap);
                
                // send the new character matrix
                weightVectorPort.broadcast(new ObjectToken(weightVector));           
            }
        }
    }
 
    private static final long serialVersionUID = 1L;
}
