/* Composite collection-oriented actor for wrapping conventional workflows.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

 package org.nddp.coactors;
  
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.List;

import org.nddp.Coactor;
import org.nddp.Collection;
import org.nddp.CollectionHandler;
import org.nddp.CollectionIOPort;
import org.nddp.CollectionManager;
import org.nddp.CollectionPath;
import org.nddp.DomainObject;
import org.nddp.InputPortBuffer;
import org.nddp.NewCollectionManager;
import org.nddp.OutputPortBuffer;
import org.nddp.CollectionHandler.TokenDisposition;
import org.nddp.exceptions.CollectionException;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.util.Parameters;

import ptolemy.actor.IOPort;
import ptolemy.actor.TypedCompositeActor;
import ptolemy.data.ObjectToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Port;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
 
/**
 * @nddp.coactor type="general"
 *
 *  @author Timothy M. McPhillips
 */

	public class CompositeCoactor extends TypedCompositeActor 
        implements Coactor {

    /** 
    	 * Constructs an instance of CollectionActor.
   	 * 
   	 * @param container The run-time container of this actor.
 	 * @param name The name of this actor.
    	 */
	public CompositeCoactor(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

    	    // call the superclass constructor
        	super(container, name);

        input = CollectionIOPort.createInputPort(this, "input");
        output = CollectionIOPort.createOutputPort(this, "output");

        collectionPath =  
            Parameters.stringParameter(this, "collectionPath", "");
    }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public Parameter collectionPath;
    public CollectionIOPort input;
    public CollectionIOPort output;
    
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    /** 
     * Handles changes in parameter values.
     *  
     * @param attribute The parameter with a new value.
     */
  	public final void attributeChanged(Attribute attribute) 
		throws IllegalActionException {
            
        if (attribute == collectionPath) {
//        		_collectionPath = new CollectionPath(collectionPath);
                
        } else {
 
            super.attributeChanged(attribute);
        	}   
  	}
  	
    public void fire() throws IllegalActionException {
 
        // read next data token from input port, passing a value
        // of false to the unwrapData parameter so that DataTokens
        // can be checked for proper nesting in collections
        Token token = input.get(false);
        
        if (token != null) {
            _collectionHandler.handleDataToken(token);
        }
    	}   

  	/** 
    	 * Initializes state of actor at start of workflow execution.
    	 */
	public void initialize() throws IllegalActionException {

        // call superclass constructor
        super.initialize();

        // validate and construct the collection path
        _collectionPath = new CollectionPath(collectionPath);

        _overriddenParametersMap.clear();
        _overridableParametersMap.clear();
        
        // enable dynamic override of all actor parameters
        for (Iterator i = attributeList(Parameter.class).iterator(); i.hasNext(); ) {
            Parameter parameter = (Parameter)i.next();
            _overridableParametersMap.put(parameter.getName(), parameter);
        }

        _collectionHandler = new CollectionHandler(this, _collectionPath,
            _overridableParametersMap, _overriddenParametersMap);    
                
        _initializeInputPortBuffers();
        
        _initializeOutputPortBuffers();
	}

    private void _initializeOutputPortBuffers() throws IllegalActionException {

        _outputPortBufferList.clear();
        
        // loop over all internal output ports
        for (Iterator i=outputPortList().iterator(); i.hasNext(); ) {

            // get the next port
            IOPort port = (IOPort) i.next();

            // ignore the externally directed output port
            if (port == output) {
                continue;
            }

            OutputPortBuffer buffer = new OutputPortBuffer(port);

            _outputPortBufferList.add(buffer);
        }
    }
    
    private void _initializeInputPortBuffers() throws IllegalActionException {
        
        _inputPortBufferMap = new HashMap();
        _fireOncePerScope = false;
        
        for (Iterator i = inputPortList().iterator(); i.hasNext(); ) {

            // get the next port
            IOPort port = (IOPort)i.next();

            // operate on all input ports other than the external input port
            if (port != input) {
    
                // create a port buffer for the port
                InputPortBuffer buffer = new InputPortBuffer(port);
                
                if (buffer.metadataKey() != null) {
                    
                    // map from type of data to the port bufcer
                    _inputPortBufferMap.put(buffer.metadataKey(), buffer);
                    
                } else {
        
                    // map from type of data to the port bufcer
                    _inputPortBufferMap.put(buffer.dataType(), buffer);
                }
                
                // enable aggregate input mode if port generates array tokens
                if (buffer.singleton() == false) {
                    _fireOncePerScope = true;
                }
            }
        }
        
        if (_inputPortBufferMap.size() != 1) {
            _fireOncePerScope = true;
        }
    }
    
	/** 
     * Adds a port to the actor at runtime.  This method is called by 
     * Vergil when the user interactively adds ports to actors from within 
     * Vergil and when an actor containing such a port is initialized from
     * its MOML description. This method should not be called under any 
     * other circumstances.
     * 
     * @param name The name of the port to create.
     * @return The new port.
     */ 
    	public Port newPort(String name) 
    		throws NameDuplicationException {
    	    
    		try {
    		    // get a write mutex for the workspace
            	_workspace.getWriteAccess();
                
            	// construct the new port
            CollectionIOPort port = new CollectionIOPort(this, name);
            	
            	// return the port
            	return port;
            	
        	} catch (IllegalActionException ex) {
        	    
            	// translate to a runtime exception since this should never 
        	    // happen.
            	throw new InternalErrorException(this, ex, null);
            	
        	} finally {
        	    
        	    	// release the workspace mutex
            	_workspace.doneWriting();
        	}
    	}
    	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

    public CollectionIOPort _getDefaultOutputPort() {
        return output;
    }
    
	public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException, CollectionException {
 
	    if (_fireOncePerScope && _collectionHandler.managesTopOfScope(collectionManager)) {
 
	        for (Iterator i=_inputPortBufferMap.values().iterator(); i.hasNext(); ) {
	            
                if (! ((InputPortBuffer)i.next()).isReady()) {
                    _clearAllPortBuffers();
                    return;
                }
            }

            for (Iterator i=_inputPortBufferMap.values().iterator(); i.hasNext(); ) {
             
                InputPortBuffer buffer = (InputPortBuffer)i.next();
                buffer.flush();
            }
            
            runCompositeWorkflow(collectionManager);
        }
    }

    public void _clearAllPortBuffers() {

        for (Iterator i=_inputPortBufferMap.values().iterator(); i.hasNext(); ) {
            
            ((InputPortBuffer)i.next()).clear();
        }
    }
          
    private void runCompositeWorkflow(CollectionManager collectionManager) 
        throws IllegalActionException {
        
        Map newCollectionMap = new HashMap();
        
        // if scope is being deleted then make parent collection 
        // the default container for new data and subcollectoins
        if (_collectionPath.deleteScope()) {
            collectionManager = collectionManager.parentCollectionManager();
        }

        // fire the subworkflow's director once
        try {
            getDirector().fire();

        } catch(IllegalActionException ex) {

            collectionManager.addException(this, new CollectionException(ex.getMessage()));
            return;
        }

        for (Iterator i=_outputPortBufferList.iterator(); i.hasNext(); ) {

            CollectionManager localCollectionManager = collectionManager;
            
            // get the next port
            OutputPortBuffer buffer = (OutputPortBuffer)i.next();
            
            Class subcollectionType = buffer.subcollectionType();
            
            if (subcollectionType != null) {

                if (newCollectionMap.containsKey(subcollectionType)) {
                        
                    localCollectionManager = (CollectionManager)newCollectionMap.get(subcollectionType);
                    
                } else {
                        
                    Collection collection = null;
                    try {
                        collection = (Collection) subcollectionType.newInstance();
                    } catch (InstantiationException e) {
                        e.printStackTrace();
                    } catch (IllegalAccessException e) {
                        e.printStackTrace();
                    }
                    localCollectionManager = new NewCollectionManager(collection, 
                        collectionManager, output);
                    
                    newCollectionMap.put(subcollectionType, localCollectionManager);
                }
            }
                
            try {

                buffer.flush(localCollectionManager);
            
            } catch(IllegalActionException ex) {
    
                localCollectionManager.addException(this, new CollectionException(ex.getMessage()));
                return;
            }
        }
    }

	public CollectionHandler.ExceptionDisposition handleCollectionException(CollectionManager
        collectionManager, CollectionException exception) {

		return CollectionHandler.PROPAGATE_EXCEPTION;
	}
	

   	public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException, CollectionException {
   	    
        if (_fireOncePerScope && _collectionHandler.managesTopOfScope(collectionManager)) {

            for (Iterator i=_inputPortBufferMap.values().iterator(); i.hasNext(); ) {
                
                InputPortBuffer buffer = (InputPortBuffer)i.next();
                buffer.clear();
            }

            for (Iterator i=_outputPortBufferList.iterator(); i.hasNext(); ) {
                
                OutputPortBuffer buffer = (OutputPortBuffer)i.next();
                buffer.clear();
            }
        }
        
        if (_collectionPath.deleteScope() && _collectionHandler.managesTopOfScope(collectionManager)) {
            
            return CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;

        } else {

            return CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
        }
   	}
   	
    
   	public CollectionHandler.TokenDisposition _handleData(CollectionManager collectionManager, 
        Token token) throws IllegalActionException, CollectionException {
 		
   	    Class dataType = token.getClass();

        boolean deleteData = false;
        
        InputPortBuffer buffer = (InputPortBuffer)_inputPortBufferMap.get(dataType);
        
        if (buffer != null) {
            
            deleteData = buffer.put(token);

            if (! _fireOncePerScope) {
                
                buffer.flush();
                
                runCompositeWorkflow(collectionManager);
            }
        }

        if (deleteData) {
            return CollectionHandler.DISCARD_TOKEN;
        } else { 
            return CollectionHandler.FORWARD_TOKEN;
        }
    }
   	
   	public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        collectionManager, DomainObject object) throws IllegalActionException,
        CollectionException {
   	    
        Class domainObjectType = object.getClass();
        boolean deleteData = false;
        
        InputPortBuffer buffer = (InputPortBuffer)_inputPortBufferMap.get(domainObjectType);
        
        if (buffer != null) {
            
            deleteData = buffer.put(new ObjectToken(object));

            if (! _fireOncePerScope) {
                
                buffer.flush();

                runCompositeWorkflow(collectionManager);
            }
        }

        if (deleteData) {
            return CollectionHandler.DISCARD_TOKEN;
        } else { 
            return CollectionHandler.FORWARD_TOKEN;
        }
   	}
   	
	public CollectionHandler.TokenDisposition _handleExceptionToken(CollectionManager
        collectionManager, ExceptionToken token) throws IllegalActionException,
        CollectionException  {
	    
	    return CollectionHandler.FORWARD_TOKEN;
    }

    public void handleLoopTermination(LoopTerminationToken token) throws IllegalActionException {
        
        throw new IllegalActionException("Actor " + this + 
                " does not support loop termination tokens");
    }
    
	public CollectionHandler.TokenDisposition _handleMetadata(CollectionManager collectionManager, 
        MetadataToken token) throws IllegalActionException, 
        CollectionException {

        String key = token.key();
        Token value = token.value();
        
        Class type = value.getClass();

        boolean deleteData = false;
        
        InputPortBuffer buffer = (InputPortBuffer)_inputPortBufferMap.get(key);
        
        if (buffer != null && buffer.dataType() == type) {
            
            deleteData = buffer.put(value);

            if (! _fireOncePerScope) {
                
                buffer.flush();
                
                runCompositeWorkflow(collectionManager);
            }
        }

        if (deleteData) {
            return CollectionHandler.DISCARD_TOKEN;
        } else { 
            return CollectionHandler.FORWARD_TOKEN;
        }
    }

  	public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
  	    
        parameter.setExpression(value.toString());
    }

    public TokenDisposition defaultDataDisposition() {
        return _defaultDataDisposition;
    }
    
    public CollectionHandler collectionHandler() {
        return _collectionHandler;
    }

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    private boolean _fireOncePerScope = false;

    private CollectionHandler _collectionHandler;
    
    private CollectionPath _collectionPath;
    
    private TokenDisposition _defaultDataDisposition = CollectionHandler.FORWARD_TOKEN;
    
    private final Map _overridableParametersMap = new HashMap();
    
    private final Map _overriddenParametersMap = new HashMap();
    
    private Map _inputPortBufferMap;
    
    private List _outputPortBufferList = new LinkedList();

    private static final long serialVersionUID = 1L;
}
