/* Actor that prompts for user approval to pass collections through.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors;

import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import javax.swing.*;

import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;

import java.awt.*;
import java.awt.event.*;
import java.lang.InterruptedException;

/**
 * @nddp.coactor type="general"
 */
public class PauseFlow extends CollectionTransformer {

    public PauseFlow(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

        super(container, name);
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void initialize() throws IllegalActionException {
        
        // call the superclass's initialize method
        super.initialize();

        // start up the graphical user interface if it is not running
        if ( _gui.isRunning() == false ) {
            _gui.start();
        }
	}

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException {

    		if (_gui.isRunning() && _gui._okToDiscardCollection(
    		        collectionManager.collection().toString())) {
        		return  CollectionHandler.IGNORE_AND_DISCARD_COLLECTION;
        } else {
	    	    return  CollectionHandler.IGNORE_AND_FORWARD_COLLECTION;
        }
   	}    

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    private Gui _gui = new Gui();

	///////////////////////////////////////////////////////////////////
	////                       private inner classes               ////

    private static class Gui {
     
        	public Gui() {
        	}
        	
		///////////////////////////////////////////////////////////////////
		////           public methods for PauseFlow.Gui                //// 
 
		public boolean isRunning() {
		    return _isRunning;
		}
		
        public void start() {
    
	        // create the top level window
	        JFrame frame = new JFrame("Pause Flow");
	        frame.getContentPane().setLayout(new BorderLayout());
	        
	        // create handler for window-close event
	        frame.addWindowListener(new WindowAdapter() {	            
	            public void windowClosing(WindowEvent e) { 
	                _isRunning = false;
	                _userSelection.value = UserSelection.CONTINUE;
	                _notifyButtonWaiter();
	            }
	        });
	        
	        // create the text field for displaying the current delimiter 
	        // contents and name
	        _textField = new JTextField("",20);
	        _textField.setEditable(false);
	        frame.getContentPane().add(_textField,"North");
	        
	        // create the skip button
	        _discardButton = new JButton("Discard");
	        frame.getContentPane().add(_discardButton, "West");
	        _discardButton.setEnabled(false);
	        
	        // create event handler for skip-button clicks
	        _discardButton.addActionListener(new ActionListener() {
	            public void actionPerformed(ActionEvent e) {         
	                _handleButtonClick(UserSelection.SKIP);
	            } 
	        });
	        
	        // create the continue button
	        _continueButton = new JButton("Continue");
	        frame.getContentPane().add(_continueButton, "East");
	        _continueButton.setEnabled(false);
	        
	        //create event handler for continue-button clicks
	        _continueButton.addActionListener(new ActionListener() {
	            public void actionPerformed(ActionEvent e) {
					_handleButtonClick(UserSelection.CONTINUE);
	            }
	        });
	        
	        // pack the components in the window and display on screen
	        frame.pack();
	        frame.setVisible(true);
	
	        _isRunning = true;
	    }
        	
		///////////////////////////////////////////////////////////////////
		////          private methods for PauseFlow.Gui                //// 
	
	    private void _disableButtons() {
	        
	 		_discardButton.setEnabled(false);
			_continueButton.setEnabled(false);
	    }
	    
	    private void _enableButtons() {
	        
			_userSelection.value = UserSelection.NULL;
			_continueButton.setEnabled(true);
			_discardButton.setEnabled(true);
	    }

	    private void _handleButtonClick(
	            UserSelectionValue value) {  
	        
	        _userSelection.value = value;
	        _disableButtons();
	        _notifyButtonWaiter();
	    }
	    
		private void _notifyButtonWaiter() {
			synchronized(_userSelection) {
				_userSelection.notify();
			}
	    }
		
		private boolean _okToDiscardCollection(String prompt)  {
	           
			_textField.setText(prompt);
			_enableButtons();
				
			while ( _isRunning & _userSelection.value == UserSelection.NULL ) {
				synchronized(_userSelection) {
		    			try {
						_userSelection.wait();
					} catch (InterruptedException e) {
					    return false;
					}
				}
			}
			
			return _userSelection.value == UserSelection.SKIP;
	    }
	
		///////////////////////////////////////////////////////////////////
		////        private variables for PauseFlow.Gui                //// 
		
	    private JButton _continueButton;
	    private JButton _discardButton;
		private boolean _isRunning = false;    
	    private JTextField _textField;
		private UserSelection _userSelection = new UserSelection();
    }
    
    private static final class UserSelection {

        	public UserSelection() {
        	}
        
		///////////////////////////////////////////////////////////////////
		////        public variables for PauseFlow.UserSelection       ////
        	
	    public static final UserSelectionValue CONTINUE = 
	        new UserSelectionValue();
	    
	    public static final UserSelectionValue NULL =
	        new UserSelectionValue();
	    
	    public static final UserSelectionValue SKIP = 
	        new UserSelectionValue();    

		///////////////////////////////////////////////////////////////////
		////       private variables for PauseFlow.UserSelection       ////
	
        private UserSelectionValue value = UserSelection.NULL;
	}

    private static final class UserSelectionValue {}
}
