/* CollectionDisplay that displays each token received in an XML-like format.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.coactors;

import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.tokens.DataToken;
import org.nddp.tokens.DomainObjectToken;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.util.Parameters;
import org.nddp.util.Xml;

import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="general"
 */
public class TokenDisplay extends CollectionDisplay  {

    public TokenDisplay(CompositeEntity container, String name)
        throws IllegalActionException, NameDuplicationException {
        
        super(container, name);

        	showDataTypes = 
        	    Parameters.booleanParameter(this, "showDataTypes", true);
        	showDomainObjectContents = 
        	    Parameters.booleanParameter(this, "showDomainObjectContents", true);
        	formatDataTokens = 
        	    Parameters.booleanParameter(this, "formatDataTokens", true);
    }
    
	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    	public Parameter formatDataTokens;
	public Parameter showDataTypes;
    	public Parameter showDomainObjectContents;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
	public void initialize() throws IllegalActionException {
        
        	super.initialize();
		_indentation.reset();    
	}
   	
    ///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
    
   	protected void _appendText(String text) {
   	    
	    super._appendText(_indentation + text);
	}

	protected void _clearDisplay() {
	    
		super._clearDisplay();
		_indentation.reset();	
    }
	
    public void _handleCollectionEnd(CollectionManager collectionManager) {
	    
        _indentation.decrease();

        _appendText(collectionManager.collection().closingDelimiter().
	        toXmlString(_showDataTypes));
    }
	
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) {
   	    
        _appendText(collectionManager.collection().openingDelimiter().
	        toXmlString(_showDataTypes));
        
        _indentation.increase();
        
        return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
   	}	
	
    public CollectionHandler.TokenDisposition _handleData(CollectionManager collectionManager, 
        Token token) {
 		
	    if (_formatDataTokens) {
		    
	        _appendText(DataToken.toXmlString(token, _showDataTypes));
	    
	    } else {
	    
	        _appendText(token.toString());
	    }
	    
        return  CollectionHandler.FORWARD_TOKEN;
  	}
   	
    public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        collectionManager, DomainObject object) {
   	    
	    _appendText(DomainObjectToken.toXmlString(object, _showDataTypes, 
            _showDomainObjectContents, _indentation));

	    return  CollectionHandler.FORWARD_TOKEN;
   	}   
   	
    public CollectionHandler.TokenDisposition _handleExceptionToken(CollectionManager
        collectionManager, ExceptionToken token)  {
	    
	    _appendText(token.toXmlString(_showDataTypes));
	    
	    return  CollectionHandler.FORWARD_TOKEN;
    }
	
	public void handleLoopTermination(LoopTerminationToken token) {
	    
	    _appendText(token.toXmlString(_showDataTypes));
    }
	
    public CollectionHandler.TokenDisposition _handleMetadata(CollectionManager collectionManager, 
        MetadataToken token) {

	    _appendText(token.toXmlString(_showDataTypes));
	    
	    return  CollectionHandler.FORWARD_TOKEN;
    }
	
    public void _handleParameterChange(Parameter parameter, Token value)
  		throws IllegalActionException {

 		if (parameter == showDataTypes) {
            
            _showDataTypes = Parameters.booleanValue(value);

 		} else if (parameter == showDomainObjectContents) {
            
            _showDomainObjectContents = Parameters.booleanValue(value);

        } else if (parameter == formatDataTokens) {
            
            _formatDataTokens = Parameters.booleanValue(value);
            
        } else {
            
            super._handleParameterChange(parameter, value);
        }
    }
  	
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

    private boolean _formatDataTokens;
    private Xml.Indentation _indentation = new Xml.Indentation("", "    ");
    private boolean _showDataTypes;
    private boolean _showDomainObjectContents;
}
