/* Wrapper actor for Phylip CONSENSE.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
package org.nddp.coactors.phylogeny;

import java.util.LinkedList;
import java.util.List;

import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.DomainObject;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.phylogeny.phylip.PhylipConsenseRunner;
import org.nddp.phylogeny.Tree;
import org.nddp.util.Parameters;
import org.nddp.util.ProcessRunner;
import org.nddp.util.ProcessRunningActor;

import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="phylogenetics"
 */
public class PhylipConsense extends ProcessRunningActor {

    public PhylipConsense(CompositeEntity container, String name)
        	throws NameDuplicationException, IllegalActionException {
        
        	super(container, name);
        
        	consensusThreshold = 
        	    Parameters.doubleParameter(this, "consensusThreshold", 1.0);
        	
        	extendedConsensus = 
        	    Parameters.booleanParameter(this, "extendedConsensus", false);
        	
		rootAtNode = 
		    Parameters.stringParameter(this, "rootAtNode", "");
		
        	rootedTrees = 
        	    Parameters.booleanParameter(this, "rootedTrees", false);
        	
        	Parameters.fix(collectionPath, "Nexus/Tree");
        	        	
		_enableParameterOverride(rootAtNode);
    }
    

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

	public Parameter consensusThreshold;
	public Parameter extendedConsensus;
	public Parameter rootAtNode;
	public Parameter rootedTrees;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public void initialize() throws IllegalActionException {
 
        	super.initialize();

        	_nexusCollectionManager = null;
    }

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    protected void _configureRunner() throws InconsistentDataException {
	    PhylipConsenseRunner runner = (PhylipConsenseRunner)_runner;
	    runner.rootAtNode(_rootAtNode);
	    runner.setConsensusThreshold(_consensusThreshold);
	    runner.setExtendendConsensus(_extendedConsensus);
	    runner.setRootedTrees(_rootedTrees);
	    runner.setInputTreeList(_inputTreeList);
    }

    protected ProcessRunner _getRunnerInstance() {
        
        	return new PhylipConsenseRunner();
    }

    public void _handleCollectionEnd(CollectionManager collectionManager) 
  		throws IllegalActionException, CollectionException {
    
		if (collectionManager == _nexusCollectionManager) {
		    
		    if (! _inputTreeList.isEmpty()) {
				_runProgram();
				Tree consensusTree = (Tree)_runResult;
		    	    	_nexusCollectionManager.addDomainObject(consensusTree);
		    }
		   	
			_nexusCollectionManager = null;
		}
    }

    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException {
    
	    	_nexusCollectionManager = collectionManager;
        	_inputTreeList = new LinkedList();
	        	
      	return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
    }

    public CollectionHandler.TokenDisposition _handleDomainObject(
        	CollectionManager fileCollectionManager, DomainObject object)
		throws IllegalActionException {

		_inputTreeList.add(object);

        return  CollectionHandler.FORWARD_TOKEN;
    }

    public void _handleParameterChange(Parameter parameter, Token newValue) 
		throws IllegalActionException {
            
        if (parameter == consensusThreshold) {
        		_consensusThreshold = Parameters.doubleValue(newValue);
        		if (_consensusThreshold < 0.5 || _consensusThreshold > 1.0) {
        		    throw new IllegalActionException(
        		            "consensusThreshold must be between 0.5 and 1.0");
        		}
        } else if (parameter == extendedConsensus ) {
        		_extendedConsensus = Parameters.booleanValue(newValue);
        } else if (parameter == rootedTrees ) {
        		_rootedTrees = Parameters.booleanValue(newValue);
        } else if (parameter == rootAtNode) {
        		_rootAtNode = Parameters.stringValue(newValue);
        } else {
            super._handleParameterChange(parameter, newValue);
        }
    }
  
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private double _consensusThreshold;
	private boolean _extendedConsensus;
	private List _inputTreeList;
	private CollectionManager _nexusCollectionManager;
	private String _rootAtNode;
	private boolean _rootedTrees;
    private static final long serialVersionUID = 1L;
 }
