/* Actor that extracts protein sequences from PDB header collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.coactors.proteins;

import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.coactors.CollectionTransformer;
import org.nddp.proteins.ProteinSequence;
import org.nddp.proteins.PDBUtilities;
import org.nddp.util.Parameters;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="proteins"
 */
public class PDBHeaderSequenceExtractor extends CollectionTransformer {

    public PDBHeaderSequenceExtractor(CompositeEntity container, String name)
		throws NameDuplicationException, IllegalActionException {
		
	    super(container, name);

	    Parameters.fix(collectionPath, "PDBHeader/String");
	}
 
	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 
   	 
    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException  {
   	     
   	     _addSequenceToHeaderCollection();
	}
   	 
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager
        collectionManager) throws IllegalActionException {
   		
   		_currentChain = "";
   		_firstChain = true;
   		
		_pdbHeaderCollectionManager = collectionManager;
   		_fileName = collectionManager.parentCollectionManager().collection().name();
		
   		return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION; 
   	}

    public CollectionHandler.TokenDisposition _handleData(CollectionManager
         collectionManager, Token token) throws IllegalActionException {

   	 	String headerLine = ((StringToken)token).stringValue();
       
        if (PDBUtilities.isPDBSequenceLine(headerLine)) {

	        	String chainID = 
	        	    PDBUtilities.chainIDFromPDBSequenceLine(headerLine);

        		if (! chainID.equals(_currentChain)) {
	
        			// store new chain ID
	        		_currentChain = chainID;	                            

	        		if (_firstChain) {
	        		    
	 				_firstChain = false;
	        		
	        		} else { 
	 				_addSequenceToHeaderCollection();
        		    }
	        		
				_sequence = new ProteinSequence();
				_sequence.setId(_fileName + ":" + chainID);
	            }

        		// add the residues on the current line of the header to
        		// the current sequence
        		// TODO eliminate dependency on PDB file format
        		_sequence.addResidues(headerLine.substring(19));
        }
   	 	
 		return  CollectionHandler.FORWARD_TOKEN;
  	}

   	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
   	 
   	private void _addSequenceToHeaderCollection() {
   	     
   	     if (_sequence.length() > 0) {
			_pdbHeaderCollectionManager.addDomainObject(_sequence);
   	     }
   	 }

   	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

    	private String _currentChain;
	private String _fileName;
	private boolean _firstChain;
    private CollectionManager _pdbHeaderCollectionManager;
    	private ProteinSequence _sequence;
    private static final long serialVersionUID = 1L;
}
