/* Actor that parses PDB file provided line by line.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.coactors.proteins;

import org.nddp.Collection;
import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.coactors.CollectionTransformer;
import org.nddp.exceptions.CollectionException;
import org.nddp.proteins.PDBCollection;
import org.nddp.proteins.PDBFooterCollection;
import org.nddp.proteins.PDBHeaderCollection;
import org.nddp.proteins.ProteinAtom;
import org.nddp.proteins.PDBUtilities;
import org.nddp.proteins.ProteinChainCollection;
import org.nddp.util.FileOrUrl;
import org.nddp.util.Parameters;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="proteins"
 */
public final class PDBParser extends CollectionTransformer {

    public PDBParser(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

        super(container, name);

        Parameters.fix(collectionPath, "TextFile/String");
    }

    ///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public void _handleCollectionEnd(CollectionManager
        fileCollectionManager) throws IllegalActionException {

        _pdbCollectionManager.addMetadata("filename", 
            fileCollectionManager.metadataValue("filename"));
    }
    
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager
        fileCollectionManager) throws IllegalActionException {
    
		_fileName = 
		    FileOrUrl.fileRoot(fileCollectionManager.collection().name());
				
		Collection pdbCollection = new PDBCollection(_fileName);
		
		_pdbCollectionManager = 
		    manageNewCollection(pdbCollection,
	            fileCollectionManager.parentCollectionManager());
  
		Collection headerCollection = new PDBHeaderCollection("");
	
		_headerCollectionManager = manageNewCollection(
	        headerCollection, _pdbCollectionManager, output);    			
		
		Collection footerCollection = new PDBFooterCollection("");
		
        	_footerCollectionManager = manageNewCollection(
    	        footerCollection, _pdbCollectionManager, output);
		
		// reset header completion flag since a new file has begun 
        	_headerComplete = false;
   	    _currentChain = "";
   	    
   	    return  CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;
    }

    public CollectionHandler.TokenDisposition _handleData(CollectionManager 
        	fileCollectionManager, Token token) 
		throws IllegalActionException, CollectionException {

        String pdbLine = ((StringToken)token).stringValue();
        
        if (PDBUtilities.isPDBAtomLine(pdbLine)) {
        
            _headerComplete = true;
            
            ProteinAtom atom = ProteinAtom.valueOfPDBAtomLine(pdbLine);
            
            // start a new chain collection if this is the first atom 
	        // in a new chain
            if ( ! atom.chainID().equals(_currentChain) ) {
                
            		_currentChain = atom.chainID();
            		
				String chainDelimiterName = _fileName + ":" + atom.chainID();
				
				Collection chainCollection = new ProteinChainCollection(chainDelimiterName);
				
		        	_atomsCollectionManager = manageNewCollection(
        	            chainCollection, 
        	            _pdbCollectionManager, 
        	            output);
            }
            
            _atomsCollectionManager.addDomainObject(atom);
            
        } else if (_headerComplete == false) {
        
            _headerCollectionManager.addDataToken(token);
            
        } else {
            
            _footerCollectionManager.addDataToken(token);
        }
        
		return  CollectionHandler.DISCARD_TOKEN;
    }


	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
   
    private CollectionManager _atomsCollectionManager;
    private String _currentChain;
    private String _fileName;
    private CollectionManager _footerCollectionManager;
    private CollectionManager _headerCollectionManager;
    private boolean _headerComplete;
	private CollectionManager _pdbCollectionManager;
}
