/* Class representing a taxonomic character.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

public class TaxonomicCharacter {

    public TaxonomicCharacter(String description, char[] allowedStates) {
        super();
        _description = description;
        _allowedStates = (char[]) allowedStates.clone();
        _minimumStateChanges = allowedStates.length -1;
    }

	public TaxonomicCharacter(String description, 
        CharacterMatrix matrix, int characterIndex) {

        	matrix._assertMatrixIsComplete();
        
        	Set allowedStatesSet = new TreeSet();
      	_description = description;
      
	    	for (int taxonIndex = 0, taxonCount = matrix.taxonCount(); 
	    		taxonIndex < taxonCount; taxonIndex++) {
	    	    
	        	char character = matrix.element(taxonIndex, characterIndex);
	        	
	        	if (character != CharacterMatrix.STANDARD_GAP_SYMBOL && 
                	character != CharacterMatrix.STANDARD_MISSING_SYMBOL) {
	        	    
		        	allowedStatesSet.add(new Character(character));
		    	}
	    	}

        	_allowedStates = new char[allowedStatesSet.size()];
        
        	int i = 0;
        	for (Iterator iterator = allowedStatesSet.iterator(); 
        		iterator.hasNext(); i++) {
        	    
        		_allowedStates[i] = ((Character)iterator.next()).charValue();
        	}	

        	_minimumStateChanges = _allowedStates.length -1;
	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public final char allowedState(int stateIndex) {
        return _allowedStates[stateIndex];
    }
    
    public final int allowedStatesCount() {
        return _allowedStates.length;
    }

    public Set allowedStateSet() {
        
        Set set = new HashSet();
        
        for (int i = 0; i < _allowedStates.length; i++) {
            set.add(new Character(_allowedStates[i]));
        }
        
        return set;
    }

    public int minimumStateChanges() {
        return _minimumStateChanges;
    }
    
    public String toString() {
        
       	StringBuffer buffer = new StringBuffer(_description + " (");
       	
       	for (int i = 0; i < _allowedStates.length - 1; i++ ) {
	       	buffer.append(_allowedStates[i] + ", ");
       	}
       	
       	if (_allowedStates.length > 0) {
       	    buffer.append(_allowedStates[_allowedStates.length - 1]);
       	}
       	
       	buffer.append(")");
       	
		return buffer.toString();
    }

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
    
    private final char[] _allowedStates;
    private final String _description;
    private final int _minimumStateChanges;
    
    public static class NexusDataType {
        
        private NexusDataType(boolean isMolecularData, String description) {
            
    			_isMolecularData = isMolecularData;
    			_description = description;
        }
        
        public boolean isMolecularData() {
            return _isMolecularData;
        }
        
        public String toString() {
            return _description;
        }
        
        private final boolean _isMolecularData;
        private final String _description;
    }
    
    public static final NexusDataType MORPHOLOGICAL_DATA = new NexusDataType(false, "MORPHOLOGICAL");
    public static final NexusDataType PROTEIN_DATA = new NexusDataType(true, "PROTEIN");
    public static final NexusDataType DNA_DATA = new NexusDataType(true, "DNA");    
    public static final NexusDataType RNA_DATA = new NexusDataType(true, "RNA");    
}
