/* Utility files for reading and writing Phylip-formatted files.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny.phylip;

import java.io.FileWriter;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.exceptions.ParseException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.DistanceMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.WeightVector;

public class PhylipFiles {

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
    public static DistanceMatrix parseDistanceMatrixFileString(String outfile) {

        Matcher tokenMatcher = 
            SPACE_SEPARATED_TOKENS_PATTERN.matcher(outfile);

        tokenMatcher.find();
        
        int matrixSize = Integer.valueOf(tokenMatcher.group()).intValue();
        
        DistanceMatrix distanceMatrix = new DistanceMatrix(matrixSize);
        
        for (int taxon1 = 0; taxon1 < matrixSize; taxon1++) {
            
            // discard the taxon name
            tokenMatcher.find();
            
            for (int taxon2 = 0; taxon2 < taxon1; taxon2++) {
                
                tokenMatcher.find();
                double distance = Double.valueOf(tokenMatcher.group()).doubleValue();
                distanceMatrix.setDistance(taxon1, taxon2, distance);
            }
        }
        
        return distanceMatrix;
    }

    public static List parseTreeFileString(String treeFileString,
            Map taxonAbbreviations) throws ParseException {

        List treeList = new LinkedList();
        
	    // find all tree statements
        Matcher commandMatcher = 
            PHYLIP_TREE_STATEMENT_PATTERN.matcher(treeFileString);
        
        while (commandMatcher.find()) {
            
			Tree tree = new Tree(commandMatcher.group(1),
                taxonAbbreviations);
			
			treeList.add(tree);
        }

        return treeList;
	}
    
	public static Map writeCharacterMatrixFile(CharacterMatrix matrix, 
        FileWriter fileWriter) throws IOException {

	    Map taxonAbbreviations = new HashMap();
	    
	    fileWriter.write(matrix.taxonCount() + " " + matrix.characterCount() + 
            "\n");
	
	    for (Iterator taxonIterator = matrix.taxonIterator(), 
	            rowIterator = matrix.rowIterator(); rowIterator.hasNext(); ) {
	        
	        String taxonName = (String)taxonIterator.next();
	        String abbreviatedTaxonName = _abbreviateTaxonName(taxonName);
	        taxonAbbreviations.put(abbreviatedTaxonName.trim(), taxonName);        
	        fileWriter.write(abbreviatedTaxonName + rowIterator.next() + "\n");
	    }
	    
	    fileWriter.close();
	    
	    return taxonAbbreviations;
	}
 

    public static Map writeDistanceMatrixFile(CharacterMatrix characterMatrix, 
            DistanceMatrix distanceMatrix, FileWriter fileWriter) throws IOException {

        Map taxonAbbreviations = new HashMap();
        
        fileWriter.write(distanceMatrix.size()+ "\n");
    
        for (Iterator taxonIterator = characterMatrix.taxonIterator(), 
                rowIterator = distanceMatrix.rowIterator(); rowIterator.hasNext(); ) {
            
            String taxonName = (String)taxonIterator.next();
            String abbreviatedTaxonName = _abbreviateTaxonName(taxonName);
            taxonAbbreviations.put(abbreviatedTaxonName.trim(), taxonName);        
            fileWriter.write(abbreviatedTaxonName + rowIterator.next() + "\n");
        }
        
        fileWriter.close();
        
        return taxonAbbreviations;
    }    

    public static CharacterMatrix writeTreeFile(List treeList, 
        FileWriter fileWriter) throws ExternalEnvironmentException {

	    Tree tree = null;
	    
        try {
	            
            for (Iterator i = treeList.iterator(); i.hasNext(); ) {
	        
	        		tree = (Tree)i.next();
            		fileWriter.write(tree.newickString(false, true, null) + ";\n");
        	    }
	    
		    assert tree != null: "Empty tree list sent to writeTreeFile";
	    	
		    fileWriter.close();
		    
        } catch (IOException e) {
			throw new ExternalEnvironmentException("Error writing tree file");
        }
        
        return null;
//	    return tree.characterMatrix();
	}

	public static void writeWeightsFile(CharacterMatrix matrix, 
        FileWriter fileWriter) throws IOException {
	    
	    for (int i = 0, count = matrix.characterCount(); i < count; i++ ) {
	        char weightCharacter = (char) (matrix.weight(i) + '0');
	        fileWriter.write(weightCharacter + " ");
	    }
	    
	    fileWriter.close();
	}

    public static void writeWeightsFile(WeightVector weightVector, 
        FileWriter fileWriter) throws IOException {
        
        for (int i = 0, count = weightVector.size(); i < count; i++ ) {
            char weightCharacter = (char) (weightVector.get(i) + '0');
            fileWriter.write(weightCharacter + " ");
        }
        
        fileWriter.close();
    }

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

	private static String _abbreviateTaxonName(String originalName) {
	    
	    StringBuffer abbreviatedName = new StringBuffer("          ");
	    abbreviatedName.insert(0,originalName).setLength(10);
	    return abbreviatedName.toString();
	}
    	
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private static final Pattern PHYLIP_TREE_STATEMENT_PATTERN = 
	    Pattern.compile("(\\([^;]*?);");

    private static final Pattern SPACE_SEPARATED_TOKENS_PATTERN = 
        Pattern.compile("(\\S+)", Pattern.DOTALL);
}
