/**
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.sdm.spa;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.expr.FileParameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import util.UUIDGen;
import util.XSLT;

//////////////////////////////////////////////////////////////////////////
//// XSLTTransformer
/*
 * Given an XSL transformation file, performs the specified transformation
 * on the input XML doc.
 *
 * <P> XSLT is designed for use as part of XSL, which is a stylesheet
 * language for XML. In addition to XSLT, XSL includes an XML vocabulary
 * for specifying formatting. XSL specifies the styling of an XML document
 * by using XSLT to describe how the document is transformed into another
 * XML document that uses the formatting vocabulary.
 * (ref: http://www.w3.org/TR/xslt)
 *
 * <P> Given an xml stream as input, XSLTTransformer  is used for  for
 * linking �almost but not quite fitting� output port and input port data
 * formats together. The actor produces an html stream that can be viewed
 * or queried using the BrowserUI actor.
 *
 * <P>The configuration window for the XSLTTransformer actor can be viewed
 * by double-clicking on the actor or by selecting �Configure� from the
 * right-click context menu. The window displays a GUI for browsing the
 * XSL script which will be used to perform the transformation.
 */

  /** Kindly correct src file to remove invalid characters in javadoc description
   @author Ilkay Altintas
   @version $Id: XSLTActor.java,v 1.18 2006/02/22 10:57:11 mangal Exp $
   @category.name data transformation
   @category.name local


   @entity.description "XSLT File Path"
        This parameter is to provide the path for the xsl file in user's
        computer.<P> This parameter can be filled in by specifying the
        file using the browse interface or by typing in the file.
        <I>The actor has <b>NO</B> default value. </I>
        Please double click on the actor to set it.

 */

public class XSLTActor
    extends TypedAtomicActor {

  public FileParameter xslFileName;

  /**
   * @entity.description String representation of the XML or HTMl input
   *   stream that needs to be transformed.
   */
  public TypedIOPort xmlIn;

  /**
   * @entity.description String representation of the output of the
   *    transformation.
   */
  public TypedIOPort htmlOut;

  public String htmlStr = "";

  public XSLTActor(CompositeEntity container, String name) throws
      NameDuplicationException, IllegalActionException {

    super(container, name);

    xmlIn = new TypedIOPort(this, "xmlIn", true, false);
    htmlOut = new TypedIOPort(this, "htmlOut", false, true);

    xmlIn.setTypeEquals(BaseType.STRING);
    htmlOut.setTypeEquals(BaseType.STRING);
    xslFileName = new FileParameter(this, "XSLT File Path");

    new Attribute(xmlIn, "_showName");
    new Attribute(htmlOut, "_showName");

    _attachText("_iconDescription",
                    "<svg>\n"
                    + "<rect x=\"0\" y=\"0\" "
                    + "width=\"60\" height=\"30\" "
                    + "style=\"fill:white\"/>\n"
                    + "<text x=\"10\" y=\"25\" "
                    + "style=\"font-size:16; fill:blue; font-family:SansSerif\">"
                    + "XSLT</text>\n"
                    + "</svg>\n");

  }

  public void fire() throws IllegalActionException {
    super.fire();

    String xmlStr = ( (StringToken) xmlIn.get(0)).stringValue();
//    System.out.println(xmlStr);
    //FIX ME: Apply XSLT transform to xmlStr and generate htmlStr...
    //htmlStr = xmlStr;

      //generate xml file name for storing the xml string.
      UUIDGen uuidgen = new UUIDGen();
      String fileName = uuidgen.generateUUID();
      System.out.println(fileName + ".xml");
      /* FIX ME: Need to delete the file if it is generated but an
               exception occurs before the end.
       */

      File xmlFile = new File(fileName + ".xml");

      try {
      BufferedWriter out = new BufferedWriter(new FileWriter(xmlFile.getPath()));
      out.write(xmlStr);
      out.close();
    } catch(Exception e) {
//      System.out.println("file out error");
    }
    File xslFile = xslFileName.asFile();

    XSLT XSLTransformer = new XSLT();
    htmlStr = XSLTransformer.Transform(xmlFile,xslFile);
    sendToken(htmlStr);
    xmlFile.delete();
  }

  protected void sendToken(Object o) throws ClassCastException,
      IllegalActionException {
    htmlOut.send(0, new StringToken( (String) o));
    //getToken will always return a stringToken here.
  }

  public boolean prefire() throws IllegalActionException {
    return super.prefire();
  }

  public boolean postfire() {
    htmlStr = "";
    return true;
  }

}
