/* An actor for connecting to SRB.

@Copyright (c) 2002-2003 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the
above copyright notice and the following two paragraphs appear in all
copies of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

                                                PT_COPYRIGHT_VERSION 2
                                                COPYRIGHTENDKEY
*/

package org.srb;

import java.io.IOException;

import ptolemy.actor.lib.Source;
import ptolemy.data.ObjectToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;
import edu.sdsc.grid.io.srb.SRBAccount;
import edu.sdsc.grid.io.srb.SRBFileSystem;

//////////////////////////////////////////////////////////////////////////
//// SRBConnect
/** An SRB connection actor.
 *
@author Bing Zhu & Efrat Jaeger
@version $Id: SRBConnect.java,v 1.10 2005/11/01 20:39:15 ruland Exp $
@since Ptolemy II 3.0.2
*/
public class SRBConnect extends Source {

 /** Construct an actor with the given container and name.
 *  @param container The container.
 *  @param name The name of this actor.
 *  @exception IllegalActionException If the actor cannot be contained
 *   by the proposed container.
 *  @exception NameDuplicationException If the container already has an
 *   actor with this name.
 */
public SRBConnect(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

    super(container, name);

    srbHost = new StringAttribute(this, "srbHost");
    srbPort = new StringAttribute(this, "srbPort");
    srbUserName = new StringAttribute(this, "srbUserName");
    srbPasswd = new StringAttribute(this, "srbPasswd");
    srbHomeCollection = new StringAttribute(this, "srbHomeCollection");
    srbMdasDomainHome = new StringAttribute(this, "srbMdasDomainHome");
    srbDefaultResource = new StringAttribute(this, "srbDefaultResource");

    // Set the type constraint.
    output.setName("SRBFileSystem");
    output.setTypeEquals(BaseType.GENERAL);
    output.setMultiport(true);
    new Attribute(output, "_showName");

    _attachText("_iconDescription",
            "<svg>\n"
            + "<rect x=\"0\" y=\"0\" "
            + "width=\"128\" height=\"30\" "
            + "style=\"fill:white\"/>\n"
            + "<text x=\"7\" y=\"24\" "
               + "style=\"font-size:12; fill:black; font-family:SansSerif\">"
               + "SRB$</text>\n"
            + "<text x=\"41\" y=\"25\" "
            + "style=\"font-size:16; fill:blue; font-family:SansSerif\">"
            + "CONNECT</text>\n"
            + "</svg>\n");

    }

  ///////////////////////////////////////////////////////////////////
  ////                     ports and parameters                  ////

   public StringAttribute srbHost;
   public StringAttribute srbPort;
   public StringAttribute srbUserName;
   public StringAttribute srbPasswd;
   public StringAttribute srbHomeCollection;
   public StringAttribute srbMdasDomainHome;
   public StringAttribute srbDefaultResource;

  ///////////////////////////////////////////////////////////////////
  ////                        public methods                     ////


  /** Connects to SRB and returns a connection reference.
   */
    public void fire() throws IllegalActionException {
        if (_first) {
            for (int i=0; i<output.getWidth(); i++)
                output.send(i, new ObjectToken(srbConnections[i]));
            _first = false;
        } else {
            for (int i=0; i<output.getWidth(); i++) {
                 // making sure that each connection is still alive
                 // if not create a new instance and send it.
                 try {
                     srbConnections[i].getHost();
                 } catch (Exception ex) {
                     try { // the connection was terminated - reconnect.
                         srbConnections[i] = new SRBFileSystem( srbAccount );
                         output.send(i, new ObjectToken(srbConnections[i]));
                     } catch (IOException ioex){
                         // if cannot reconnect to srb throw the failure reason.
                         throw new IllegalActionException(this,
                                 "SRB connection closed on channel " + i + " due to " +
                                ex.getMessage() + ".\n Could not reconnect to SRB: " +
                                ioex.getMessage());
                     }
                 }
             }
        }
    }

    /** Connect to SRB account.
     */
    public void initialize() throws IllegalActionException {
        super.initialize();

        _first = true;
        String _srbHost = srbHost.getExpression();
        String _srbPort = srbPort.getExpression();
        String _srbUserName = srbUserName.getExpression();
        String _srbPasswd = srbPasswd.getExpression();
        String _srbHomeCollection = srbHomeCollection.getExpression();
        String _srbMdasDomainHome = srbMdasDomainHome.getExpression();
        String _srbDefaultResource = srbDefaultResource.getExpression();

        // reset existing connections.
        if (srbConnections != null) {
            for (int i=0; i< srbConnections.length; i++) {
                srbConnections[i] = null;
            }
        }

        try {
            srbAccount = new SRBAccount(_srbHost, Integer.parseInt(_srbPort), _srbUserName, _srbPasswd,
                               _srbHomeCollection, _srbMdasDomainHome, _srbDefaultResource);

            int outWidth = output.getWidth();
            srbConnections = new SRBFileSystem[outWidth];
            for (int i=0; i<outWidth; i++) {
                try {
                    srbConnections[i] = new SRBFileSystem( srbAccount );
                } catch ( Exception ex ) {
                    ex.printStackTrace();
                    for (int j=0; j<=i; j++) {
                        srbConnections[j] = SRBUtil.closeConnection(srbConnections[j]);
                    }
                    throw new IllegalActionException(this,
                              "Could not create an SRB connection for channel " + i
                            + ": " + ex.getMessage());
                }
            }
        } catch ( Exception ex ) {
            ex.printStackTrace();
            throw new IllegalActionException(this, ex.getMessage());
        }
    }

    /** Disconnect from SRB.
     */
    public void wrapup() {
        _first = true;
        System.out.println(this.getName() + ":");
        for (int i=0; i<srbConnections.length; i++) {
            //System.out.println(srbConnections[i].toString());
            srbConnections[i] = SRBUtil.closeConnection(srbConnections[i]);
        }
        srbAccount = null;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private members                   ////

    /** SRB account
     */
    private SRBAccount srbAccount = null;

    /** SRB connections array.
     */
    private SRBFileSystem[] srbConnections = null;

    /** Indicates first iteration - send all connections
     */
    private boolean _first = true;

}
