/** Image AddText is an image manipulation actor for adding texts to
 * images such as title, sidenotes on the image etc.
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


package util;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.Token;
import ptolemy.data.StringToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.AWTImageToken;
import ptolemy.data.*;
import ptolemy.actor.parameters.PortParameter;

import java.io.*;
import java.util.*;
import java.awt.image.BufferedImage;
import java.net.MalformedURLException;
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.File;
import java.lang.Integer;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;


//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;



/**
 * ImageAddText can be used to add textual information to the image
 *
 @author Nandita Mangal
 @version $Id: ImageAddText.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */

public class ImageAddText extends TypedAtomicActor
{


    ////////////////////////////////////////////////////
    ////        Parameters and Ports                ////

     
    
  
    /*
     * Specifiy the font of text
     */
    public StringParameter textFont;
  
     /*
     * Specifiy the text color
     */
    public StringParameter textColor;


     /*
     * Specifiy the text style
     */
    public StringParameter textStyle;
    
    
    /*
     * Specifiy the text size
     */
    public StringParameter textSize;

     /*
      * Change text alignment
     */
     public StringParameter textAlignment;

     /*
     * The input image to have text written on
     */    
    public TypedIOPort imgSrc;

     /*
     * The final image
     */
    public TypedIOPort imgDest;


    /*
    * The text content
    */ 
    public PortParameter textContent; 

   
    /*
     * Specifiy the textPosition X Coordinate
     */
    public PortParameter textLocationX_coordinate;

    
    /*
     * Specifiy the textPosition Y Coordinate
     */
    public PortParameter textLocationY_coordinate;
 

  /**
   * constructor for ImageAddText with the given container and name.
   *
   *@param  container                     The container.
   *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageAddText(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
         imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);


	 imgDest = new TypedIOPort(this, "Dest Image", false, true);     
         imgDest.setTypeEquals(BaseType.OBJECT);
         imgDest.setMultiport(false);

         textContent = new PortParameter(this, "Text Content");  
        // textContent.setExpression("Text to be Added");

        textLocationX_coordinate = new PortParameter(this,"X Coordinate");
        textLocationY_coordinate = new PortParameter(this,"Y Coordinate");

	 textColor = new StringParameter(this,"Text Color:");
         textColor.addChoice("Red");
         textColor.addChoice("White");
         textColor.addChoice("Blue");
	 textColor.addChoice("Green");
         textColor.addChoice("Black");
         textColor.addChoice("Yellow");


         textFont = new StringParameter(this,"Text Font:");
         GraphicsEnvironment graphicsEvn = GraphicsEnvironment.getLocalGraphicsEnvironment();
        //get all the available fonts
        String availFonts[] = graphicsEvn.getAvailableFontFamilyNames(); 
        for(int i=0;i<availFonts.length;i++)
 		textFont.addChoice(availFonts[i]);
         textStyle = new StringParameter(this,"Text Style:");
         textStyle.addChoice("Plain");
         textStyle.addChoice("Bold");
	 textStyle.addChoice("Italic");
 
	 textSize = new StringParameter(this,"Text Size:");
	 
         textAlignment = new StringParameter(this, "Change Text Alignment:");
         textAlignment.addChoice("None");
	 textAlignment.addChoice("Left-Vertical");
	 textAlignment.addChoice("Right-Vertical");
         

  }


  //////////////////////////////////////////////////////////////////////
  ////                      public methods                          ///

  /**
   * Retrieve the input image and calculate parameters such as textSize,Color,location , font etc.
   * Draw the text string on the image after the above.
   * 
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();

    /* PlanarImages in JAI are not inherited from the Java Image class and hence
    *  are stored in normal ObjectToken and transferred through actor ports. However
    * normal Java AWT images are stored in AWTImageTokens.
    */

    BufferedImage srcBuffered=null;


    Object inputToken = imgSrc.get(0);

    if(inputToken instanceof AWTImageToken)  //then we know its an AWT Image (*not PlanarImage)
    {
    
    	Image src = (Image)((AWTImageToken)inputToken).getValue(); 
        srcBuffered  = toBufferedImage(src);    

    }
    else if(inputToken instanceof ObjectToken) //then its a PlanarImage probably.	
    {
	if (  ((ObjectToken)inputToken).getValue() instanceof PlanarImage)
	 {
		PlanarImage pi=(PlanarImage)((ObjectToken)inputToken).getValue(); 
		srcBuffered = pi.getAsBufferedImage();			  
         } 

    }

    //parse text position parameters.
    textLocationX_coordinate.update();
    textLocationY_coordinate.update();

    Token t_x = textLocationX_coordinate.getToken();
    Token t_y = textLocationY_coordinate.getToken();
    String x_str="";
    String y_str="";   

    if(t_x instanceof IntToken)
          x_str= new String ( ((IntToken)t_x).intValue() + "");
    else if(t_x instanceof DoubleToken)
	    x_str= new String ( ((DoubleToken)t_x).intValue() + "");

    if(t_y instanceof IntToken)
         y_str= new String ( ((IntToken)t_y).intValue() + "");
    else if (t_y instanceof DoubleToken)
	   y_str= new String ( ((DoubleToken)t_y).intValue() + "");

  
    float position_x  = Float.parseFloat(x_str);
    float position_y =  Float.parseFloat(y_str);


    String textAdd="";

    textContent.update();
    Token t = textContent.getToken();
    if(t instanceof StringToken)
        textAdd = ((StringToken)t).stringValue();

    Color color_=null;	

    if((textColor.stringValue()).equals("Red"))
           color_=Color.red;
    else if((textColor.stringValue()).equals("White"))
	 color_=Color.white;   
    else if((textColor.stringValue()).equals("Blue"))
	 color_=Color.blue;
    else if((textColor.stringValue()).equals("Green"))
	 color_=Color.green;
    else if((textColor.stringValue()).equals("Black"))
	 color_=Color.black;
    else if((textColor.stringValue()).equals("Yellow"))
	 color_=Color.yellow;

   int style_=0;

   if( (textStyle.stringValue()).equals("Plain") )
	style_=Font.PLAIN;
   else if( (textStyle.stringValue()).equals("Bold") )
	style_=Font.BOLD;
   else if( (textStyle.stringValue()).equals("Italic") )
	style_=Font.ITALIC;
	
   //create text font
    Font  font_= new Font(textFont.stringValue(),style_,Integer.parseInt(textSize.stringValue()));	

    String text_Alignment = textAlignment.stringValue();
    
    if(!( text_Alignment.equals("None") ) )
    {
	if(text_Alignment.equals("Left-Vertical"))
	{
		srcBuffered = rotateImage(srcBuffered,-1.570796327);	 	
	}
        else if(text_Alignment.equals("Right-Vertical"))
		srcBuffered = rotateImage(srcBuffered,+1.570796327);    

    }
   

    //draw the text on image
    Graphics2D graphics = srcBuffered.createGraphics();
    graphics.setColor(color_);
    graphics.setFont(font_);		
    graphics.drawString(textAdd , position_x ,position_y);
    graphics.dispose();

    if(text_Alignment.equals("Left-Vertical"))
    	srcBuffered = rotateImage(srcBuffered,+1.570796327);  
    else if(text_Alignment.equals("Right-Vertical"))
	srcBuffered = rotateImage(srcBuffered,-1.570796327);  

    //output to the port      
    imgDest.broadcast(new AWTImageToken(srcBuffered));
 
  }


  /**
  Rotate the image for text alignment purposes.
  */
  private BufferedImage rotateImage(BufferedImage srcBuffered, double radians)
  {
	AffineTransform tx = new AffineTransform();
	tx.rotate(radians,srcBuffered.getWidth()/2,srcBuffered.getHeight()/2);	
        AffineTransformOp op = new AffineTransformOp(tx,AffineTransformOp.TYPE_BILINEAR);
        srcBuffered = op.filter(srcBuffered,null);
	return srcBuffered;
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

   /**
     * Returns a buffered image with the contents of an image
     * @author: The Java Developers Almanac 1.4
     * @return: BufferedImage 
     */

    public static BufferedImage toBufferedImage(Image image) {
        if (image instanceof BufferedImage) {
            return (BufferedImage)image;
        }


        // This code ensures that all the pixels in the image are loaded
        image = new ImageIcon(image).getImage();

        // Determine if the image has transparent pixels; for this method's
        // implementation, see e661 Determining If an Image Has Transparent Pixels
        boolean hasAlpha = hasAlpha(image);
    
        // Create a buffered image with a format that's compatible with the screen
        BufferedImage bimage = null;
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        try {
            // Determine the type of transparency of the new buffered image
            int transparency = Transparency.OPAQUE;

     if (hasAlpha) {
                transparency = Transparency.BITMASK;
            }
    
            // Create the buffered image
            GraphicsDevice gs = ge.getDefaultScreenDevice();
            GraphicsConfiguration gc = gs.getDefaultConfiguration();
            bimage = gc.createCompatibleImage(
                image.getWidth(null), image.getHeight(null), transparency);
        } catch (HeadlessException e) {
            // The system does not have a screen
        }
    
        if (bimage == null) {
            // Create a buffered image using the default color model
            int type = BufferedImage.TYPE_INT_RGB;
            if (hasAlpha) {
                type = BufferedImage.TYPE_INT_ARGB;
            }
            bimage = new BufferedImage(image.getWidth(null), image.getHeight(null), type);
        }
    
        // Copy image to buffered image
        Graphics g = bimage.createGraphics();
    
        // Paint the image onto the buffered image
        g.drawImage(image, 0, 0, null);
        
        g.dispose();
    
        return bimage;
    }


    /**
     * This method returns true if the specified image has transparent pixels
     * @author: The Java Developers Almanac 1.4 
     * @return: boolean if contains transparent pixels or not.
     */

     public static boolean hasAlpha(Image image) {
        // If buffered image, the color model is readily available
        if (image instanceof BufferedImage) {
            BufferedImage bimage = (BufferedImage)image;
            return bimage.getColorModel().hasAlpha();
        }
    
        // Use a pixel grabber to retrieve the image's color model;
        // grabbing a single pixel is usually sufficient
         PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
        try {
            pg.grabPixels();
        } catch (InterruptedException e) {
     //while(true) {System.err.println("checknow");}
        }
    
        // Get the image's color model
        ColorModel cm = pg.getColorModel();

        return cm.hasAlpha();
    }

}

 
