/** Image Blur is an image manipulation actor for blurring the desired
 * image as per the desired level.
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package util;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.IntToken;
import ptolemy.data.Token;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.AWTImageToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.actor.parameters.PortParameter;
 
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.lang.Float;
import java.lang.Integer;

//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;
import java.awt.image.renderable.ParameterBlock;
import javax.media.jai.KernelJAI;

/**
 * ImageBlur can be used to blur a given image as per the 
 * desired level.
 *
 @author Nandita Mangal
 @version $Id: ImageBlur.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */

public class ImageBlur extends TypedAtomicActor
{

    ///////////////////////////////////////////////////////////////////////
    ////                   Parameters and Ports                       ////

   
    /*
     * Level of Smoothing Desired
     */
    public PortParameter blurLevel;
    /*
     * The input image to be manipulated
     */
    public TypedIOPort imgSrc;

    /*  
     * The output manipulated image
     */
    public TypedIOPort imgDest;


  /**
   * constructor for ImageBlur with the given container and name.
   *
   *@param  container                     The container.
    *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageBlur(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
	 imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);

         blurLevel = new PortParameter(this, "Blur Level");
         blurLevel.setExpression("2");
         blurLevel.addChoice("3");
         blurLevel.addChoice("4");
    
       
         imgDest = new TypedIOPort(this, "Dest Image", false, true);  
	 imgDest.setTypeEquals(BaseType.OBJECT);
         imgDest.setMultiport(false);
        
  }
    ////////////////////////////////////////////////////////////////////////
    ////                 public methods                                ////

 
  /**
   * The source image is retrieved from the input port of the actor
   * and the level of blur desired. Using JAI operator convolve and
   * the kernel matrix (based on blur level) we create a blurred image.
   * 
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();
    PlanarImage input=null;
    Object inputToken = imgSrc.get(0);
  
    if(inputToken instanceof AWTImageToken)
    {
    	Image src = (Image)((AWTImageToken)inputToken).getValue(); 
    	input= JAI.create("awtimage",src);	    
    }
    else if (inputToken instanceof ObjectToken)
    {
	input =(PlanarImage)((ObjectToken)inputToken).getValue();

    }
    
    int kernelSize=0;

    blurLevel.update();
    Token t = blurLevel.getToken();
    if(t instanceof IntToken)
    {
        kernelSize= ((IntToken)t).intValue();
    }

    //we create kernelMatrix such that all the entries add to 1     
    float[] kernelMatrix = new float[kernelSize*kernelSize];
   
    for(int k=0;k<kernelMatrix.length;k++)
         kernelMatrix[k] = 1.0f/(kernelSize*kernelSize);
    KernelJAI kernel = new KernelJAI(kernelSize,kernelSize,kernelMatrix);
    PlanarImage output = JAI.create("convolve", input, kernel);

    //output the blurred image. 
    imgDest.broadcast(new ObjectToken(output));


 
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

   


}
