/** Image Color Convert is an image manipulation actor converting certain
 * colors to desired color (e.g transparent)
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


package util;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.AWTImageToken;


import java.io.*;
import java.util.*;
import java.awt.image.BufferedImage;
import java.net.MalformedURLException;
import javax.imageio.*;
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.File;
import java.lang.Integer;
import java.awt.Color.*;
import java.awt.color.ColorSpace;




/**
 * ImageColorConvert can be used to convert colors on the image.
 @author Nandita Mangal
 @version $Id: ImageCrop.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */
public class ImageColorConvert extends TypedAtomicActor
{

  ///////////////////////////////////////////////////////////////////////
    ////                   Parameters and Ports                       ////

    /*
     * The original color in the image
    */
    public StringParameter originalColor;
  
    /*
     * The desired color in the image
    */	
    public StringParameter desiredColor;

    /*
     * The source image
    */
    public TypedIOPort imgSrc;

    /*
     * The final image
    */	
    public TypedIOPort imgDest;
 

  /**
   * constructor of imageColorConvertor with given container & name
   *
   *@param  container                     The container.
   *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageColorConvert(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
	 imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);

	 imgDest = new TypedIOPort(this, "Dest Image", false, true);  
	 imgDest.setTypeEquals(BaseType.OBJECT);
         imgDest.setMultiport(false);

	
         originalColor = new StringParameter(this, "Original color:");
         originalColor.addChoice("White");
         originalColor.addChoice("Red");
         originalColor.addChoice("Blue");
	 originalColor.addChoice("Green");

	 desiredColor = new StringParameter(this,"Desired color:");
	 desiredColor.addChoice("Transparent");
	 
	 
  }

   ////////////////////////////////////////////////////////////////////////
   ////                  public methods                               ////
  
  /**
   *Get the source image from the input port and change the desired color to
   * transparent. (Can be used in the process of overlaying images)
   * 
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();
    
    Image image = (Image)((AWTImageToken)(imgSrc.get(0))).getValue(); 
 
    String colorSrc= (String)originalColor.stringValue();
    String colorDesired = (String)desiredColor.stringValue();
  
    Color color =null;

    if(colorDesired.equals("Red"))
	color = new Color(0).red;
    else if(colorDesired.equals("Blue"))
	color = new Color(0).blue;
    else if(colorDesired.equals("Green"))
	color= new Color(0).green;
    else
	color=new Color(0).white;	


    if(colorDesired.equals("Transparent"))
	image=makeColorTransparent(image,color);
    else
	image=  makeColorChange(image, color);

    
    imgDest.broadcast(new AWTImageToken(image));
 
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }
  
  /**
   *Traverse through the image's pixels and convert the original color to desired
   * color.
   * @author: Real'sJavaHowTo   * 
   */

  public static Image makeColorChange (Image im, final Color colorSrc) {
    ImageFilter filter = new RGBImageFilter() {
      // the color we are looking for... Alpha bits are set to opaque
      public int markerRGB = colorSrc.getRGB() | 0xFF000000;
      public final int filterRGB(int x, int y, int rgb) {
        if ( ( rgb | 0xFF000000 ) == markerRGB ) {
          // Mark the alpha bits as zero - transparent
          return 0x00FFFFFF & rgb;
          }
        else {
          // nothing to do
          return rgb;
          }
        }
      }; 

    ImageProducer ip = new FilteredImageSource(im.getSource(), filter);
    return Toolkit.getDefaultToolkit().createImage(ip);
    }

   /**
   *Traverse through the image's pixels and convert the original color to transparent
   * @author: Real'sJavaHowTo
   * 
   */
  
   public static Image makeColorTransparent (Image im, final Color colorSrc) {
    ImageFilter filter = new RGBImageFilter() {

    
      // the color we are looking for... Alpha bits are set to opaque
      public int markerRGB = colorSrc.getRGB() | 0xFF000000;
      public final int filterRGB(int x, int y, int rgb) {
        if ( ( rgb | 0xFF000000 ) == markerRGB ) {
          // Mark the alpha bits as zero - transparent
          return 0x00FFFFFF & rgb;
          }
        else {
          // nothing to do
          return rgb;
          }
        }
      }; 

    ImageProducer ip = new FilteredImageSource(im.getSource(), filter);
    return Toolkit.getDefaultToolkit().createImage(ip);
    }

  
  ///////////////////////////////////////////////////////////////////
  ///                   private methods                         ////

  private Image src;
  private Image dest;

}

 