/**
 *  '$RCSfile: EcogridJavaToDigirJavaQueryTransformer.java,v $'
 *    Purpose: A Class that implements replication for digir
 *  Copyright: 2000 Regents of the University of California and the
 *             National Center for Ecological Analysis and Synthesis
 *    Authors: Chad Berkley
 *    Release: @release@
 *
 *   '$Author: ruland $'
 *     '$Date: 2005-12-20 20:10:55 $'
 * '$Revision: 1.20 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ecoinformatics.ecogrid.digir.impl;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.apache.axis.types.URI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.ecoinformatics.ecogrid.digir.impl.providerinfo.ProviderInfo;
import org.ecoinformatics.ecogrid.queryservice.query.*;


/* ----------------------------------------------------------------- 
 * Transforms an EcogridQuery to a Digir Query
 * ----------------------------------------------------------------- */
public class EcogridJavaToDigirJavaQueryTransformer
{
   private Hashtable operatorNameMatch  = null;
   
   public static final String DARWINSCHEMA1 = "http://digir.sourceforge.net/schema/conceptual/darwin/2003/1.0/darwin2.xsd";
   //public static final String DARWINSCHEMA2 = "http://bnhm.berkeley.museum/manis/DwC/darwin2jrw030315.xsd";
   public static final String DARWINSCHEMA2 = "http://kuecogrid.ittc.ku.edu:8080/portal/darwin2jrw030315.xsd";
   
   private int mDarwinCoreVersion = 2; 
   
   static Log logger = LogFactory.getLog(EcogridJavaToDigirJavaQueryTransformer.class.getName());
   
   /**
    * Default constructor
    *
    */
   public EcogridJavaToDigirJavaQueryTransformer()
   {
      setupOperatorMatchTable();
   }
   
   /**
    * Sets the version of the Darwin Core to be used
    * @return the version number
    */
   public int getDarwinCoreVersion()
   {
      return mDarwinCoreVersion;
   }
   
   /**
    * Sets the version of the Darwin Core to be used
    * @param aVersion
    */
   public void setDarwinCoreVersion(int aVersion)
   {
      mDarwinCoreVersion = aVersion;
   }
  
   public String transform(QueryType ecogridQuery, ProviderInfo pi ) throws Exception
   {
      Vector v = new Vector();
      v.add( pi );
      return transform( ecogridQuery, v );
   }

   /* ----------------------------------------------------------------- 
    * Method to transfer a ecogrid query java object to digir query java object
    * ----------------------------------------------------------------- */
   protected void transform(QueryType ecogridQuery, StringBuffer digirQuery) throws Exception
   {

      // set up query title
      String [] titleList = ecogridQuery.getTitle();
      String title = titleList[0];
      if (logger.isDebugEnabled()) {
         logger.debug("query title - " + title );
      }

      try {
         // transfer AND, OR, Condition to QueryGroup.
         transferCore(ecogridQuery, digirQuery);
         
      } catch (Exception e) {
         logger.error(e);
      }
      digirQuery.append("</request>");
      
      if (logger.isDebugEnabled()) {
         logger.debug("The digir query is: "+ digirQuery );
      }
      if (logger.isDebugEnabled()) {
         logger.debug("---- in the end of transform ----" );
      }
   } // transfer
  
   public String transform(QueryType ecogridQuery, Vector aProvidersInfo) throws Exception
   {
      logger.debug("---- in the begin of transform ----");
       
      StringBuffer digirQuery = new StringBuffer();
      digirQuery.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
      digirQuery.append("<request xmlns=\"http://digir.net/schema/protocol/2003/1.0\"");
      digirQuery.append(" xmlns:darwin=\"http://digir.net/schema/conceptual/darwin/2003/1.0\"");
      digirQuery.append(" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");
      digirQuery.append(" xsi:schemaLocation=\"http://digir.net/schema/protocol/2003/1.0");
      digirQuery.append(" http://digir.sourceforge.net/schema/protocol/2003/1.0/digir.xsd");
      digirQuery.append(" http://digir.net/schema/conceptual/darwin/2003/1.0");
      digirQuery.append(" "+ (mDarwinCoreVersion == 2 ? DARWINSCHEMA2 : DARWINSCHEMA1)+ "\">");
      digirQuery.append("<header>");
      digirQuery.append("<version>0.92</version>");
      digirQuery.append("<sendTime>"+DateUtil.formatNow()+"</sendTime>");
      digirQuery.append("<source>127.0.0.1</source>");
      for (Enumeration e = aProvidersInfo.elements(); e.hasMoreElements();) {
         ProviderInfo providerInfo = (ProviderInfo)e.nextElement();
         digirQuery.append("<destination resource=\""+providerInfo.getResourceCode()+"\">"+providerInfo.getUrl()+"</destination>");
      }
      digirQuery.append("<type>search</type>");
      digirQuery.append("</header>");
      
      transform(ecogridQuery, digirQuery);
      
      return digirQuery.toString();
   } // transfer
   
   
   /**
    * Returns whether a name is in the list
    * @param aFieldList
    * @param aFieldName
    * @return
    */
   private boolean isInList(Vector aFieldList, String aFieldName)
   {
      for (Enumeration e = aFieldList.elements();e.hasMoreElements();) {
         if (((String)e.nextElement()).equals(aFieldName)) {
            return true;
         }
      }
      return false;
   }
   
   /* ----------------------------------------------------------------- */
   /* Method to transfer returnfield */
   /* ----------------------------------------------------------------- */
   private void transferReturnField(QueryType ecogridQuery,
                                    StringBuffer digirQuery)
   {
      digirQuery.append("<records start=\"0\">");
      digirQuery.append("<structure>");
      digirQuery.append("<xsd:element name=\"record\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\">");
      digirQuery.append("<xsd:complexType>");
      digirQuery.append("<xsd:sequence>");
      
      logger.debug("start of transferReturnField");

      String [] returnFieldList = ecogridQuery.getReturnField();
      if (returnFieldList != null) {
         // convert array to Vector
         Vector fieldList = new Vector(Arrays.asList(returnFieldList));
         
         // Fix up the Lat/Long names for Darwin Core version two
         // eventually we will do a transform at some point for converting
         
         if (mDarwinCoreVersion == 2)  {
            for (int i=0;i<returnFieldList.length;i++) {
               if (returnFieldList[i].equals("Latitude")) {
                  returnFieldList[i] = "DecimalLatitude";
               } 
               else if (returnFieldList[i].equals("Longitude")) {
                  returnFieldList[i] = "DecimalLongitude";
               }
            }
         }
         
         // The following names are required for constructing the unique identifier
         // so if the original query did not ask for them we need to make sure we add them
         String[] requiredIdentifierNames = {"InstitutionCode", "CollectionCode", "CatalogNumber", "CatalogNumberText"};
         for (int i=0;i<requiredIdentifierNames.length;i++) {
            if (!isInList(fieldList, requiredIdentifierNames[i])) {
               fieldList.addElement(requiredIdentifierNames[i]);
            }
         }
         
         for (Enumeration e = fieldList.elements();e.hasMoreElements();) {
            String returnField = (String)e.nextElement();
            if (returnField.charAt(0) == '/') {
             returnField = returnField.substring(1);
            }
            digirQuery.append("<xsd:element ref=\"darwin:"+returnField+"\"/>");
            if (logger.isDebugEnabled()) {
               logger.debug("returnfield is " + returnField);
            } 
         }
      }
      
      digirQuery.append("</xsd:sequence>");
      digirQuery.append("</xsd:complexType>");
      digirQuery.append("</xsd:element>");
      digirQuery.append("</structure>");
      digirQuery.append("</records>");
      
      logger.debug("end of transferReturnField");
      
   } // tansferReturnField
  

   /* ----------------------------------------------------------------- */
   /* Method to transfer AND, OR, Condition to query group */
   /* ----------------------------------------------------------------- */
   private void transferCore(QueryType ecogridQuery, StringBuffer digirQuery) 
      throws Exception
   {
      // set up return fields
      
      digirQuery.append("<search><filter>");

      logger.debug("start of transferQueryGroup()");
      
      // first level, ecogridQuery can only have one AND or one OR or one Condition
      if (ecogridQuery.getAND() != null) {
         logger.debug("first level is AND");
         // And relation in first level
         digirQuery.append("<and>");
         handleANDChildren(digirQuery, ecogridQuery.getAND());
         digirQuery.append("</and>");
      }
      else if (ecogridQuery.getOR() != null) {
         logger.debug( "first level is OR" );
         // OR relation in first level
         digirQuery.append("<or>");
         handleORChildren(digirQuery, ecogridQuery.getOR());
         digirQuery.append("</or>");
      }
      else if (ecogridQuery.getCondition() != null) {
         logger.debug("first level is Condition");
         // Condition in first level
         handleConditionType(digirQuery, ecogridQuery.getCondition());
      }
      else {
         if (logger.isDebugEnabled()) {
            logger.debug("Wrong format of query");
         } 
         throw new Exception("Wrong Ecogrid Query Format");
      }

      logger.debug("end of transferQueryGroup()");
      
      digirQuery.append("</filter>");
      transferReturnField(ecogridQuery, digirQuery);
      
      digirQuery.append("<count>true</count></search>");
      
   } // tansferQueryGroup
   
   
   /* ----------------------------------------------------------------- */
   /* Handle ANDType's children, querygroup is same level to andtype */
   /* ----------------------------------------------------------------- */
   private void handleANDChildren(StringBuffer digirQuery, ANDType andParent)
      
   {
      logger.debug("start of handleANDChildren()(parent is AND)");
      // make sure parameter is not null
      if (digirQuery == null || andParent == null) {
         logger.debug("query group is null or andparent is null");
         return;
      } 

      // if ANDType's children is array, the code need be modified.
      if (andParent.getAND() != null) {
         for ( int i = 0; i<andParent.getAND().length; i++) {
            logger.debug("add AND childen to parent AND");
            handleANDType(digirQuery, andParent.getAND(i));
         }
      }
      
      if (andParent.getOR() != null) {
         for ( int i = 0; i<andParent.getOR().length; i++) {
            logger.debug("add OR childen to parent AND");
            handleORType(digirQuery, andParent.getOR(i));
         } 
      } 
      
      if (andParent.getCondition() != null) {
         for ( int i= 0; i<andParent.getCondition().length; i++) {
            logger.debug("add Condition childen to parent AND");
            handleConditionType(digirQuery, andParent.getCondition(i));
         } 
      } 

         logger.debug("end of handleANDChildren()(parent is AND)");
   } // handlANDChildren
  

   /* ----------------------------------------------------------------- */
   /* Handle ORType's children */
   /* ----------------------------------------------------------------- */
   private void handleORChildren(StringBuffer digirQuery, ORType orParent)
      
   {
      logger.debug("start of handleORChildren()(parent is OR)");

      // make sure parameter is not null
      if (digirQuery == null || orParent == null) {
         return;
      } 
      
      // if ANDType's children is array, the code need be modified
      if (orParent.getAND() != null) {
         for ( int i = 0; i<orParent.getAND().length; i++) {
            logger.debug("add AND childen to parent OR");
            handleANDType(digirQuery, orParent.getAND(i));
         } // for
      } // if

      if (orParent.getOR() != null) {
         for ( int i = 0; i<orParent.getOR().length; i++) {
            logger.debug("add OR childen to parent OR");
            handleORType(digirQuery, orParent.getOR(i));
         } // for
      } // if
      
      if (orParent.getCondition() != null) {
         for ( int i= 0; i<orParent.getCondition().length; i++) {
            logger.debug("add Condtion childen to parent OR");
            handleConditionType(digirQuery, orParent.getCondition(i));
         }
      }

      if (logger.isDebugEnabled()) {
         logger.debug("end of handleORChildren()(parent is OR)");
      }
   }
  

   /* ----------------------------------------------------------------- */
   /* Method to handle AND type, querygroup is higer level than andType */
   /* ----------------------------------------------------------------- */
   private void handleANDType(StringBuffer digirQuery, ANDType and)
   {
      logger.debug("start handleANDType()");
      // make sure parameter is not null
      if (digirQuery == null || and == null) {
         return;
      }
      
      digirQuery.append("<and>");
      // handle childen of kid
      handleANDChildren(digirQuery, and);
      digirQuery.append("</and>");
      logger.debug("end handleANDType()");
   } // handleANDType
   
   
   /* ----------------------------------------------------------------- */
   /* Method to handle OR type */
   /* ----------------------------------------------------------------- */
   private void handleORType(StringBuffer digirQuery, ORType or)
   {
      logger.debug("start handleORType()");
      // make sure parameter is not null
      if (or == null) {
         return;
      }
      digirQuery.append("<or>");
      // handle childen of kid
      handleORChildren(digirQuery, or);
      logger.debug("end handleORType()");
      digirQuery.append("</or>");
   } // handleORType
   

   /* ----------------------------------------------------------------- */
   /* Method to handle Condition type */
   /* ----------------------------------------------------------------- */
   private void handleConditionType(StringBuffer digirQuery, ConditionType condition)
   {
      logger.debug("start of handleConditionType()");
      if (condition == null) {
         return;
      }

      // get path
      String pathString = condition.getConcept();
      if (logger.isDebugEnabled()) {
         logger.debug("path express is " + pathString);
      } 
      
      // get operator
      OperatorType operator = condition.getOperator();
      String operatorString = operator.getValue();
      String digirOperator = lookupDigirOperator(operatorString);
      
      // get value
      String       value    = condition.get_value();
      if (logger.isDebugEnabled()) {
         logger.debug("value for path is " + value);
      }

      digirQuery.append("<"+digirOperator+"><darwin:"+pathString+">"+value+"</darwin:"+
                        pathString+"></"+digirOperator+">");

      logger.debug("end of handleConditionType()");  
  } // handleConditionType
  

   /* ----------------------------------------------------------------- */
   /* Method to find a digir operator for in input string */
   /* ----------------------------------------------------------------- */
   private String lookupDigirOperator(String input)
   {
      if (logger.isDebugEnabled()) {
         logger.debug("operator in ecogrid(input) is " + input);
      } 
      String operator = null;
      operator = (String) operatorNameMatch.get(input);
      if (logger.isDebugEnabled()) {
         logger.debug("operator in digir(output) is " + operator);
      }
      return operator;
      
   } // lookupDigirOperator
  

   /* ----------------------------------------------------------------- */
   /* Method to set up hashtable for search mode (operator) */
   /* ----------------------------------------------------------------- */
   private void setupOperatorMatchTable()
   {
      operatorNameMatch = new Hashtable();
      // "LIKE";
      operatorNameMatch.put(OperatorType._value1, "like");
      
      // "NOT LIKE";
      operatorNameMatch.put(OperatorType._value2, "notLike");
      
      // "EQUALS";
      operatorNameMatch.put(OperatorType._value3, "equals");
      
      // "NOT EQUALS"
      operatorNameMatch.put(OperatorType._value4, "notEquals");
      
      // "LESS THAN";
      operatorNameMatch.put(OperatorType._value5, "lessThan");
      
      // "LESS THAN OR EQUALS";
      operatorNameMatch.put(OperatorType._value6, "lessThanEquals");
      
      // "GREATER THAN";
      operatorNameMatch.put(OperatorType._value7, "greaterThan");
      
      // "GREATER THAN OR EQUALS";
      operatorNameMatch.put(OperatorType._value8, "greaterThanEquals");
      
   }
}
