/**
 *  '$RCSfile: EcoRegMetacat.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: rspears $'
 *     '$Date: 2004-06-10 13:08:19 $'
 * '$Revision: 1.6 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
 
package org.ecoinformatics.ecogrid.registry.impl;

import java.util.Vector;

import edu.ucsb.nceas.metacat.client.*;
import edu.ucsb.nceas.utilities.IOUtil;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.registry.EcoRegInterface;
import org.ecoinformatics.ecogrid.registry.EcoRegConstants;
import org.ecoinformatics.ecogrid.registry.RegEntry;

// For Generic XML Stuff
import java.io.*;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

// For generating the doc id
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Date;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

/**
 * Implementation of a registry each records contains "name", "registry name", and "registry address"
 * @author  Rod Spears
 */
public class EcoRegMetacat implements EcoRegInterface {
  
  protected final String ENTRY   = "entry";
  protected final String REGNAME = "regname";
  protected final String REGADDR = "regaddr";

  protected final String NAME_XPATH     = "dataset/registry/identifier/name";
  protected final String RESOURCE_XPATH = "dataset/registry/identifier/resource";

  //private String  _metacatUrl = "http://knb.ecoinformatics.org/knb/metacat";
  private String  _metacatUrl = "http://metacat.nceas.ucsb.edu/knb/servlet/metacat";
  private String  _username   = "uid=rods,o=KU,dc=ecoinformatics,dc=org";
  private String  _password   = "@metacat.password@";
  private String  _prefix     = "ecogridreg";
  private Metacat _metacatObj = null;

  /**
   * Default Constructor
   */
  public EcoRegMetacat() {
    EcogridUtils.setDebug(true);
    EcogridUtils.debugMessage("Connecting to Metacat....");
    if (init()) {
      if (login()) {
        EcogridUtils.debugMessage("Logged into Metacat is: " + _metacatUrl+" just fine.");
      } else {
        EcogridUtils.debugMessage("Metacat login failed ["+_username+"]["+_password+"]");
      }
    } else {
      EcogridUtils.debugMessage("Metacat init failed.");
    }
  }
  
  /**
   * Establish a testing framework by initializing appropriate objects
   * @return status of whether the init was performed correctly 
   * (i.e. meaning was the metacat obj correctly obtained from the factory)
   */
  public boolean init()
  {
    try {
      EcogridUtils.debugMessage("Metacat URL: " + _metacatUrl);
      _metacatObj = MetacatFactory.createMetacatConnection(_metacatUrl);
      return true;

    } catch (MetacatInaccessibleException mie) {
      EcogridUtils.debugMessage("Metacat is: " + _metacatUrl);
      EcogridUtils.debugMessage("Metacat connection failed." + mie.getMessage());
    }
    return false;
  }
   
  /**
   * Login with valid credentials
   */
  public boolean login()
  {
    // Try a valid login
    try {
      String response = _metacatObj.login(_username, _password);
      EcogridUtils.debugMessage("Login response: " + response);

      if (response == null || response.indexOf("<login>") == -1) {
        return false;
      }

      String sessionId = _metacatObj.getSessionId();
      EcogridUtils.debugMessage("Session ID: " + _metacatObj.getSessionId());

      if (sessionId == null || response.indexOf(sessionId) == -1) {
        return false;
      }
      return true;

    } catch (MetacatAuthException mae) {
      EcogridUtils.debugMessage("Authorization failed:\n" + mae.getMessage());

    } catch (MetacatInaccessibleException mie) {
      EcogridUtils.debugMessage("Metacat Inaccessible:\n" + mie.getMessage());

    }
    return false;
  }

  /**
   * Test the logout() function. When logout, user will be public, it couldn't
   * insert a document.
   */
  public boolean logout()
  {
    try {
      _metacatObj.logout();
      return true;

    } catch (MetacatException me) {
      EcogridUtils.debugMessage("Metacat Error:\n" + me.getMessage());

    } catch (Exception e) {
      EcogridUtils.debugMessage("General exception:\n" + e.getMessage());

    }
    return false;
  }

  /**
   * Create a hopefully unique docid for testing insert and update. Does
   * not include the 'revision' part of the id.
   *
   * @return a String docid based on the current date and time
   */
  private String generateDocid()
  {
    final int millisecs = 60 * 60 * 1000;

    StringBuffer docid = new StringBuffer(_prefix);
    docid.append(".");

    // Create a calendar to get the date formatted properly
    String[]       ids = TimeZone.getAvailableIDs(-8 * millisecs);
    SimpleTimeZone pdt = new SimpleTimeZone(-8 * millisecs, ids[0]);

    pdt.setStartRule(Calendar.APRIL, 1, Calendar.SUNDAY, 2 * millisecs);
    pdt.setEndRule(Calendar.OCTOBER, -1, Calendar.SUNDAY, 2 * millisecs);

    Calendar calendar  = new GregorianCalendar(pdt);
    Date     trialTime = new Date();
    calendar.setTime(trialTime);

    docid.append(calendar.get(Calendar.YEAR));
    docid.append(calendar.get(Calendar.DAY_OF_YEAR));
    docid.append(calendar.get(Calendar.HOUR_OF_DAY));
    docid.append(calendar.get(Calendar.MINUTE));
    docid.append(calendar.get(Calendar.SECOND));

    return docid.toString();
  }

  /**
    * Construct lookup entry query
    * @param aName name of service
    * @return the query document as a string
    */
  public String getLookupEntry(String aName)
  {

    StringBuffer queryStr = new StringBuffer();
    queryStr.append("<?xml version=\"1.0\"?>");
    queryStr.append("<pathquery version=\"1.2\">");
    queryStr.append("  <querytitle>Ecogrid Registry </querytitle>");
    queryStr.append("  <filterdoctype>-//ecoinformatics.org//ecoreg-dataset-1.0//EN</filterdoctype>");
    queryStr.append("  <returndoctype>-//ecoinformatics.org//ecoreg-dataset-1.0//EN</returndoctype>");
    queryStr.append("  <returnfield>dataset/registry/identifier/name</returnfield>");
    queryStr.append("  <returnfield>dataset/registry/identifier/resource</returnfield>");
    queryStr.append("  <returnfield>docid</returnfield>");
    //queryStr.append("  <querygroup operator=\"INTERSECT\">");
    //queryStr.append("    <querygroup operator=\"UNION\">");
    queryStr.append("      <querygroup operator=\"UNION\">");
    queryStr.append("        <queryterm searchmode=\"equals\" casesensitive=\"false\">");
    queryStr.append("          <value>" + aName + "</value>");
    queryStr.append("          <pathexpr>"+NAME_XPATH+"</pathexpr>");
    queryStr.append("        </queryterm>");
    queryStr.append("      </querygroup>");
    //queryStr.append("    </querygroup>");
    //queryStr.append("  </querygroup>");
    queryStr.append("</pathquery>");

    return queryStr.toString();
  }


  /**
    * Constructs the XML document representing the registry entry
    * @param aEntry Registry Entry
    */
  public String constructRegEntryDoc(RegEntry aEntry)
  {
    if (aEntry == null) return null;

    StringBuffer docStr = new StringBuffer();

    docStr.append("<?xml version=\"1.0\"?>");
    docStr.append("<!DOCTYPE dataset PUBLIC \"-//ecoinformatics.org//ecoreg-dataset-1.0//EN\" \"http://129.237.201.166:8080/ogsa/ecogridregistry.205.22.dtd\">");
    docStr.append("<dataset>");
    docStr.append("  <registry>");
    docStr.append("    <identifier>");
    docStr.append("      <name>" + aEntry.getName() + "</name>");
    docStr.append("      <resource>" + aEntry.getRegAddr() + "</resource>");
    docStr.append("    </identifier>");
    docStr.append("    <metadata>");
    docStr.append("      <information></information>");
    docStr.append("    </metadata>");
    docStr.append("  </registry>");
    docStr.append("</dataset>");

    return docStr.toString();
  }

  /**
  * Creates an entry in the registry, "name" must be unique
  * @param aName name of service
  * @param aRegName urn name of service
  * @param aRegAddr web address of the service
  * @return status code from EcoRegInterface's error codes
  */
  public int registerEntry(String aName, String aRegName, String aRegAddr)
  {
    //EcogridUtils.debugMessage("******* registerEntry [" + aName + "]");
    RegEntry entry = getEntry(aName);
    if (entry == null)
    {
      String regEntryId = generateDocid() + ".1";

      entry = new RegEntry(aName, regEntryId, aRegAddr);
      String docStr = constructRegEntryDoc(entry);
      try {

        //EcogridUtils.debugMessage("******* Trying Registered : " + aName + "   " + entry.getRegName());

        String response = _metacatObj.insert(regEntryId, new StringReader(docStr), null);
        if (response.indexOf("<success>") == -1 || response.indexOf(regEntryId) == -1) {
          EcogridUtils.debugMessage("******* resultset on error : " + response);
          return EcoRegConstants.kError;
        }
        
        //EcogridUtils.debugMessage("******* Registered OK : " + aName);
        //**********************
        // XXX HACK!
        logout();
        login();
        //**********************
        return EcoRegConstants.kNoError;

      } catch (MetacatInaccessibleException mie) {
        EcogridUtils.debugMessage("Metacat Inaccessible:\n" + mie.getMessage());

      } catch (InsufficientKarmaException ike) {
        EcogridUtils.debugMessage("Insufficient karma:\n" + ike.getMessage());

      } catch (MetacatException me) {
        EcogridUtils.debugMessage("Metacat Error:\n" + me.getMessage());

      } catch (Exception e) {
        EcogridUtils.debugMessage("General exception:\n" + e.getMessage());
      }

      return EcoRegConstants.kError;

    } else {
      EcogridUtils.debugMessage("******* registerEntry entry [" + aName + "] was found and shouldn't have been.");
    }
    return EcoRegConstants.kDupEntryError;
  }

  /**
   * Updates existing entry the registry database
   * @param aName name of service
   * @param aRegName urn name of service
   * @param aRegAddr web address of the service
   * @return status code from EcoRegInterface's error codes
   */
  public int updateEntry(String aName, String aRegName, String aRegAddr)
  {
    //EcogridUtils.debugMessage("******* updateEntry [" + aName + "]");
    RegEntry entry = getEntry(aName);
    //EcogridUtils.debugMessage("******* updateEntry entry[" + entry + "]");
    if (entry != null) {
      int status = removeEntry(aName);
      //EcogridUtils.debugMessage("******* updateEntry status [" + status + "]");
      if (status == EcoRegConstants.kNoError)
      {
        return registerEntry(aName, aRegName, aRegAddr);

      } else {
        return status;
      }
    }
    //EcogridUtils.debugMessage("******* DONE updateEntry [" + aName + "]");
    return EcoRegConstants.kNotFoundError;
  }

  /**
   * Removes an entry in the registry, "name" must be unique
   * @param aName name of service
   * @return status code from EcoRegInterface's error codes
   */
  public int removeEntry(String aName)
  {

    //EcogridUtils.debugMessage("******* removeEntry - [" + aName + "]");
    RegEntry entry = getEntry(aName);
    if (entry != null) {
      //EcogridUtils.debugMessage("\n*******\nAbout to delete : [" + aName + "]   [" + entry.getRegName()+"]");
      try {
        String response = _metacatObj.delete(entry.getRegName());
        EcogridUtils.debugMessage(response);

        if (response.indexOf("<success>") != -1) {
          return EcoRegConstants.kNoError;
        }

      } catch (MetacatInaccessibleException mie) {
        EcogridUtils.debugMessage("Metacat Inaccessible:\n" + mie.getMessage());

      } catch (InsufficientKarmaException ike) {
        EcogridUtils.debugMessage("Insufficient karma:\n" + ike.getMessage());

      } catch (MetacatException me) {
        EcogridUtils.debugMessage("Metacat Error:\n" + me.getMessage());

      } catch (Exception e) {
        EcogridUtils.debugMessage("General exception:\n" + e.getMessage());
      }
      return EcoRegConstants.kError;
    }

    return EcoRegConstants.kNotFoundError;
  }

  /**
   * Looks up an entry by name
   * @param aName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntry(String aName)
  {
    //EcogridUtils.debugMessage("******* getEntry - [" + aName + "]");
    try {
      StringReader strReader     = new StringReader(getLookupEntry(aName));
      Reader       resultsReader = _metacatObj.query(strReader);

      String result = IOUtil.getAsString(resultsReader, true);
      if (result == null || result.length() == 0) {
        EcogridUtils.debugMessage("****** getEntry - resultset was null or zero len ");
        return null;
      }

      //System.err.println("*************\ngetEntry result:\n" + result + "\n***************");

      String name     = "";
      String resource = "";
      String regName  = "";

      Document docRoot = EcogridUtils.convertXMLStr2DOM(result);
      if (docRoot != null) {
        Node docNode = EcogridUtils.findNode(docRoot, "document");
        if (docNode != null) {
          NodeList list = docNode.getChildNodes();
          for (int i=0;i<list.getLength();i++) 
          {
            Node child = list.item(i);
            if (child != null)
            {
              String nodeName = child.getNodeName();
              if (nodeName != null && nodeName.compareTo("docid") == 0) {
                regName = EcogridUtils.getNodeValue(child);
              }

              String attrVal = EcogridUtils.findAttrValue(child, "name");
              if (attrVal != null) {
                if (attrVal.compareTo("dataset/registry/identifier/name") == 0) {
                  name = EcogridUtils.getNodeValue(child);

                } else if (attrVal.compareTo("dataset/registry/identifier/resource") == 0) {
                  resource = EcogridUtils.getNodeValue(child);
                }
              } else {
                EcogridUtils.debugMessage("****** getEntry - attrVal == null");
              }
            }
          }
        } else {
          EcogridUtils.debugMessage("****** getEntry - docNode == null");
        }
      } else {
        EcogridUtils.debugMessage("****** getEntry - docRoot == null");
      }

      //System.err.println("getEntry name[" + name+"]  regName["+regName+"]  resource["+resource+"]");
      if (name.length() > 0) {
        return new RegEntry(name, regName, resource);
      }

    } catch (MetacatInaccessibleException mie) {
      EcogridUtils.debugMessage("Metacat Inaccessible:\n" + mie.getMessage());

    } catch (Exception e) {
      EcogridUtils.debugMessage("General exception:\n" + e.getMessage());
    }

    EcogridUtils.debugMessage("****** getEntry - returning null");
    return null;
  }

  /**
   * Looks up a entry by Registration name (URN)
   * @param aRegName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntryByRegName(String aRegName) 
  {
    return null;
  }
 
  /**
    * Construct lookup entry query
    * @param aName name of service
    * @return the query document as a string
    */
  public String getLookupAllEnties()
  {

    StringBuffer queryStr = new StringBuffer();
    queryStr.append("<?xml version=\"1.0\"?>");
    queryStr.append("<pathquery version=\"1.2\">");
    queryStr.append("  <querytitle>Ecogrid Registry </querytitle>");
    queryStr.append("  <filterdoctype>-//ecoinformatics.org//ecoreg-dataset-1.0//EN</filterdoctype>");
    queryStr.append("  <returndoctype>-//ecoinformatics.org//ecoreg-dataset-1.0//EN</returndoctype>");
    queryStr.append("  <returnfield>dataset/registry/identifier/name</returnfield>");
    queryStr.append("  <returnfield>dataset/registry/identifier/resource</returnfield>");
    queryStr.append("  <returnfield>docid</returnfield>");
    queryStr.append("  <querygroup operator=\"UNION\">");
    queryStr.append("    <queryterm searchmode=\"contains\" casesensitive=\"false\">");
    queryStr.append("      <value>%</value>");
    queryStr.append("      <pathexpr>"+NAME_XPATH+"</pathexpr>");
    queryStr.append("    </queryterm>");
    queryStr.append("  </querygroup>");
    queryStr.append("</pathquery>");

    return queryStr.toString();
  }

  /**
   * Returns an array list of all registry entries
   * @return returns an XML string of all the entries in the registry
   */
  public RegEntry[] getList()
  {
   
    Vector items = new Vector();
    //EcogridUtils.debugMessage("\n******\nAbout to getList : ");

    try {
      StringReader strReader     = new StringReader(getLookupAllEnties());
      Reader       resultsReader = _metacatObj.query(strReader);

      String result = IOUtil.getAsString(resultsReader, true);
      if (result == null || result.length() == 0) {
        return null;
      }

      //System.err.println("*************\ngetList result:\n" + result + "\n***************");


      Document docRoot = EcogridUtils.convertXMLStr2DOM(result);
      //if (docRoot != null) {
      //EcogridUtils.printNode(docRoot, 0);
      Node resultsetNode = EcogridUtils.findNode(docRoot, "resultset");
      if (resultsetNode != null) {
        NodeList rsNodelist = resultsetNode.getChildNodes();
        for (int ii=0;ii<rsNodelist.getLength();ii++) 
        {
          Node docChild = rsNodelist.item(ii);
          if (docChild != null && docChild.getNodeName().compareTo("document") == 0) {
            String name     = "";
            String resource = "";
            String regName  = "";
            NodeList list = docChild.getChildNodes();
            for (int i=0;i<list.getLength();i++) 
            {
              Node child = list.item(i);
              if (child != null)
              {
                String nodeName = child.getNodeName();
                if (nodeName != null && nodeName.compareTo("docid") == 0) {
                  regName = EcogridUtils.getNodeValue(child);
                }

                String attrVal = EcogridUtils.findAttrValue(child, "name");
                if (attrVal != null) {
                  if (attrVal.compareTo(NAME_XPATH) == 0) {
                    name = EcogridUtils.getNodeValue(child);

                  } else if (attrVal.compareTo(RESOURCE_XPATH) == 0) {
                    resource = EcogridUtils.getNodeValue(child);
                  }
                }
              }
            }
            //System.err.println("\n getList name[" + name+"]  regName["+regName+"]  resource["+resource+"]");
            items.add(new RegEntry(name, regName, resource));
          }
        }
      }

      if (items.size() > 0) {
        RegEntry[] array = new RegEntry[items.size()];
        for (int i=0;i<items.size();i++)
        {
          RegEntry entry = (RegEntry)items.elementAt(i);
          array[i] = new RegEntry(entry);
        }
        return array;

      } else {
        EcogridUtils.debugMessage("No items were found in vector.");
      }

    } catch (MetacatInaccessibleException mie) {
      EcogridUtils.debugMessage("Metacat Inaccessible:\n" + mie.getMessage());

    } catch (Exception e) {
      EcogridUtils.debugMessage("General exception:\n" + e.getMessage());
    }

    return null;
  }

  /**
   * Returns an XML list of all registry entries
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLListing()
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.setLength(0);
    strBuf.append("<listing>\n");
    RegEntry[] array = getList();
    if (array != null) {
      for (int i=0;i<array.length;i++)
      {
        array[i].toXMLStrBuf(strBuf);
      }
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Returns an XML string representing the entry
   * @param aName name of service
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByName(String aName)
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.setLength(0);
    strBuf.append("<listing>\n");
    RegEntry entry = getEntry(aName);
    if (entry != null)
    {
      entry.toXMLStrBuf(strBuf);
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Returns an XML string representing the entry
   * @param aRegName name of entry to be returned
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByRegName(String aRegName)
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.append("<listing>\n");
    RegEntry entry = getEntryByRegName(aRegName);
    if (entry != null)
    {
      entry.toXMLStrBuf(strBuf);
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Closes registry - writes it to disk
   */
  public boolean closeRegistry()
  {
    return _metacatObj != null;
  }

  /**
   * Opens registry by name (initially it is the file name)
   * @param aFileName name of DB to be opened
   * @return returns true for success, false for failure
   */
  public boolean openRegistry(String aFileName) {

    //if (init() && _metacatObj != null) {
    //  return login();
    //}

    return _metacatObj != null;
  }

  protected void finalize()
  {
    try {
      _metacatObj.logout();
    } catch (MetacatException me) {
      EcogridUtils.debugMessage("Metacat Error:\n" + me.getMessage());

    } catch (Exception e) {
      EcogridUtils.debugMessage("General exception:\n" + e.getMessage());
    }
  }
  
}
