/**
 *    '$RCSfile: EcogridProcessor.java,v $'
 *
 *     '$Author: rspears $'
 *       '$Date: 2004-10-27 13:21:33 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.ecoinformatics.ecogrid.registry.ui;

import java.net.URL;
import java.util.Enumeration;
import java.util.Vector;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.client.EcogridAuthClient;
import org.ecoinformatics.ecogrid.registry.stub.EcoRegInterfaceLevelOnePortType;
import org.ecoinformatics.ecogrid.registry.stub.EcogridRegEntryType;
import org.ecoinformatics.ecogrid.registry.stub._EcogridGetAllRegServicesRequestElement;
import org.ecoinformatics.ecogrid.registry.stub._EcogridGetAllRegServicesResponseElement;
import org.w3c.dom.Document;
/**
 * Represents an item in the cache
 */
public class EcogridProcessor implements Runnable
{
    private static final String AUTH_SERVICE = "http://sasa.sdsc.edu:8080/ogsa/services/org/ecoinformatics/ecogrid/EcoGridLevelOneAuthenticationService";
    
    public static final int PROCESS_OK       = 0;
    public static final int PROCESS_BUSY     = 1;
    public static final int PROCESS_ERROR    = 2;
    public static final int PROCESS_COMPLETE = 3;
    public static final int PROCESS_NOTCOMP  = 4;
    
    public static final int DO_NONE      = 0;
    public static final int DO_LOGIN     = 1;
    public static final int DO_LOGOUT    = 2;
    public static final int DO_GET_ALL   = 3;
    public static final int DO_ADD       = 4;
    public static final int DO_UPDATE    = 5;
    public static final int DO_REMOVE    = 6;
    
    public  Thread                          mThread       = null;
    
    private Vector                          mListeners    = new Vector();
    private Vector                          mNotifierList = new Vector();
    private EcogridRegEntryType             mRegEntry     = null;
    private EcogridRegEntryType[]           mRegEntries   = null;
    
    private int                             mWorkType     = DO_NONE;
    private int                             mStatus       = PROCESS_NOTCOMP;
    private String                          mStatusStr    = null;
    
    private EcoRegInterfaceLevelOnePortType mRegistry     = null;
    private String                          mSessionId    = null;
    private String                          mUsername     = null;
    private String                          mPassword     = null;
    
    /**
     * Constructor 
     *
     */
    public EcogridProcessor(EcoRegInterfaceLevelOnePortType aRegistry,
                            String              aSessionId,
                            int                 aWorkType, 
                            EcogridRegEntryType aRegEntry,
                            EcogridProcessorListener aListener)
    {
        mRegistry  = aRegistry;
        mSessionId = aSessionId;
        mWorkType  = aWorkType;
        mRegEntry  = aRegEntry;
        addListener(aListener);
    }
    

    /**
     * Constructor for logging in
     * @param aRegistry
     * @param aUsername
     * @param aPassword
     * @param aListener
     */
    public EcogridProcessor(EcoRegInterfaceLevelOnePortType aRegistry,
                            String              aUsername,
                            String              aPassword,
                            EcogridProcessorListener aListener)
    {
        mRegistry  = aRegistry;
        mWorkType  = DO_LOGIN;
        mUsername  = aUsername;
        mPassword  = aPassword;
        addListener(aListener);
    }
    
    /**
     * Constructor for loggin out
     * @param aRegistry
     * @param aSessionId
     * @param aListener
     */
    public EcogridProcessor(EcoRegInterfaceLevelOnePortType aRegistry,
                            String              aSessionId,
                            EcogridProcessorListener aListener)
    {
        mWorkType  = DO_LOGOUT;
        mSessionId = aSessionId;
        addListener(aListener);
    }
    
   
    /**
     * Add a listener 
     * @param aListener the listener to add
     */
    public void addListener(EcogridProcessorListener aListener)
    {
        if (aListener != null)
        {
            mListeners.addElement(aListener);
        }
    }
    
    /**
     * Removes a listener 
     * @param aListener the listener to remove
     */
    public void removeListener(EcogridProcessorListener aListener)
    {
        if (aListener != null)
        {
            mListeners.removeElement(aListener);
        }
    }
    
    /**
     * Removes all the listeners
     */
    public void removeAllListeners()
    {
        mListeners.clear();
    }
    
    /**
     * Notifies the listener that the "getting" of the data has completed
     *
     */
    public synchronized void notifyListeners()
    {
        mNotifierList = new Vector(mListeners);
        
        for (Enumeration e = mNotifierList.elements(); e.hasMoreElements();)
        {
            EcogridProcessorListener l = (EcogridProcessorListener)e.nextElement();
            l.complete(this);
        }
    }
    
    /**
     * Return the status of getting the data
     * @return
     */
    public boolean isReady()
    {
        return mStatus == PROCESS_OK || mStatus == PROCESS_COMPLETE;
    }
    
    /**
     * @return Returns the mStatus.
     */
    public int getStatus()
    {
        return mStatus;
    }
    
    /**
     * @return Returns the mStatusStr.
     */
    public String getStatusStr()
    {
        return mStatusStr;
    }
    
    /**
     * 
     * @return
     */
    public int getWorkType()
    {
        return mWorkType;
    }
    
    /**
     * 
     * @return
     */
    public String getSessionId()
    {
      return mSessionId;
    }
    
    /**
     * 
     * @return
     */
    public EcogridRegEntryType getRegEntry()
    {
        return mRegEntry;
    }
    
    /**
     * 
     * @return
     */
    public EcogridRegEntryType[] getRegEntries()
    {
        return mRegEntries;
    }
    
    
    /**
     * Returns whether it is busy getting the data
     * @return
     */
    public boolean isBusy()
    {
        return mStatus == PROCESS_BUSY;
    }
    
    /**
     * 
     *
     */
    protected void doLogin()
    {
        try
        {
            URL GSH = new URL(AUTH_SERVICE);
            
            // create a client object
            EcogridAuthClient client = new EcogridAuthClient(GSH);
            client.createEcoGridAuthLevelOnePortType();
            
            mSessionId = client.login_action(mUsername, mPassword);
            mStatus = (mSessionId != null && mSessionId.length() > 0) ? PROCESS_OK : PROCESS_ERROR;
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            System.out.println("Exception in login : "+ e.getMessage());
            e.printStackTrace();
            mStatus = PROCESS_ERROR;
        }
    }
    
    /**
     * 
     *
     */
    protected void doLogout()
    {
        try
        {
            URL GSH = new URL(AUTH_SERVICE);
            // create a client object
            EcogridAuthClient client = new EcogridAuthClient(GSH);
            client.createEcoGridAuthLevelOnePortType();
            client.logout_action(mSessionId);
            mStatus = PROCESS_OK;
            
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            System.out.println("Exception in logout : "+ e.getMessage());
            e.printStackTrace();
            mStatus = PROCESS_ERROR;
        }
    }
    
    /**
     * 
     *
     */
    protected void doGetAllServices()
    {
        try
        {
            if (mRegistry != null)
            {
                _EcogridGetAllRegServicesRequestElement allSviceRsqtElement = new _EcogridGetAllRegServicesRequestElement();
                allSviceRsqtElement.setSessionId(mSessionId);
                _EcogridGetAllRegServicesResponseElement getAllRspnsElement = mRegistry
                        .getAllRegServices(allSviceRsqtElement);
                if (getAllRspnsElement != null)
                {
                    mRegEntries = getAllRspnsElement.getGetAllServicesReturn().getRegEntry();
                } else
                {
                    mRegEntries = null;
                }
                mStatus = PROCESS_OK;
            } else
            {
                mStatus = PROCESS_ERROR;
            }
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            mStatus = PROCESS_ERROR;
        }
    }
    
    /**
     * 
     *
     */
    protected void doAdd()
    {
        try
        {
            if (mRegistry != null)
            {
                mStatusStr = mRegistry.addRegEntry(mSessionId, mRegEntry);
                mStatus = PROCESS_OK;
            } else
            {
                mStatus = PROCESS_ERROR;
            }
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            mStatus = PROCESS_ERROR;
        }
    }
    
    /**
     * 
     *
     */
    protected void doUpdate()
    {
        try
        {
            if (mRegistry != null)
            {
                mStatusStr = mRegistry.updateRegEntry(mSessionId, mRegEntry);
                mStatus = PROCESS_OK;
            } else
            {
                mStatus = PROCESS_ERROR;
            }
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            mStatus = PROCESS_ERROR;
        }
        
    }
    
    /**
     * 
     *
     */
    protected void doRemove()
    {
        try
        {
            if (mRegistry != null)
            {
                mStatusStr = mRegistry.removeRegEntry(mSessionId, mRegEntry.getServiceName());
                mStatus = PROCESS_OK;
            } else
            {
                mStatus = PROCESS_ERROR;
            }
        } catch (Exception e)
        {
            mStatusStr = getErrorMsgFromXML(e.toString());
            mStatus = PROCESS_ERROR;
        }
    }
    
    /**
     * 
     *
     */
    public void doWork()
    {
        switch (mWorkType)
        {
            case DO_NONE:
            break;
            case DO_LOGIN:
                doLogin();
                break;
                
            case DO_LOGOUT:
                doLogout();
                break;
                
            case DO_GET_ALL:
                doGetAllServices();
                break;
                
            case DO_ADD:
                doAdd();
                break;
                
            case DO_UPDATE:
                doUpdate();
                break;
                
            case DO_REMOVE:
                doRemove();
                break;
        }
    }
    
    /**
     * Helper function to exract the error msg from the XML that is returned.
     * @param aErrMsg XML Error Message
     * @return text string
     */
    private String getErrorMsgFromXML(String aErrMsg)
    {
      try
      {
          int inx = aErrMsg.indexOf("?>");
          if (inx > -1)
          {
              aErrMsg = aErrMsg.substring(inx+2, aErrMsg.length());
          }
          Document dom = EcogridUtils.convertXMLStr2DOM(aErrMsg);
          String errMsg = EcogridUtils.findNodeValue(dom, "error");
          if (errMsg != null) 
          {
              return errMsg;
          }
          
      } catch (Exception e)
      {
      }
      return "N/A";
    }
    
    //----------------------------------------------------------------
    //-- Runnable Interface
    //----------------------------------------------------------------
    public void start() 
    {
        if (mThread == null) 
        {
            mThread = new Thread(this);
            mThread.setPriority(Thread.MIN_PRIORITY);
            mThread.start();
        }
    }

    /**
     * 
     *
     */
    public synchronized void stop() 
    {
        if (mThread != null) 
        {
            mThread.interrupt();
        }
        mThread = null;
        notifyAll();
    }

    /**
     * 
     */
    public void run() 
    {
        Thread me = Thread.currentThread();

        if (!isReady())
        {
            mStatus = PROCESS_BUSY;
            try 
            {
                Thread.sleep(200);
            } catch (InterruptedException e) { }
            
            while (mThread == me && mStatus == PROCESS_BUSY) 
            {
                // do work here 
                doWork();
                if (mStatus == PROCESS_BUSY)
                {
                    try 
                    {
                        Thread.sleep(200);
                    } catch (InterruptedException e) { }
                }
            }
        }
        
        notifyListeners();
        
        mThread = null;
    }
    
}

