package edu.ucsb.nceas.metacat;

import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSException;

import javax.servlet.http.HttpServletRequest;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.Set;
import java.util.Collections;
import java.util.HashSet;
import java.util.Arrays;

/** Anything Metacat might use to authenticate a user -- currently username,
 *  password, and/or GSI credential */
public class AuthInfo implements Externalizable {
	private static final long serialVersionUID = 1; // never, ever change this

	private String username;
	private String password;
	private GSSContext gssContext;
	private boolean contextDiscarded = false;
	private String userDN;
	private String authenticated = AUTHENTICATED_UNKNOWN;
	private boolean local;

	/** Values for {@link #setAuthenticated(String)}. */
	public static final String AUTHENTICATED_PASSED = "passed",
		AUTHENTICATED_FAILED = "failed", AUTHENTICATED_UNKNOWN = "unknown";

	/** Used internally to validate values of {@link #setAuthenticated}. */
	private static final Set LEGAL_VALUES_AUTHENTICATED = Collections
		.unmodifiableSet(new HashSet(Arrays.asList(new String[]
			{ AUTHENTICATED_FAILED, AUTHENTICATED_PASSED, AUTHENTICATED_UNKNOWN })));

	public AuthInfo(String username, String password) {
		this.username = username;
		this.password = password;
	}

	public AuthInfo
		(String username, String password, GSSContext gssContext, String userDN,
		 HttpServletRequest request)
	{
		this(username, password);
		this.gssContext = gssContext;
		this.userDN = userDN;
		if (request != null) {
			// Is there a better way to do this?  This seems hackish, but
			// it seems to work.  Are there good attacks against it?
			local = "127.0.0.1".equals(request.getRemoteAddr());
		}
	}

	/** For serialization only (required by {@link Externalizable}).
	 *  @deprecated */
	public AuthInfo() { }

	/** The username provided by the user.  Likely to be null if
	 *  {@link #getGssContext() a GSI context} is present. */
	public String getUsername() { return username; }

	/** Is the password null or zero-length? */
	public boolean isPasswordBlank() {
		return password == null || password.length() == 0;
	}

	public void setUsername(String username) { this.username = username; }

	/** Has this auth info already been authenticated?
	 *  Initially {@link #AUTHENTICATED_UNKNOWN}.
	 *  @return either {@link #AUTHENTICATED_FAILED}, {@link #AUTHENTICATED_PASSED},
	 *  or {@link #AUTHENTICATED_UNKNOWN}. */
	public String getAuthenticated() { return authenticated; }

	/** Has this auth info already been authenticated?
	 *  Initially {@link #AUTHENTICATED_UNKNOWN}.
	 *  @param authenticated either {@link #AUTHENTICATED_FAILED},\
	 *  {@link #AUTHENTICATED_PASSED}, or {@link #AUTHENTICATED_UNKNOWN} */
	public void setAuthenticated(String authenticated) {
		if (!LEGAL_VALUES_AUTHENTICATED.contains(authenticated))
			throw new IllegalArgumentException
				("Illegal parameter value: \"" + authenticated + "\".");
		this.authenticated = authenticated;
	}

	/** Shortcut version of {@link #setAuthenticated(String)}. */
	public void setAuthenticated(boolean authenticated) {
		this.authenticated = authenticated
			? AUTHENTICATED_PASSED : AUTHENTICATED_FAILED;
	}

	/** Shortcut to check whether {@link #getAuthenticated()} equals
	 *  {@link #AUTHENTICATED_PASSED}. */
	public boolean isAuthenticated() {
		return AUTHENTICATED_PASSED.equals(authenticated);
	}

	/** Is the authenticating party local?  That is, did the request that is
	 *  being authenticated originate on the local machine?  Determinded by
	 *  checking whether {@link HttpServletRequest#getRemoteAddr()} is local
	 *  (127.0.0.1). */
	public boolean isLocal() { return local; }

	/** The password provided by the user.  Likely to be null if
	 *  {@link #getGssContext() a GSI context} is present. */
	public String getPassword() { return password; }

	/** The Distinguished Name of the user who is requesting authentication
	 *  as represented in {@link #getGssContext()}.
	 *  May be available even if {@link #getGssContext()} is null and
	 *  {@link #isContextDiscarded()} is true. */
	public void setUserDN(String userDN) { this.userDN = userDN; }

	/** The Distinguished Name of the user who is requesting authentication
	 *  as represented in {@link #getGssContext()}.
	 *  May be available even if {@link #getGssContext()} is null and
	 *  {@link #isContextDiscarded()} is true. */
	public String getUserDN() { return userDN; }

	/** An alternative to username/password: PKI-based grid security. */
	public GSSContext getGssContext() { return gssContext; }

	/** The credential may be null now even though it was originally present,
	 *  if this object has been serialized -- it may have been discarded during
	 *  serialization, since {@link GSSContext} implementations are not
	 *  generally serializable. */
	public boolean isContextDiscarded() { return contextDiscarded; }

	public String toString() { return toString(false); }

	public String toString(boolean verbose) {
		String result = "";
		if (username != null)
			result += username;
		if (userDN != null) {
			if (verbose || result.length() == 0)
				result += (result.length() == 0 ? "" : " ")
					+ "[GSI DN: \"" + userDN + "\"; context "
					+ (gssContext == null ? "absent" : "present") + "]";
		}
		else if (gssContext != null) {
			if (verbose || result.length() == 0)
				result += (result.length() == 0 ? "" : " ")
					+ describeGssContext();
		}
		if (local) result += " (local)";
		return result;
	}

	private boolean firstDescribeEx = true;
	private String describeGssContext() {
		if (gssContext == null) return "null";
		else {
			try {
				return gssContext.getSrcName().toString();
			} catch (GSSException e) {
				if (firstDescribeEx) e.printStackTrace();
				firstDescribeEx = false;
				return "[error: " + e.getMessage() + "]";
			}
		}
	}

	public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
		in.readInt(); // version
		if (in.readBoolean()) username = in.readUTF();
		if (in.readBoolean()) password = in.readUTF();
		if (in.readBoolean()) userDN = in.readUTF();
		authenticated = in.readUTF(); // not allowed to be null
		gssContext = (GSSContext) in.readObject();
		contextDiscarded = in.readBoolean();
	}

	public void writeExternal(ObjectOutput out) throws IOException {
		out.writeInt(1);
		out.writeBoolean(username != null);
		if (username != null) out.writeUTF(username);
		out.writeBoolean(password != null);
		if (password != null) out.writeUTF(password);
		out.writeBoolean(userDN != null);
		if (userDN != null) out.writeUTF(userDN);
		out.writeUTF(authenticated); // not alowed to be null
		// can't generally serialize a credential, but we'll try
		try {
			out.writeObject(gssContext);
			out.writeBoolean(false); // credential not discarded
		} catch(Exception e) {
			out.writeObject(null);
			out.writeBoolean(true); // credential discarded
		}
	}
}
