package edu.ucsb.nceas.metacat;

import edu.ucsb.nceas.utilities.Options;

import java.io.*;
import java.util.Map;
import java.util.HashMap;
import java.util.Date;

/** Maps users' GSI Distinguished Names (DN) to usernames in a different
 *  authentication realm, such as LDAP.  Uses a properties file in WEB-INF
 *  whose keys are GSI DNs and whose values are usernames in the other
 *  realm, except that instead of "=" to separate the keys and values, use ":".
 *
 *  <p>Unfamiliar DNs: if a DN is unfamiliar, a default value can be specified
 *  in the map file.  Also, by default, any unfamiliar names will be written
 *  back to the file, with their foreign usernames set to the default value.</p>
 *
 *  <p>Example:</p>
 *
 *  <pre>
 *  default:uid=anonymous,o=nceas,dc=ecoinformatics,dc=org
 *  /C=US/O=National Center for Ecological Analysis and Synthesis/CN=Jan Kim:uid=jhkim,o=nceas,dc=ecoinformatics,dc=org
 *  /C=US/O=University of California at Santa Barbara/CN=Gerald Ford:uid=gf,o=ucsb,dc=edu
 *  </pre> */
public class GsiMapfile implements GsiToUsernameMap {
	private static final String FILENAME = "gsimap.properties";

	public static final String DEFAULT_KEY = "default", DEFAULT_VALUE = "public";

	private Map map = new HashMap();

	/** If true, attempt to write unfamiliar names back into the file. */
	private boolean writeUnknowns = true;

	/** The lastModified time of our file the last time we read it. */
	private long lastRead = Long.MIN_VALUE;

	public GsiMapfile() {}

	/** Reload from disk if we're out of date.  This isn't really necessary,
	 *  since Metacat will create a new instance of this class for each session,
	 *  but hey, maybe someday in the future it will matter. */
	private void reloadIfNecessary() throws IOException {
		File file = getFile();
		long lastMod = file.lastModified();
		if (lastMod > lastRead) {
			MetaCatUtil.debugMessage("GSI Map File = \"" + file.getCanonicalPath()
				+ "\" (last mod " + new Date(lastMod)
				+ "; last read " + new Date(lastRead) + ")", 40);
			// note: assign this before we actually read, so that if it is
			// updated while we read, at least it will appear to be out of date
			// at the next check
			lastRead = lastMod;
			FileReader in = null;
			try {
				if (!file.exists())
					if (!file.createNewFile()) throw new IOException
						("Unable to create new file at \"" + file.getCanonicalPath() + "\".");
				in = new FileReader(file);
				LineNumberReader reader = new LineNumberReader(in);
				for (String s = reader.readLine(); s != null; s = reader.readLine()) {
					s = s.trim();
					if (s.length() == 0) continue;
					if (s.charAt(0) == '#') continue; // skip comments
					int colon = s.indexOf(":");
					if (colon < 0)
						MetaCatUtil.debugMessage("Can't parse line #"
							+ reader.getLineNumber() + " from " + getFile().getCanonicalPath()
							+ " -- missing colon (\":\"): \"" + s + "\".", 20);
					else
						map.put(s.substring(0, colon), s.substring(colon + 1));
				}
			} finally {
				//noinspection EmptyCatchBlock
				try { if (in != null) in.close(); }
				catch(Exception e) {}
			}
		}
	}

	private static final Object FILE_SYNC = new Object();

	private static File getFile() {
		File metacatConfigFile = Options.getInstance().getPropertyFile();
		return new File(metacatConfigFile.getParent(), FILENAME);
	}

	public String getUsername(String gssDN) {
		try {
			reloadIfNecessary();
		} catch(Exception e) {
			String msg = "Exception loading GSI map file: " + e.getMessage();
			if (MetaCatUtil.debugMessage(msg, 20))
				e.printStackTrace();
		}
		String result = (String) map.get(gssDN);
		if (result == null) {
			result = (String) map.get(DEFAULT_KEY);
			if (result == null) result = DEFAULT_VALUE;
			// if the GSS DN is already the default value, there's probably a
			// short circuit going on somewhere -- here's a circuit breaker
			if (result.equals(gssDN)) result = null;
			map.put(gssDN, result);
			if (writeUnknowns) {
				synchronized(FILE_SYNC) {
					try {
						FileWriter writer = new FileWriter(getFile(), true);
						writer.write(System.getProperty("line.separator") + gssDN + ":" + result);
						writer.close();
					} catch (IOException e) {
						String msg = "Unable to write to "
							+ getFile().getPath() + ":" + e.getMessage();
						if (MetaCatUtil.debugMessage(msg, 20))
							e.printStackTrace();
					}
				}
			}
		}
//		new Exception("=-= Mapping \"" + gssDN + "\" to \"" + result).printStackTrace();
		return result;
	}

	/** Write unfamiliar DNs back into the file as the default value?  Default true. */
	public boolean getWriteUnknowns() { return writeUnknowns; }

	/** Write unfamiliar DNs back into the file default value?  Default true. */
	public void setWriteUnknowns(boolean writeUnknowns) {
		this.writeUnknowns = writeUnknowns;
	}
}
