/**
 *  '$RCSfile: EcogridUtils.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: ruland $'
 *     '$Date: 2005-12-15 21:18:44 $'
 * '$Revision: 1.17 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.ecoinformatics.ecogrid;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.StringReader;
import java.util.Calendar;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * Utility Class with Helper method for creating a DOM from a file or a string;
 *   debug output; 
 */
public class EcogridUtils
{

    private static boolean _printAll = false; // indicates whether compareDOMS should print all the node info

    private static boolean _debugOn = false;
    private static int     _level   = 100;

    public static void    setDebug(boolean aOn) { _debugOn = aOn; }
    public static void    setLevel(int aLevel)  { _level   = aLevel; }
    public static boolean isDebugOn()           { return _debugOn; }

    
   /**
     * Utility method to print debugging messages. User can set debug level
     * for this message. The number is fewer, the message is more important
     * @param msg, the content of the message
     * @param _level, an integer indicating the message debug leve
     */
    public static void debugMessage(String aMsg, int aLevel)
    {
      if (_debugOn)
      {
        //don't allow the user set debugLevel less than or equals 0
        if (aLevel < 0)
        {
          aLevel = 0;
        }

        if (aLevel <= _level)
        {
          System.err.println("EcogridUtils: " +aMsg);
        }
      }
    } // debugMessage

    /**
      * Returns the String representation of the Node Type
      * @param aMsg message to be output
      */
    public static void debugMessage(String aMsg)
    {
      debugMessage(aMsg, 1);
    }

    /**
      * Returns the String Representation of the Node Type
      * @param aPrintAll indicats whether compareDOMs should print all the DOM Node info
      */
	  public static void setPrintAll(boolean aPrintAll)
    {
      _printAll = aPrintAll;
    }

    /**
      * Returns the String representation of the Node Type
      * @param aType Node type to get the string description of
      */
	  public static String getNodeTypeStr(int aType)
    {
      String str = "N/A";
      switch (aType) 
      {
        case Node.ATTRIBUTE_NODE:        str = "The node is an Attr."; break;
        case Node.CDATA_SECTION_NODE:    str = "The node is a CDATASection."; break;
        case Node.COMMENT_NODE:          str = "The node is a Comment."; break;
        case Node.DOCUMENT_FRAGMENT_NODE: str = "The node is a DocumentFragment."; break;
        case Node.DOCUMENT_NODE:         str = "The node is a Document."; break;
        case Node.DOCUMENT_TYPE_NODE:    str = "The node is a DocumentType."; break;
        case Node.ELEMENT_NODE:          str = "The node is an Element."; break;
        case Node.ENTITY_NODE:           str = "The node is an Entity."; break;
        case Node.ENTITY_REFERENCE_NODE: str = "The node is an EntityReference."; break;
        case Node.NOTATION_NODE:         str = "The node is a Notation."; break;
        case Node.PROCESSING_INSTRUCTION_NODE: str = "The node is a ProcessingInstruction."; break;
        case Node.TEXT_NODE:             str = "The node is a Text node."; break; 
      }
      return str;
    }

    /** ------------------------------------------------------------
     * Gets the String value of a node. First checks it's value and if
     * that is null then it checks to see if it has a child node and gets
     * the value of the first child.
     * Assumption: That the first child is a #text node, 
     *             delibertly NOT checking the first node's type
     * @param aNode Parent to search (should be the document root)
     * @return Returns the value of the node
     * -------------------------------------------------------------- */
    public static String getNodeValue(Node aNode)
    {
      String value = null;
      if (aNode.getNodeValue() != null)
      {
        value = aNode.getNodeValue() != null ? aNode.getNodeValue().trim() : null;
      }
      else 
      {
        NodeList list = aNode.getChildNodes();
        if (list.getLength() == 1)
        {
          Node child = list.item(0);
          if (child != null) 
          {
             value = child.getNodeValue() != null ? child.getNodeValue().trim() : null;
          }
        } else {
          //System.out.println("For Node "+aNode.getNodeName()+" Num Child: "+list.getLength());
        }
      }
      return value;
    }

    /** ------------------------------------------------------------
     * Finds the first node of a given type
     * @param aNode Parent to search (should be the document root)
     * @param aName Name of node to find
     * @return Returns the node of that name or null
     * -------------------------------------------------------------- */
    public static Node findNode(Node aNode, String aName)
    {
        String name = aNode.getNodeName() != null ? aNode.getNodeName().trim() : "";
        if (aName.equalsIgnoreCase(name)) 
        {
            return aNode;
        }

        NodeList list = aNode.getChildNodes();
        if (list != null)
        {
            for (int i = 0; i < list.getLength(); i++)
            {
                Node child = list.item(i);
                if (child != null)
                {
                    Node node = findNode(child, aName);
                    if (node != null)
                    {
                        return node;
                    }
                }
            }
        }
      return null;
    }
    
    /**
     * Find a node in the vector of node names 
     * @param aNode the current parent node
     * @param aNodeNames a list of node names to be search
     * @param aDepth the current index of name in the list to be searched for
     * @return the found node
     */
    protected static Node findNodeWithXPath(Node aNode, Vector aNodeNames, int aDepth)
    {
        if (aDepth == aNodeNames.size())
        {
            return null;
        }
        String pathNodeName = (String)aNodeNames.elementAt(aDepth);
        NodeList list = aNode.getChildNodes();
        if (list != null)
        {
            for (int i=0;i<list.getLength();i++) 
            {
                Node child = list.item(i);
                if (child != null)
                {
                    String nodeName = child.getNodeName();
                    if (pathNodeName.equals(nodeName))
                    {
                        if (aDepth == aNodeNames.size()-1)
                        {
                            return child;
                        }
                        Node node = findNodeWithXPath(child, aNodeNames, aDepth+1);
                        if (node != null) 
                        {
                          return node;
                        }
                    }
                }
            }
        }
        return null;
    }
    
    /**
     * Find a single node in a DOM tree with an XPath string. 
     * If the path starts with a '/' then the first name MUST match the 
     * the root name of the DOM, if not then the first name is the name of 
     * a child node.
     * @param aRootNode the root node of the search
     * @param aXPath the XPath string of names to be searched
     * @return the node
     */
    public static Node findNodeWithXPath(Node aRootNode, String aXPath)
    {
        Vector nodeNames = new Vector();
        StringTokenizer st = new StringTokenizer(aXPath, "/");
        while (st.hasMoreTokens()) {
            nodeNames.addElement(st.nextToken());
        }
        
        int depth = 0;
        /*if (aXPath.charAt(0) == '/')
        {
            String nodeName = aRootNode.getNodeName() != null ? aRootNode.getNodeName().trim() : "";
            String pathNodeName = (String)nodeNames.elementAt(0);
            if (!nodeName.equals(pathNodeName)) 
            {
              return null;
            }
            depth = 1;
        }*/
        
        return findNodeWithXPath(aRootNode, nodeNames, depth);
    }
   
    /** ------------------------------------------------------------
     * Searches for the Node by the param "aName" and returns its value
     * @param aNode Parent to search 
     * @param aName Name of node to find
     * @return Returns the node's value as a string
     * -------------------------------------------------------------- */
    public static String findNodeValue(Node aNode, String aName)
    {
      String value = null;
      Node node = findNode(aNode, aName);
      if (node != null)
      {
        value = getNodeValue(node);
      }
      return value;
    }

    /** ------------------------------------------------------------
     * Gets an attribute value for the named node.
     * @param aNode Parent to search (should be the document root)
     * @param aName Name of node to find
     * @param aAttr Name of attribute to return
     * @return Returns the attribute's value as a string
     * -------------------------------------------------------------- */
    public static String findAttrValueForNode(Node aNode, String aName, String aAttr)
    {
      String value = null;
      Node node = findNode(aNode, aName);
      if (node != null)
      {
        NamedNodeMap map = node.getAttributes();
        if (map != null)
        {
          Node attrNode = map.getNamedItem(aAttr);
          if (attrNode != null)
          {
            value = getNodeValue(attrNode);
          }
        }
      }
      return value;
    }

    /** ------------------------------------------------------------
     * Gets an attribute value for the node.
     * @param aNode Parent to search (should be the document root)
     * @param aAttr Name of attribute to return
     * @return Returns the attribute's value as a string
     * -------------------------------------------------------------- */
    public static String findAttrValue(Node aNode, String aAttr)
    {
      String value = null;
      if (aNode != null)
      {
        NamedNodeMap map = aNode.getAttributes();
        if (map != null)
        {
          Node attrNode = map.getNamedItem(aAttr);
          if (attrNode != null)
          {
            value = getNodeValue(attrNode);
          }
        }
      }
      return value;
    }

    /**
      * Prints a DOM Tree (recursive)
      * @param aNode parent node of tree to be printed
      * @param aLevel indicates the current indentation level
      */
	  public static void printNode(Node aNode, int aLevel)
    {
      if (aNode == null) {
        return;
      }

      String spaces = "";
      for (int i=0;i<aLevel;i++) 
      {
        spaces += "..";
      }

      System.out.println(spaces+"Name:  " + aNode.getNodeName());
      System.out.println(spaces+"Type:  " + aNode.getNodeType());
      System.out.println(spaces+"Value: [" + aNode.getNodeValue()+"]");
      NodeList list = aNode.getChildNodes();
      if (list != null) {
        for (int i=0;i<list.getLength();i++) 
        {
          Node child = list.item(i);
          printNode(child, aLevel+1);
        }
      }
    }

    /**
      * Reads in an XML document and returns a String of the file's contents
      * @param aFileName file name of XML file to be read
      */
	  public static String readXMLFile2Str(String aFileName)
    {
      try 
      {
        String buffer = "";
        FileReader     fileReader = new FileReader(aFileName);
        BufferedReader bufReader  = new BufferedReader(fileReader);
        String line = bufReader.readLine();
        while (line != null) {
          buffer += line + "\n";
          line = bufReader.readLine();
        }
        return buffer;
      } // try
      catch (Exception e) 
      {
        System.err.println("readXMLFile2Str - Exception: "+ e);
      }
      return null;
    }

    /**
      * Reads in an XML document and returns the Document node of the DOM tree
      * @param aFileName file name of XML file to be read
      */
	  public static Document readXMLFile2DOM(String aFileName)
    {
        if (aFileName == null)
        {
            System.err.println("*** EcogridUtils readXMLFile2DOM file name is null!");
            return null;
        }
        
        if (aFileName.length() == 0)
        {
            System.err.println("*** EcogridUtils readXMLFile2DOM file name is null!");
            return null;
        }
        
        File file = new File(aFileName);
        if (file == null || !file.exists() || file.length() == 0)
        {
            System.err.println("*** EcogridUtils readXMLFile2DOM - file "+aFileName+" does not exist or is zero length!");
            return null;
        }

        FileInputStream fis;
        try {
           fis = new FileInputStream( file );
        }
        catch( FileNotFoundException e ) {
            System.err.println("*** EcogridUtils readXMLFile2DOM - file "+aFileName+" does not exist!");
            return null;
        }
        return convertXMLStr2DOM(new InputSource(fis));
    }

    /**
      * Convert/Parses an XML string into a DOM tree
      * @param aXMLStr XML string (document)
      */
   public static Document convertXMLStr2DOM(InputSource aXMLStream)
    {
      try 
      {
         TransformerFactory tFactory = TransformerFactory.newInstance();

        if (tFactory.getFeature(DOMSource.FEATURE) && tFactory.getFeature(DOMResult.FEATURE))
        {
          //Instantiate a DocumentBuilderFactory.
          DocumentBuilderFactory dFactory = DocumentBuilderFactory.newInstance();

          // And setNamespaceAware, which is required when parsing xsl files
          dFactory.setNamespaceAware(true);

          //Use the DocumentBuilderFactory to create a DocumentBuilder.
          DocumentBuilder dBuilder = dFactory.newDocumentBuilder();

          //Use the DocumentBuilder to parse the XML input.
          Document xmlDoc = dBuilder.parse(aXMLStream);

          return xmlDoc;
  
	      }
      } // try
      catch (Exception e) {
        System.err.println("convertXMLStr2DOM - Exception: "+ e);
        e.printStackTrace();
      }
      return null;
    }
    
    /**
      * Convert/Parses an XML string into a DOM tree
      * @param aXMLStr XML string (document)
      */
   public static Document convertXMLStr2DOM(String aXMLStr)
    {
       
       StringReader strReader = new StringReader(aXMLStr);
       InputSource inpSrc = new InputSource(strReader);

       return convertXMLStr2DOM( inpSrc );

    }
    
    /**
      * Writes an XML DOM to a file
      * @param aFileName The file name to be written
      * @param aDOM THe DOM to be serialized to a file.
      */
	  public static boolean writeXMLToFile(String aFileName, Document aDOM)
    {/*
      try
      {
        File file = new File(aFileName);
        if (file != null)
        {
          file.createNewFile();
          FileOutputStream fileOut = new FileOutputStream(file);
          if (file != null)

          {
            Serializer serializer = SerializerFactory.getSerializer
                                   (OutputPropertiesFactory.getDefaultMethodProperties("xml"));
            //serializer.setOutputStream(fileOut);
            serializer.setOutputStream(System.out);
            //System.out.println(aDOM.getNode());
            serializer.asDOMSerializer().serialize(aDOM);
            return true;
          }
        }
      }
      catch (Exception e)
      {
        System.err.println("writeXMLToFile - " + e);
      }*/
      return false;
    }
    
    /**
      * Compares two DOM Trees
      * @param aNode1 DOM Tree to be compared
      * @param aNode2 Second DOM tree to be comapred
      * @param aLevel indentation level when printed
      * @param aPrintAll indicates whether it should print all the node info
      */
	  public static boolean compareDOMs(Node aNode1, Node aNode2, int aLevel, boolean aPrintAll) 
    {
      String spaces = "";
      if (aPrintAll)
      {
        for (int i=0;i<aLevel;i++) 
          spaces += "..";
      }

      boolean isOK = true;

      NodeList list1 = aNode1.getChildNodes();
      NodeList list2 = aNode2.getChildNodes();

      String name1 = aNode1.getNodeName() != null ? aNode1.getNodeName().trim() : "";
      String name2 = aNode2.getNodeName() != null ? aNode2.getNodeName().trim() : "";
      if (!name1.equalsIgnoreCase(name2)) 
      {
        System.out.println(spaces+"NodeName not equal ["+name1+"]["+name2+"]");
        isOK = false;
      } 
      else if (aPrintAll) 
      {
        System.out.println(spaces+"NodeName:  " + name1);
      }

      if (aNode1.getNodeType() != aNode2.getNodeType()) {
        System.out.println(spaces+"NodeType not equal ["+getNodeTypeStr(aNode1.getNodeType())+"]["+getNodeTypeStr(aNode2.getNodeType())+"]");
        isOK = false;
      } 
      else if (aPrintAll) 
      {
        System.out.println(spaces+"NodeType:  " + getNodeTypeStr(aNode1.getNodeType()));
      }

      String value1 = aNode1.getNodeValue() != null ? aNode1.getNodeValue().trim() : "";
      String value2 = aNode2.getNodeValue() != null ? aNode2.getNodeValue().trim() : "";
      String skip = "--skip--";
      if (!value1.equalsIgnoreCase(value2) && !value1.equalsIgnoreCase(skip) && !value2.equalsIgnoreCase(skip)) {
        System.out.println(spaces+"NodeValue not equal ["+value1+"]["+value2+"]");
        isOK = false;
      } 
      else if (aPrintAll) 
      {
        System.out.println(spaces+"NodeValue:  " + value1);
      }

      String uri1 = aNode1.getNamespaceURI() != null ? aNode1.getNamespaceURI().trim() : "";
      String uri2 = aNode2.getNamespaceURI() != null ? aNode2.getNamespaceURI().trim() : "";
      if (!uri1.equalsIgnoreCase(uri2)) {
        System.out.println(spaces+"NamespaceURI not equal ["+uri1+"]["+uri2+"]");
        isOK = false;
      } 
      else if (aPrintAll) 
      {
        System.out.println(spaces+"NamespaceURI:  " + uri1);
      }


      if (list1.getLength() != list2.getLength()) 
      {
        System.out.println(spaces+"Num Children not equal ["+list1.getLength()+"]["+list2.getLength()+"]");
        System.out.println(spaces+"-----------------------------");
        System.out.println(spaces+"Node1:");
        for (int i=0;i<list1.getLength();i++) 
        {
          Node child = list1.item(i);
          String value = child.getNodeValue() != null ? child.getNodeValue().trim() : "";
          System.out.println(spaces+"..");
          System.out.println(spaces+"..Name:  " + child.getNodeName());
          System.out.println(spaces+"..Type:  " + child.getNodeType());
          System.out.println(spaces+"..Value: [" + value + "]");
        }

        System.out.println(spaces+"-----------------------------");
        System.out.println(spaces+"Node2:");
        for (int i=0;i<list2.getLength();i++) 
        {
          Node child = list2.item(i);
          String value = child.getNodeValue() != null ? child.getNodeValue().trim() : "";
          System.out.println(spaces+"..");
          System.out.println(spaces+"..Name:  " + child.getNodeName());
          System.out.println(spaces+"..Type:  " + child.getNodeType());
          System.out.println(spaces+"..Value: [" + value + "]");
        }
        System.out.println(spaces+"-----------------------------");

        isOK = false;
      } 
      else if (aPrintAll) 
      {
        System.out.println(spaces+"Num Children: " + list1.getLength());
      }

      if (!isOK) 
        return false;

      for (int i=0;i<list1.getLength();i++) 
      {
        Node child1 = list1.item(i);
        Node child2 = list2.item(i);
        if (!compareDOMs(child1, child2, aLevel+1, _printAll)) 
        {
          return false;
        }
      }
      return true;
    }

    /**
      * Compares to XML File's DOMs
      * @param aDOM1Filename Name of XML file representing a XML document to be compared
      * @param aDOM2Filename Name of second XML file representing a XML document to be compared
      */
	  public static boolean compareDOMs(String aDOM1Filename, String aDOM2Filename)
    {
      Document dom1 = readXMLFile2DOM(aDOM1Filename);
      if (dom1 == null) 
      {
        System.out.println("Error - Couldn't find xml file: " + aDOM1Filename);
        return false;
      }

      Document dom2 = readXMLFile2DOM(aDOM2Filename);
      if (dom2 == null) 
      {
        System.out.println("Error - Couldn't find xml file: " + aDOM2Filename);
        return false;
      }

      return compareDOMs((Node)dom1, (Node)dom2, 0, _printAll);
    }


    /**
      * Escape String for HTML/XML (code borrowed from DiGIR
      * @param s string to be escaped
      * @return escaped string
      */
    public static final String escapeHTML(String s)
    {
      if (s == null || s.length() == 0) {
        return "";
      }

      StringBuffer sb = new StringBuffer();
      int n = s.length();
      for (int i = 0; i < n; i++)
      {
        char c = s.charAt(i);
        switch (c)
        {
          case '<': sb.append("&lt;"); break;
          case '>': sb.append("&gt;"); break;
          case '"': sb.append("&quot;"); break;
          case '&': sb.append("&amp;"); break;
        //all these should be uncommented so the escaping is complete
/*
          case '': sb.append("&agrave;");break;
          case '': sb.append("&Agrave;");break;
          case '': sb.append("&acirc;");break;
          case '': sb.append("&Acirc;");break;
          case '': sb.append("&auml;");break;
          case '': sb.append("&Auml;");break;
          case '': sb.append("&aring;");break;
          case '': sb.append("&Aring;");break;
          case '': sb.append("&aelig;");break;
          case '': sb.append("&AElig;");break;
          case '': sb.append("&ccedil;");break;
          case '': sb.append("&Ccedil;");break;
          case '': sb.append("&eacute;");break;
          case '': sb.append("&Eacute;");break;
          case '': sb.append("&egrave;");break;
          case '': sb.append("&Egrave;");break;
          case '': sb.append("&ecirc;");break;
          case '': sb.append("&Ecirc;");break;
          case '': sb.append("&euml;");break;
          case '': sb.append("&Euml;");break;
          case '': sb.append("&iuml;");break;
          case '': sb.append("&Iuml;");break;
          case '': sb.append("&ocirc;");break;
          case '': sb.append("&Ocirc;");break;
          case '': sb.append("&ouml;");break;
          case '': sb.append("&Ouml;");break;
          case '': sb.append("&oslash;");break;
          case '': sb.append("&Oslash;");break;
          case '': sb.append("&szlig;");break;
          case '': sb.append("&ugrave;");break;
          case '': sb.append("&Ugrave;");break;         
          case '': sb.append("&ucirc;");break;         
          case '': sb.append("&Ucirc;");break;
          case '': sb.append("&uuml;");break;
          case '': sb.append("&Uuml;");break;
          case '': sb.append("&reg;");break;         
          case '': sb.append("&copy;");break;   
          case '': sb.append("&euro;"); break;
          // be carefull with this one (non-breaking whitee space)
          case ' ': sb.append("&nbsp;");break;
          */         
          default:  sb.append(c); break;
        }
      }

      return sb.toString();
    }

   /**
    * Creates a Calerndar object from a string
    * @param aDateStr the date to be converted
    * @return the date
    */
   public static Calendar getCalendarDate(String aDateStr)
   {
     //TimeZone local = TimeZone.getTimeZone(System.getProperty("user.timezone"));
     Calendar sendDate = Calendar.getInstance();
     if (aDateStr != null && aDateStr.length() > 18)
     {
       try {
         int yr, mn, dy;
         if (aDateStr.charAt(2) == '-')
         {
           // 012345678901234567890
           // 29-01-2004 13:13:51-0600
           yr  = Integer.valueOf(aDateStr.substring(6,10)).intValue();
           mn  = Integer.valueOf(aDateStr.substring(3,5)).intValue();
           dy  = Integer.valueOf(aDateStr.substring(0,2)).intValue();
         } else
         {
           // 012345678901234567890
           // 2004-01-29 13:13:51-0600
           yr  = Integer.valueOf(aDateStr.substring(0,4)).intValue();
           mn  = Integer.valueOf(aDateStr.substring(5,7)).intValue();
           dy  = Integer.valueOf(aDateStr.substring(8,10)).intValue();
         }
         int hr  = Integer.valueOf(aDateStr.substring(11,13)).intValue();
         int min = Integer.valueOf(aDateStr.substring(14,16)).intValue();
         int sec = Integer.valueOf(aDateStr.substring(17,19)).intValue();
         sendDate.set(yr, mn-1, dy, hr, min, sec);
         EcogridUtils.debugMessage("********* "+yr+"/"+mn+"/"+dy+" "+hr+":"+min+":"+sec);

       }
       catch (Exception e)
       {
         sendDate.setTime(new Date());
       }
     }
     else
     {
       sendDate.setTime(new Date());
     }
     return sendDate;
   }
   
   /**
    * Parses for node name and namespace
    * @param aStr the string to be parsed usually a DOM node name
    * @param aNodeName the name of the node
    * @param aNameSpace the namespace for the node
    */
   public static void parseForNameAndNamespace(String       aStr, 
                                               StringBuffer aNodeName,
                                               StringBuffer aNameSpace)
    {
        aNodeName.setLength(0);
        aNameSpace.setLength(0);
        if (aStr != null)
        {
            int inx = aStr.indexOf(':');
            if (inx != -1)
            {
                aNodeName.append(aStr.substring(inx + 1, aStr.length()));
                aNameSpace.append(aStr.substring(0, inx));
            } else 
            {
                aNodeName.append(aStr);
            }
        }
    }

    // check env to see if it is a factory service. Default value is 'true'.
    public static boolean isaFcatoryServiceByEnv()
    {
        String str = System.getProperty("FACTORY");
        if(str == null)
          return true;

        if(str.compareToIgnoreCase("no") == 0)
          return false;
    
        return true;
    }

    public static boolean printStackTraceByEnv()
    {
        String str = System.getProperty("PRINT_STACK_TRACE");
        if(str == null)
          return false;

        if(str.compareToIgnoreCase("yes") == 0)
          return true;
    
        return false;
    }
}

