package org.ecoinformatics.ecogrid.digir.impl.digirquery;

import java.util.Iterator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.apache.axis.types.URI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.ecoinformatics.ecogrid.queryservice.util.EcogridResultsetFactory;
import org.ecoinformatics.ecogrid.queryservice.query.QueryType;
import org.ecoinformatics.ecogrid.queryservice.query.QueryTypeNamespace;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetType;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadata;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataNamespace;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataSystem;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataRecordStructure;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeResultsetMetadataRecordStructureReturnField;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeRecord;
import org.ecoinformatics.ecogrid.queryservice.resultset.ResultsetTypeRecordReturnField;

//
// The constructor, initialized, and extractResultset are not thread safe.
public class DigirResultsetTransformer
{

   // Mapping of EcogridQuery returnField Names to Field Ids.
   private Map ecogridFields;
   // Set representation of above.
   private Set ecogridFieldsSet;

   // Mapping from EcogridQuery returnField Names to DarwinCore Names.
   private Map ecogridFieldsToDarwinCore;

   private Map systemMap;
   private Set recordIdSet;
   private Vector records;

   private String namespace;
   private String namespacePrefix;

   private int systemCount;
   private int recordCount;

   static Log logger = LogFactory.getLog(DigirResultsetTransformer.class.getName());

   public DigirResultsetTransformer() {
      ecogridFields = new HashMap();
      ecogridFieldsToDarwinCore = new HashMap();
      systemMap = new HashMap();
      recordIdSet = new HashSet();
      records = new Vector();
      systemCount = 0;
      recordCount = 0;
   }

   public void initializeFieldnames(QueryType ecogridQuery) {
      String [] returnFieldList = ecogridQuery.getReturnField();

      for(int i=0; i<returnFieldList.length;i++) {
         String field = returnFieldList[i];
         ecogridFields.put( field, "f" + i );

         String digirField = field;
         if ( digirField.charAt(0) == '/' ) {
            digirField = digirField.substring(1);
         }
         ecogridFieldsToDarwinCore.put( field, digirField );

         if ( logger.isDebugEnabled() ) {
            logger.debug("Adding field mapping: Ecogrid = " + field  + " DarwinCore = " + digirField + " Id = f" + i);
         }
      }

      ecogridFieldsSet = ecogridFields.entrySet();

      QueryTypeNamespace qns = ecogridQuery.getNamespace();
      namespacePrefix = qns.getPrefix();
      namespace = qns.get_value().toString();
      
   }

   synchronized public String addSystem( String system ) {

      String systemId;
      Object o = systemMap.get( system );
      if ( o == null ) {
         systemId = "s" + Integer.toString( systemCount );
         systemCount++;
         systemMap.put( system, systemId );
      } else {
         systemId = (String) o;
      }
      return systemId;
   }

   synchronized private void addRecordId( String recordId ) {
      if ( recordIdSet.contains( recordId ) ) {
         logger.error( "Resultset already contains record Id: " + recordId );
         return;
      }
      recordIdSet.add( recordId );
   }

   synchronized public void addRecord( String systemId, String recordId, Map recordValues ) {

      addRecordId( recordId );

      ResultsetTypeRecord record = new ResultsetTypeRecord();
      
      // preincrement recordCount because the ecogrid resultset is numbered from 1..n
      record.setNumber( ++recordCount );
      record.setSystem( systemId );
      record.setIdentifier( recordId );

      ResultsetTypeRecordReturnField[] returnFieldList = new ResultsetTypeRecordReturnField[ ecogridFieldsSet.size() ];

      //
      // Note:
      // These convolutions are necessary because the Kepler client
      // requires the metadata return fields to be
      // in the same order as the record's return fields.
      //
      Iterator setI = ecogridFieldsSet.iterator();
      for( int i = 0; i< ecogridFieldsSet.size(); i++ ) {

         Map.Entry e = (Map.Entry) setI.next();

         String ecofieldName = (String) e.getKey();
         String fieldId = (String) e.getValue();
         String digirFieldName = (String) ecogridFieldsToDarwinCore.get( ecofieldName );

         returnFieldList[i] = new ResultsetTypeRecordReturnField("");
         returnFieldList[i].setId( fieldId );

         // If there is no corresponding Digir Field set empty.
         if( digirFieldName == null ) {
            continue;
         }

         String digirValue = (String) recordValues.get( digirFieldName );

         if ( digirValue == null ) {
            continue;
         }

         returnFieldList[i].set_value( digirValue );

      }

      record.setReturnField( returnFieldList );

      records.add( record );

   }

   public ResultsetType extractResultset() {

      ResultsetType resultset = new ResultsetType();

      //
      // Hacked values.  Copied from old code.
      try {
      resultset.setSystem( new URI("http://knb.ecoinformatics.org"));
      } catch (Exception e) // Really for MalformedURLException
         {}
      resultset.setResultsetId("digir.001");

      //
      // Base Metadata Structure.
      ResultsetTypeResultsetMetadata metadata = new ResultsetTypeResultsetMetadata();
      resultset.setResultsetMetadata( metadata );

      java.util.Calendar currenttime = java.util.Calendar.getInstance();
      currenttime.setTime( new java.util.Date() );
      metadata.setSendTime(  currenttime );
      metadata.setStartRecord(1);
      metadata.setEndRecord(recordCount);
      metadata.setRecordCount(recordCount);

      //
      // Populate Metadata Structure with System Mappings.
      Iterator systemsIt = systemMap.entrySet().iterator();
      ResultsetTypeResultsetMetadataSystem systems[] = new ResultsetTypeResultsetMetadataSystem[ systemMap.size() ];
      for( int i=0; i< systemMap.size(); i++ ) {
         Map.Entry s = (Map.Entry) systemsIt.next();
         String systemName = (String) s.getKey();
         // Some resources have spaces in the names.  Converting these to %20 lets us put them in the system urls.
         systemName = systemName.replaceAll(" ", "%20");
         logger.debug("Adding system name = "+ systemName );
         systems[i] = EcogridResultsetFactory.createSystem( systemName, (String) s.getValue() );
      }
      metadata.setSystem( systems );

      //
      // Populate Metadata Structure with Field Mappings.
      ResultsetTypeResultsetMetadataRecordStructure struct = new ResultsetTypeResultsetMetadataRecordStructure();

      Iterator fieldsIt = ecogridFieldsSet.iterator();
      ResultsetTypeResultsetMetadataRecordStructureReturnField[] fields = new ResultsetTypeResultsetMetadataRecordStructureReturnField[ ecogridFieldsSet.size() ];
      for( int i=0; i< ecogridFieldsSet.size(); i++ ) {
         Map.Entry e = (Map.Entry) fieldsIt.next();
         fields[i] = EcogridResultsetFactory.createRecordStructReturnField( (String) e.getValue(),
                                                                            null,
                                                                            (String) e.getKey() );
      }
      struct.setReturnField( fields );
      
      metadata.setRecordStructure( struct );

      //
      // Populate Records

      ResultsetTypeRecord[] recordsArray = new ResultsetTypeRecord[0];
      recordsArray = (ResultsetTypeRecord[]) records.toArray( recordsArray );
      resultset.setRecord( recordsArray );

      return resultset;
   }
}
