 /**
 *  '$RCSfile: EcogridQueryParser.java,v $'
 *    Purpose: A Class that loads eml-access.xml file containing ACL
 *             for a metadata document into relational DB
 *  Copyright: 2000 Regents of the University of California and the
 *             National Center for Ecological Analysis and Synthesis
 *    Authors: Jivka Bojilova
 *    Release: @release@
 *
 *   '$Author: ruland $'
 *     '$Date: 2005-12-15 21:18:45 $'
 * '$Revision: 1.7 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.ecoinformatics.ecogrid.queryservice.util;

import org.apache.axis.types.URI;
import org.apache.xpath.XPathAPI;

import org.ecoinformatics.ecogrid.queryservice.query.*;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

import java.io.*;

import javax.xml.parsers.DocumentBuilderFactory;
/**
 * A class to parse ecogrid xml query document to java object
 */
public class EcogridQueryParser
{
   //fields
   private QueryType _ecogridQuery;
   private Node _queryNode;

   //Constant
   private static final String QUERYIDATTRIBUTE = "@queryId";
   private static final String SYSTEMATTRIBUTE  = "@system";
   private static final String NAMESPACE        = "namespace";
   private static final String PREFIX           = "prefix";
   private static final String RETURNFIELD      = "returnField";
   private static final String TITLE            = "title";
   private static final String FIRSTAND         = "AND";
   private static final String FIRSTOR          = "OR";
   private static final String FIRSTCONDITION   = "condition";
   private static final String AND              = "AND";
   private static final String OR               = "OR";
   private static final String CON              = "condition";
   private static final String OPERATOR         = "operator";
   private static final String CONCEPT          = "concept";
   
    /**
      * Reads in a EcoGrid Query and returns a Query object
      * @param aFileName Name of XML file representing a XML document to be compared
      */
    public static QueryType readEcogridXMLQuery(String aFileName) {

      QueryType query = null;
      try
      {
        File queryXMLFile = new File(aFileName);
        if (queryXMLFile != null)
        {
          FileReader xmlReader = new FileReader(queryXMLFile);
          if (xmlReader != null)
          {
            EcogridQueryParser parser = new EcogridQueryParser(xmlReader);
            if (parser != null)
            {
              parser.parseXML();
              query = parser.getEcogridQuery();
            } else
            {
                System.err.println("EcogridQueryParser was null.");
            }
          } else
          {
              System.err.println("Couldn't create xmlReader.");
          }
        } else 
        {
            System.err.println("Couldn't open file["+aFileName+"]");
        }

      } 
      catch (Exception e) 
      {
        System.err.println("readEcogridXMLQuery exception: "+e);
        e.printStackTrace();
      }

      return query;
    }

   /**
    * Constructor with reader for parsing XML
    * @param  xmlInput  xml source
    */
   public EcogridQueryParser (Reader xmlInput)  throws Exception
   {
      //create new ecogrid query java object
      _ecogridQuery = new QueryType();
      //build dom tree
      InputSource in = new InputSource(xmlInput);
      DocumentBuilderFactory dfactory = DocumentBuilderFactory.newInstance();
      dfactory.setNamespaceAware(false);
      Document doc = dfactory.newDocumentBuilder().parse(in);
      
      // Find the query node, all searches are relative to it.
      NodeList queryIdList = XPathAPI.selectNodeList(doc, "/query");
      _queryNode = queryIdList.item(0);

    }  //EcogridQueryParser

   /**
    * Constructor wityh DOM node point to Query node
    * @param  aQueryDOMNode  DOM Node
    */
   public EcogridQueryParser (Node aQueryDOMNode)
   {
      //create new ecogrid query java object
      _ecogridQuery = new QueryType();
      _queryNode    = aQueryDOMNode;
    }  //EcogridQueryParser

    /**
     * Method to get ecogridquery
     */
   public QueryType getEcogridQuery()
   {
     return _ecogridQuery;
   }

   /**
    * Method to set ecogrid query
    */
   public void setEcogridQuery(QueryType query)
   {
      _ecogridQuery = query;
   }

   /**
    *Method set up ecogrid query object by parsing xml
    */
   public void parseXML() throws Exception
   {
     setupQueryIdAndSystem();
     setupNamespace();
     setupReturnField();
     setupTitle();
     setupFirstRelation();
   }//parseXML

   private Node getAttrNode(String aDocXPath, String aNodeXPath)
   {
       Node node = _queryNode;
       try 
       {
           if (_queryNode.getNodeType() == Node.DOCUMENT_NODE)
           {
             NodeList queryIdList = XPathAPI.selectNodeList(_queryNode, aDocXPath);
             node = queryIdList.item(0);
           }
           else
           {
               NodeList queryIdList = XPathAPI.selectNodeList(_queryNode, aNodeXPath);
               node = queryIdList.item(0);
           }
       } catch (Exception e)
       {
           System.err.println(e);
       }
       return node;
   }
   /* setup queryid and system */
   private void setupQueryIdAndSystem() throws Exception
   {
     // The _queryNode may be a document node qparsed in from an XML document or may be
     // a generic DOM node (a subtree) of an existing tree
     // get query id
     NodeList queryIdList = XPathAPI.selectNodeList(_queryNode, QUERYIDATTRIBUTE);
     String queryId = ((Node)queryIdList.item(0)).getNodeValue();
     
     //System.out.println("queryId is: "+ queryId);
     // set query id
     _ecogridQuery.setQueryId(queryId);
     // get system
     NodeList systemList = XPathAPI.selectNodeList(_queryNode, SYSTEMATTRIBUTE);
     String system = (systemList.item(0)).getNodeValue();
     //System.out.println("system is: "+ system);
     URI systemUri = new URI(system);
     //set system
     _ecogridQuery.setSystem(systemUri);
     
    } // setupQueryIdAndSystem

   
   /* setup namepsace */
   private void setupNamespace() throws Exception
   {
     // get namespace
     NodeList namespaceList = XPathAPI.selectNodeList(_queryNode, NAMESPACE);
     int length = namespaceList.getLength();
     if ( length == 0 ) {
    	 return;
     }
     QueryTypeNamespace[] namespaces = new QueryTypeNamespace[length];
     for (int i = 0; i < length; i++)
     {
       Element namespaceElement = (Element) namespaceList.item(i);
       // get name space and prefix attribute
       String namespaceString = namespaceElement.getFirstChild().getNodeValue();
       //System.out.println("namespace in xml is: " + namespaceString);
       //URI uri = new URI(namespaceString);
       // set up a QueryType_namespace object
       QueryTypeNamespace namespace = new QueryTypeNamespace(namespaceString);
       String prefix = namespaceElement.getAttribute(PREFIX);
       namespace.setPrefix(prefix);
       namespaces[i] = namespace;
     }
     // set up the QueryTypeNamespace object to ecogrid query object
     _ecogridQuery.setNamespace(namespaces);
   }//setupNamespace

   /* setup returnfield */
   private void setupReturnField() throws Exception
   {
     // get retrunfield
     NodeList returnFieldList = XPathAPI.selectNodeList(_queryNode, RETURNFIELD);
     String [] returnFieldArray ;
     int length = returnFieldList.getLength();
     if (length != 0)
     {
       returnFieldArray = new String [length];

       for (int i=0; i<length; i++)
       {
         Node returnfield = returnFieldList.item(i);
         String returnfieldString = returnfield.getFirstChild().getNodeValue();
         returnFieldArray[i] = returnfieldString;
       }
       // set up the QueryTypeNamespace object to ecogrid query object
       _ecogridQuery.setReturnField(returnFieldArray);
     }
   }//setupReturnField

   /* setup title */
   private void setupTitle() throws Exception
   {
     // get retrunfield
     NodeList titleList = XPathAPI.selectNodeList(_queryNode, TITLE);
     String [] titleArray ;
     int length = titleList.getLength();
     if (length != 0)
     {
       titleArray = new String [length];

       for (int i=0; i<length; i++)
       {
         Node title = titleList.item(i);
         String titleString = title.getFirstChild().getNodeValue();
         //System.out.println("title is: " + titleString);
         titleArray[i] = titleString;
       }
       // set up the QueryTypeNamespace object to ecogrid query object
       _ecogridQuery.setTitle(titleArray);
     }
   }//setupReturnField

   /* setup first relation type */
   private void setupFirstRelation() throws Exception
   {
     NodeList firstAND = XPathAPI.selectNodeList(_queryNode, FIRSTAND);
     NodeList firstOR  = XPathAPI.selectNodeList(_queryNode, FIRSTOR);
     NodeList firstCondition = XPathAPI.selectNodeList(_queryNode, FIRSTCONDITION);
     int andLength = firstAND.getLength();
     int orLength  = firstOR.getLength();
     int conLength = firstCondition.getLength();
     if ( andLength != 0 && orLength ==0 && conLength == 0)
     {
       //first is and
       ANDType and = new ANDType();
       handleANDType(and, firstAND.item(0));
       _ecogridQuery.setAND(and);

     }
     else if (orLength != 0 && andLength ==0 && conLength == 0)
     {
       // first is or
       ORType or = new ORType();
       handleORType(or, firstOR.item(0));
       _ecogridQuery.setOR(or);
     }
     else if (conLength != 0 &&orLength == 0 && andLength == 0  )
     {
       // first is condition
       ConditionType condition = handleConditionType(firstCondition.item(0));
       _ecogridQuery.setCondition(condition);
     }
     else
     {
       throw new Exception ("Invalid ecogrid query!");
     }
   }//setupFirstRelation

   /*handle and type*/
   private void handleANDType(ANDType and, Node andNode) throws Exception
   {
      //make sure parameter works
      if (and == null || andNode == null)
      {
        return;
      }
      //System.out.println("in handleANDType");
      NodeList andList = XPathAPI.selectNodeList(andNode, AND);
      NodeList orList  = XPathAPI.selectNodeList(andNode, OR);
      NodeList conList = XPathAPI.selectNodeList(andNode, CON);
      int andLength = andList.getLength();
      int orLength  = orList.getLength();
      int conLength = conList.getLength();
      // and type children
      if (andLength != 0)
      {
        ANDType[] andArray = new ANDType[andLength];
        // get and type children
        for ( int i=0; i<andLength; i++)
        {
          ANDType child = new ANDType();
          andArray[i] = child;
          Node childNode = andList.item(i);
          // call handleANDype again
          handleANDType(child, childNode);
        }
        //set children to parent
        and.setAND(andArray);
      }
      //or type children
      if (orLength != 0)
      {
        ORType[] orArray = new ORType[orLength];
        // get or type children
        for ( int i=0; i< orLength; i++)
        {
          ORType child = new ORType();
          orArray[i] = child;
          Node childNode = orList.item(i);
          //call handleORType
          handleORType(child, childNode);
        }
        and.setOR(orArray);
      }
      // conditiontype
      if ( conLength != 0 )
      {
        ConditionType[] conditionArray = new ConditionType[conLength];
        for ( int i=0; i< conLength; i++)
        {
          Node childNode = conList.item(i);
          ConditionType condition = handleConditionType(childNode);
          conditionArray[i] = condition;
        }//for
        and.setCondition(conditionArray);
      }//if
   }//handleANDType

    /*handle and type*/
   private void handleORType(ORType or, Node orNode) throws Exception
   {
      // make sure parameter works
      if ( or == null || orNode == null)
      {
        return;
      }
      //System.out.println("in handleORType");
      NodeList andList = XPathAPI.selectNodeList(orNode, AND);
      NodeList orList  = XPathAPI.selectNodeList(orNode, OR);
      NodeList conList = XPathAPI.selectNodeList(orNode, CON);
      int andLength = andList.getLength();
      int orLength  = orList.getLength();
      int conLength = conList.getLength();
      // and type children
      if (andLength != 0)
      {
        ANDType[] andArray = new ANDType[andLength];
        // get and type children
        for ( int i=0; i<andLength; i++)
        {
          ANDType child = new ANDType();
          andArray[i] = child;
          Node childNode = andList.item(i);
          // call handleANDype again
          handleANDType(child, childNode);
        }
        //set children to parent
        or.setAND(andArray);
      }
      //or type children
      if (orLength != 0)
      {
        ORType[] orArray = new ORType[orLength];
        // get or type children
        for ( int i=0; i< orLength; i++)
        {
          ORType child = new ORType();
          orArray[i] = child;
          Node childNode = orList.item(i);
          //call handleORType
          handleORType(child, childNode);
        }
        or.setOR(orArray);
      }
      // conditiontype
      if ( conLength != 0 )
      {
        ConditionType[] conditionArray = new ConditionType[conLength];
        for ( int i=0; i< conLength; i++)
        {
          Node childNode = conList.item(i);
          ConditionType condition = handleConditionType(childNode);
          conditionArray[i] = condition;
        }//for
        or.setCondition(conditionArray);
      }//if
   }//handleANDType

   /* handle condition type */
   private ConditionType handleConditionType(Node condition) throws Exception
   {
     ConditionType conditionObject = new ConditionType();
     ANDType andObject = null;
     ORType   orObject = null;
     OperatorType operatorObject = null;
     String conceptObject = new String();

     if (condition == null)
     {
       return conditionObject;
     }

     Element coditionElement = (Element) condition;
     // get attribute operator from node
     String operatorString = coditionElement.getAttribute(OPERATOR);
     //System.out.println("operator string is "+ operatorString);
     operatorObject = OperatorType.fromString(operatorString);
     // get attribute concept
     String conceptString = coditionElement.getAttribute(CONCEPT);
     //System.out.println("concept string is "+ conceptString);
     conceptObject = conceptString;
     // get element value
     String value = coditionElement.getFirstChild().getNodeValue();
     //System.out.println("value is " + value);
     // set conditiontype
     conditionObject.set_value(value);
     conditionObject.setOperator(operatorObject);
     conditionObject.setConcept(conceptObject);
     return conditionObject;
   }//handleConditionType

   public static void main( String[] args ) {
      System.out.println( "Processing file: " + args[0] );

      QueryType qt = EcogridQueryParser.readEcogridXMLQuery( args[0] );

      System.out.println( "Result: ");
      System.out.println( EcogridQueryTransformer.toXMLString( qt ) );

   }
}
