/* TreeDecomposer actor decomposes a tree into subtrees.

 Copyright (c) 1998-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
*/

package org.cipres.kepler;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.Token;
import ptolemy.data.ObjectToken;
import ptolemy.data.ArrayToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.data.type.Type;
import ptolemy.data.expr.Parameter;

import org.cipres.CipresIDL.TreeDecompose;
import org.cipres.util.tree.TreePruner;
import org.cipres.CipresIDL.Tree;
import org.cipres.CipresIDL.TreeScore;
import org.cipres.helpers.CipresRegistry;
import org.cipres.helpers.RegistryEntryWrapper;
import org.cipres.service_impl.RecIDcm3Impl;

//////////////////////////////////////////////////////////////////////////
//// TreeDecomposer
/**
 * This is the implementation of the TreeDecomposer actor using Kepler.
 * This actor decomposes a tree into subtrees.
 * 
 * @author Zhijie Guan  
 * guan@sdsc.edu
*/

public class TreeDecomposer extends TypedAtomicActor {

	/** Construct TreeDecomposer source with the given container and name.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the entity cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
	
    public TreeDecomposer(CompositeEntity container, String name)
            throws NameDuplicationException, IllegalActionException  {
        super(container, name);
        
		inputTree = new TypedIOPort(this, "Tree Input", true, false);
		inputTree.setTypeEquals(BaseType.GENERAL);

        outputTrees = new TypedIOPort(this, "Trees Output", false, true);
        // Set the type constraint.
        outputTrees.setTypeEquals(BaseType.GENERAL);
        
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"20\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");
		
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    /** The output port.  The type of this port will be set to String.
     */
	public TypedIOPort inputTree = null;
    public TypedIOPort outputTrees = null;
	
	///////////////////////////////////////////////////////////////////
	////                    functional variables                   ////
	private org.cipres.CipresIDL.Tree finalTree = null;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** Send the token in the value parameter to the output.
     *  @exception IllegalActionException If it is thrown by the
     *   send() method sending out the token.
     */
    public void fire() throws IllegalActionException {
        super.fire();
		if (inputTree.hasToken(0)) {		
			
			RegistryEntryWrapper treeDecomposerWrapper = null;
			Tree[] resultTrees = null;
			ObjectToken[] treeTokens = null;
			
			try {
				TreePruner treePruner = new TreePruner();		// tree pruner
				
				treeDecomposerWrapper = CipresRegistry.getCipresServiceWrapper(TreeDecompose.class, null, null);
				// get the TreeDecompose service
				TreeDecompose decomposer = (TreeDecompose)treeDecomposerWrapper.getService();
				Tree wholeTree = (Tree)((ObjectToken)inputTree.get(0)).getValue();					
				treePruner.setTree( wholeTree.m_newick );		// set the treePruner
				
				int[][] decompositions = decomposer.leafSetDecompose( wholeTree );
				resultTrees = new Tree[decompositions.length];
				treeTokens = new ObjectToken[decompositions.length];
				for	(int i = 0; i < decompositions.length; i++)
				{
					Tree tempTree = new Tree();
					tempTree.m_newick = treePruner.pruneTree(decompositions[i]);
					tempTree.m_leafSet = decompositions[i];
					// resultTrees[i] = RecIDcm3Impl.fix_tree( tempTree );
					
					// fix finalTree. This section should be replaced once the fix_tree is open to public access
					tempTree.m_newick = tempTree.m_newick.trim();
					if (tempTree.m_newick.lastIndexOf(';') == -1)
					{
						tempTree.m_newick += ";";
					}
					if (tempTree.m_score == null)
					{
						tempTree.m_score = new TreeScore();
						tempTree.m_score.noScore(0);
					}
					
					resultTrees[i] = tempTree; // temporary used to avoid the fix_tree access. Will change it 
					resultTrees[i].m_name = "subset " + i;
					
					treeTokens[i] = new ObjectToken(resultTrees[i]);
					
					// for debug
					System.out.println("TreeDecomposer: resultTrees[" + i + "] = " + resultTrees[i].m_newick);
				}
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			finally {
				if (treeDecomposerWrapper != null) {
					treeDecomposerWrapper.releaseService();
				}
			}
			
			// for ArrayToSequence
			// outputTrees.send(0, new ArrayToken( treeTokens ));
			outputTrees.send(0, new ObjectToken(resultTrees));
		}				
    }
}

