/**
 *    '$RCSfile: EcoGridQueryServicesController.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/05 14:42:42 $'
 *   '$Revision: 1.13 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.ecogrid;

import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ecoinformatics.seek.ecogrid.exception.InvalidEcoGridServiceException;
import org.ecoinformatics.util.Config;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class will controll the current service user want to search:
 * user can add, remove and update the service list
 * @author Jing Tao
 *  
 */

public class EcoGridQueryServicesController extends EcoGridServicesController
{
  private Vector currentQueryServicesList = new Vector(); // the list in 
                                                          // configure file
  private Vector selectedSearchingServicesList    = new Vector(); // the service
                                                           // list user selected
  
  private static Log log;
  private static boolean isDebugging;
	  
  static {
  	log = LogFactory.getLog( "org.ecoinformatics.seek.ecogrid.EcoGridQueryServicesController" );
  	isDebugging = log.isDebugEnabled();
  }
  
  private static EcoGridQueryServicesController  controller = null;
  
  /**
   * Method to get an object from this singleton class
   *
   */
  public static EcoGridQueryServicesController getInstance()
  {
      if ( controller == null)
      {
          controller = new EcoGridQueryServicesController();
      }
      return controller;
  }

  /**
   * Default constructor. It will read services from configure file first.
   */
  private EcoGridQueryServicesController() 
  {
    readQueryServiceFromConfig();
  }//EcoGridServicesController

  /*
   * Read query service from configure file and put the service into a vector
   */
  private void readQueryServiceFromConfig()
  {
    String xpath  = "//" + MetadataSpecificationInterface.ECOGRIDPATH + "/" + 
                   MetadataSpecificationInterface.SERVICESLIST + "/"+
                   MetadataSpecificationInterface.SERVICE + "/" + 
                   MetadataSpecificationInterface.SERVICETYPE;
    NodeList typeList = Config.getNodeListFromPath(xpath);
    // find service type is ecogrid query 
    if (typeList != null)
    {
      int length = typeList.getLength();
      for (int i=0; i<length; i++)
      {
        Node serviceTypeNode = typeList.item(i);
        if (serviceTypeNode != null && serviceTypeNode.getFirstChild()!= null)
        {
          String serviceTypeString = serviceTypeNode.getFirstChild().getNodeValue();
          if(isDebugging){
        	  log.debug("The service type from configure file is " + 
                      serviceTypeString);
          }
          //if this is query service, transfer the service node(servicetype' parent
          // node) into java object and put it into query service list
          if (serviceTypeString != null && 
               serviceTypeString.equals(MetadataSpecificationInterface.QUERYSERVIETYPE))
          {
            Node queryServiceNode = serviceTypeNode.getParentNode();
            EcoGridService queryService = generateServiceFromServiceNode(queryServiceNode);
            if (queryService != null)
            {
              // check duplicate
              int duplicateIndex = -2;
              try
              {
                duplicateIndex = isDuplicateService(queryService, currentQueryServicesList);
              }
              catch(InvalidEcoGridServiceException e)
              {
            	  log.debug("The error for checking duplicate ecogrid service in readQueryServiceFromConfig is ", e);
              }
              // if it is not duplicate, the index should be -1
              if (duplicateIndex ==-1)
              {
            	  if(isDebugging) {
            		  log.debug("read service " + queryService.getServiceName() +
                              " from config file to vector");
            	  }
                currentQueryServicesList.add(queryService);
              }
            }
          }//if
        }//if
      }//for
      currentQueryServicesList=SelectableEcoGridService.
           transferServiceVectToDefaultSelectedServiceVect(currentQueryServicesList);
    }//if

  }//readQueryServiceFromConfig

  
 
  /**
   * Remove a given service from current query service list. The key for removing
   * is endpoint. If endpoint is same, we think they are same service
   * @param service EcoGridService
   */
  public void removeService(EcoGridService service)
  {
    if (service == null)
    {
      //if service is null, we don't need do anything
     return;
    }//if
    else
    {
       String endPoint = service.getEndPoint();
       if (endPoint == null || endPoint.trim().equals(""))
       {
         return;
       }//if
       else
       {
          // go through the current query service list and find a servcie
          // which has same endpoint to the given servcie. The remove it.
          int size = currentQueryServicesList.size();
          for (int i=0; i <size; i++)
          {
            EcoGridService currentService = (EcoGridService)
                                          currentQueryServicesList.elementAt(i);
            if (currentService != null)
            {
              String currentEndPoint = currentService.getEndPoint();
              if (currentEndPoint != null && currentEndPoint.equals(endPoint))
              {
                currentQueryServicesList.remove(i);
                if (isDebugging) {
                	log.debug("Delete service " + service.getServiceName() + 
                            " from list");
                }
                return;
              }//if
            }//if
          }//for
      }//else
    }//else    
  }//removeQueryService

  /**
   * Add a service to query service list. If new service has same endpoints to 
   * service already in current. Then we will compare document type array and
   * add new document type into service
   * If the service is not a query service, tt will throw an exception
   * @param service EcoGridService
   * @throws InvalidEcoGridServiceException
   */
  public void addService(EcoGridService service) throws InvalidEcoGridServiceException
  {
     String serviceType = service.getServiceType();
     if (serviceType == null ||
            !serviceType.equals(MetadataSpecificationInterface.QUERYSERVIETYPE))
     {
       throw new InvalidEcoGridServiceException("The service type is not " +
                     "query type and couldn't be add into list");
     }//if

     // isDuplicateService 
     int index = isDuplicateService(service, currentQueryServicesList);
     if (index ==-1)
     {
       // it is new end point
    	 if (isDebugging) {
    		 log.debug("Add service "+ service.getServiceName() +" into list");
    	 }
       currentQueryServicesList.add(service);  
     }
     else
     {
       // we should compare document type list if new service has same point
       EcoGridService currentService = (EcoGridService)currentQueryServicesList.elementAt(index);
       DocumentType[] currentDocTypeList = currentService.getDocumentTypeList();
       DocumentType[] newDocTypeList     = service.getDocumentTypeList();
       if (currentDocTypeList == null || currentDocTypeList.length ==0)
       {
         // if current serive doesn't have any document type, just set the new one
         currentService.setDocumentTypeList(newDocTypeList);
       }
       else if (newDocTypeList != null)
       {
         int sizeOfnew = newDocTypeList.length;
         int sizeofCurrent = currentDocTypeList.length;
         Vector newDocTypeVector = new Vector();
         // go through new document type
         for (int j=0; j< sizeOfnew; j++)
         {
           boolean existed = false;
           DocumentType newType = newDocTypeList[j];
           if (newType == null)
           {
             continue;
           }
           String newNamespace = newType.getNamespace();
           if (newNamespace == null || newNamespace.trim().equals(""))
           {
             continue;
           }
           for (int i=0; i< sizeofCurrent; i++)
           {
             DocumentType currentType = currentDocTypeList[i];
             if (currentType == null)
             {
               continue;
             }
             else
             {
               String currentNamespace = currentType.getNamespace();
               if (currentNamespace == null || currentNamespace.trim().equals(""))
               {
                 continue;
               }
               else if (currentNamespace.equals(newNamespace))
               {
                 existed = true;
               }//
             }//else
             
           }//for
           // if the namespace is a new space, add this documettype into the array
           if (!existed)
           {
             newDocTypeVector.add(newType);
           }
         }//for
         //if we do get some new document type(newDocTypeVector is not empty)
         // we should a new doctype into list
         if (!newDocTypeVector.isEmpty())
         {
           DocumentType[] updatedDocTypeList = addNewDocType(currentDocTypeList,
                                                    newDocTypeVector);
           currentService.setDocumentTypeList(updatedDocTypeList);
         }//if
         
       }//else if
     }//else
  }//addQueryService
  
  /*
   * Method to add a vector document type to array
   */
  private DocumentType[] addNewDocType(DocumentType[]currentArray, Vector newTypeVector)
  {
    int arraySize = currentArray.length;
    int vectorSize = newTypeVector.size();
    int newSize =  arraySize + vectorSize;
    DocumentType[] newArray = new DocumentType[newSize]; 
    //copy the array
    for (int i=0; i<arraySize; i++)
    {
      newArray[i]= currentArray[i];
    }//for
    //copy the vector
    for (int j=0; j<vectorSize; j++)
    {
      newArray[arraySize+j] = (DocumentType)newTypeVector.elementAt(j);
    }//for
    return newArray;
  }//addNewDocType
  
  /**
   * Method to update a existed EcoGridService
   * @param service EcoGridService
   * @throws InvalidEcoGridServiceException
   */
  public void updateService(EcoGridService service) throws InvalidEcoGridServiceException
  {
    if (service == null)
    {
      throw new InvalidEcoGridServiceException("Couldn't use a null service" +
                                               " to update a list");
    }
    // make sure the service is query type
    String serviceType = service.getServiceType();
    String endPoint    = service.getEndPoint();
    if (serviceType == null || 
            !serviceType.equals(MetadataSpecificationInterface.QUERYSERVIETYPE))
    {
      throw new InvalidEcoGridServiceException("The service type is not " +
                     "query type and couldn't be updated into list");
    }//if
    if (endPoint == null || endPoint.trim().equals(""))
    {
      throw new InvalidEcoGridServiceException("End point couldn't be null in "+ 
                                  "the new service which will update the list");
    }
    boolean success = false;
    //go through the vector
    int size = currentQueryServicesList.size();
    for (int i=0; i<size; i++)
    {
      EcoGridService currentService = (EcoGridService)currentQueryServicesList.elementAt(i);
      String currentEndPoint = currentService.getEndPoint();
      if (currentEndPoint != null && currentEndPoint.equals(endPoint))
      {
        //remove the old one
        currentQueryServicesList.remove(i);
        currentQueryServicesList.add(i, service);
        if(isDebugging) {
        	log.debug("Update the service " + service.getServiceName());
        }
        success = true;
      }
    }//for
    // couldn't find a service to update and will throw a exception
    if (!success)
    {
      throw new InvalidEcoGridServiceException("Couldn't find a target service " +
                                                     "to update in the list");
    }
  }//updateService
  
  /**
   * Print out every service in the list
   */
  public void print()
  {
    if (currentQueryServicesList != null)
    {
      int size = currentQueryServicesList.size();
      for (int i=0; i<size; i++)
      {
        EcoGridService service = (EcoGridService)currentQueryServicesList.elementAt(i);
        service.print();
      }//for
    }//if
  }//print
  
  /**
   * Method to get current query service list from controller
   * @return Vector
   */
  public Vector getServicesList()
  {
    return this.currentQueryServicesList;
  }//getServicesList
  
  /**
   * Method to get selected service with selected documents types from service
   * list vector.
   * If a service selected, but without any selected documents, this service
   * couldn't be in this vector
   */
  public Vector getSelectedServicesList()
  {
      selectedSearchingServicesList = new Vector();
      for (int i=0; i<currentQueryServicesList.size(); i++)
      {
          EcoGridService service = (EcoGridService)currentQueryServicesList.elementAt(i);
          if (service != null)
          {
              SelectableEcoGridService selectableService = 
                   (SelectableEcoGridService) service;
              if (selectableService.getSelectableServiceName().getIsSelected() == true)
              {
                  SelectableDocumentType[] docList = selectableService.getSelectedDocumentTypeList();
                  if( docList != null && docList.length != 0) 
                  {
                     //System.out.println("Add service !!!!!!!!!!!!!!!!! " + selectableService.getEndPoint() +"into selected service list");
                     selectedSearchingServicesList.add(selectableService);
                  }
              }
          }
      }
      return this.selectedSearchingServicesList;
  }
  
  
  /**
   * Method to set service list
   * @param currentQueryServiceList Vector
   */
  public void setServiceList(Vector currentQueryServiceList)
  {
    this.currentQueryServicesList = currentQueryServiceList;
  }//setServiceList
}//EcoGridServiceController
